# -*- coding: utf-8 -*-
"""
/***************************************************************************
 XyzTilesBasemapLoader
                                 A QGIS plugin
 Quickly load multiple open-source basemaps as XYZ tiles with one click. Includes popular sources like OSM, Google, Bing, and more
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-02
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Anustup Jana
        email                : anustupjana21@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.utils import iface

# Initialize Qt resources from file resources.py
from .resources import *
import os.path


class XyzTilesBasemapLoader:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'XyzTilesBasemapLoader_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&XYZ Tiles Basemap Loader')

        # Check if plugin was started the first time in current QGIS session
        self.first_start = None

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        return QCoreApplication.translate('XyzTilesBasemapLoader', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        # Use file system path directly to load the icon
        icon_path = os.path.join(os.path.dirname(__file__), 'icon.png')
        print(f"Loading icon from file system path: {icon_path}")
        icon = QIcon(icon_path)
        if icon.isNull():
            print("Failed to load icon from file system path.")
        else:
            print("Successfully loaded icon from file system path.")

        self.add_action(
            icon_path,
            text=self.tr(u'XYZ Tiles Basemap Loader'),
            callback=self.run,
            parent=self.iface.mainWindow())

        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&XYZ Tiles Basemap Loader'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""
        # Removed dialog initialization and display
        # Directly load basemaps and show message

        # Define basemap sources
        sources = [
            ["connections-xyz", "Google Maps", "", "", "", "https://mt1.google.com/vt/lyrs=m&x={x}&y={y}&z={z}", "", "19", "0"],
            ["connections-xyz", "Google Satellite", "", "", "", "https://mt1.google.com/vt/lyrs=s&x={x}&y={y}&z={z}", "", "19", "0"],
            ["connections-xyz", "Google Terrain", "", "", "", "https://mt1.google.com/vt/lyrs=t&x={x}&y={y}&z={z}", "", "19", "0"],
            ["connections-xyz", "Google Terrain Hybrid", "", "", "", "https://mt1.google.com/vt/lyrs=p&x={x}&y={y}&z={z}", "", "19", "0"],
            ["connections-xyz", "Google Satellite Hybrid", "", "", "", "https://mt1.google.com/vt/lyrs=y&x={x}&y={y}&z={z}", "", "19", "0"],
            ["connections-xyz", "Esri Boundaries Places", "", "", "Requires ArcGIS Online subscription", "https://server.arcgisonline.com/ArcGIS/rest/services/Reference/World_Boundaries_and_Places/MapServer/tile/{z}/{y}/{x}", "", "20", "0"],
            ["connections-xyz", "Esri Gray (dark)", "", "", "Requires ArcGIS Online subscription", "http://services.arcgisonline.com/ArcGIS/rest/services/Canvas/World_Dark_Gray_Base/MapServer/tile/{z}/{y}/{x}", "", "16", "0"],
            ["connections-xyz", "Esri Gray (light)", "", "", "Requires ArcGIS Online subscription", "http://services.arcgisonline.com/ArcGIS/rest/services/Canvas/World_Light_Gray_Base/MapServer/tile/{z}/{y}/{x}", "", "16", "0"],
            ["connections-xyz", "Esri Hillshade", "", "", "Requires ArcGIS Online subscription", "http://services.arcgisonline.com/ArcGIS/rest/services/Elevation/World_Hillshade/MapServer/tile/{z}/{y}/{x}", "", "12", "0"],
            ["connections-xyz", "Esri National Geographic", "", "", "Requires ArcGIS Online subscription", "http://services.arcgisonline.com/ArcGIS/rest/services/NatGeo_World_Map/MapServer/tile/{z}/{y}/{x}", "", "12", "0"],
            ["connections-xyz", "Esri Navigation Charts", "", "", "Requires ArcGIS Online subscription", "http://services.arcgisonline.com/ArcGIS/rest/services/Specialty/World_Navigation_Charts/MapServer/tile/{z}/{y}/{x}", "", "12", "0"],
            ["connections-xyz", "Esri Ocean", "", "", "Requires ArcGIS Online subscription", "https://services.arcgisonline.com/ArcGIS/rest/services/Ocean/World_Ocean_Base/MapServer/tile/{z}/{y}/{x}", "", "10", "0"],
            ["connections-xyz", "Esri Physical Map", "", "", "Requires ArcGIS Online subscription", "https://services.arcgisonline.com/ArcGIS/rest/services/World_Physical_Map/MapServer/tile/{z}/{y}/{x}", "", "10", "0"],
            ["connections-xyz", "Esri Satellite", "", "", "Requires ArcGIS Online subscription", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}", "", "17", "0"],
            ["connections-xyz", "Esri Shaded Relief", "", "", "Requires ArcGIS Online subscription", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Shaded_Relief/MapServer/tile/{z}/{y}/{x}", "", "17", "0"],
            ["connections-xyz", "Esri Standard", "", "", "Requires ArcGIS Online subscription", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}", "", "17", "0"],
            ["connections-xyz", "Esri Terrain", "", "", "Requires ArcGIS Online subscription", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Terrain_Base/MapServer/tile/{z}/{y}/{x}", "", "13", "0"],
            ["connections-xyz", "Esri Transportation", "", "", "Requires ArcGIS Online subscription", "https://server.arcgisonline.com/ArcGIS/rest/services/Reference/World_Transportation/MapServer/tile/{z}/{y}/{x}", "", "20", "0"],
            ["connections-xyz", "Esri Topo World", "", "", "Requires ArcGIS Online subscription", "http://services.arcgisonline.com/ArcGIS/rest/services/World_Topo_Map/MapServer/tile/{z}/{y}/{x}", "", "20", "0"],
            ["connections-xyz", "OpenStreetMap Standard", "", "", "OpenStreetMap contributors, under ODbL", "http://tile.openstreetmap.org/{z}/{x}/{y}.png", "", "19", "0"],
            ["connections-xyz", "OpenStreetMap H.O.T.", "", "", "OpenStreetMap contributors, under ODbL", "http://tile.openstreetmap.fr/hot/{z}/{x}/{y}.png", "", "19", "0"],
            ["connections-xyz", "OpenTopoMap", "", "", "Kartendaten: © OpenStreetMap-Mitwirkende, SRTM | Kartendarstellung: © OpenTopoMap (CC-BY-SA)", "https://tile.opentopomap.org/{z}/{x}/{y}.png", "", "17", "1"],
            ["connections-xyz", "Strava All", "", "", "OpenStreetMap contributors, under ODbL", "https://heatmap-external-b.strava.com/tiles/all/bluered/{z}/{x}/{y}.png", "", "15", "0"],
            ["connections-xyz", "Strava Run", "", "", "OpenStreetMap contributors, under ODbL", "https://heatmap-external-b.strava.com/tiles/run/bluered/{z}/{x}/{y}.png?v=19", "", "15", "0"],
            ["connections-xyz", "CartoDb Dark Matter", "", "", "Map tiles by CartoDB, under CC BY 3.0. Data by OpenStreetMap, under ODbL.", "http://basemaps.cartocdn.com/dark_all/{z}/{x}/{y}.png", "", "20", "0"],
            ["connections-xyz", "CartoDb Dark Matter (No Labels)", "", "", "Map tiles by CartoDB, under CC BY 3.0. Data by OpenStreetMap, under ODbL.", "http://basemaps.cartocdn.com/dark_nolabels/{z}/{x}/{y}.png", "", "20", "0"],
            ["connections-xyz", "CartoDb Positron", "", "", "Map tiles by CartoDB, under CC BY 3.0. Data by OpenStreetMap, under ODbL.", "http://basemaps.cartocdn.com/light_all/{z}/{x}/{y}.png", "", "20", "0"],
            ["connections-xyz", "CartoDb Positron (No Labels)", "", "", "Map tiles by CartoDB, under CC BY 3.0. Data by OpenStreetMap, under ODbL.", "http://basemaps.cartocdn.com/light_nolabels/{z}/{x}/{y}.png", "", "20", "0"],
            ["connections-xyz", "Bing VirtualEarth", "", "", "", "http://ecn.t3.tiles.virtualearth.net/tiles/a{q}.jpeg?g=1", "", "19", "1"]
        ]

        # Add sources to QGIS browser
        for source in sources:
            connectionType = source[0]
            connectionName = source[1]
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/authcfg", source[2])
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/password", source[3])
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/referer", source[4])
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/url", source[5])
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/username", source[6])
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/zmax", source[7])
            QSettings().setValue(f"qgis/{connectionType}/{connectionName}/zmin", source[8])

        # Update QGIS GUI
        self.iface.reloadConnections()

        # Show success message
        self.iface.messageBar().pushMessage(
            "Success",
            "Base maps added successfully, but if not visible in XYZ Tiles. Please restart QGIS.",
            level=3,  # Qgis.Info
            duration=5
        )