# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2

import logging
from typing import TYPE_CHECKING

from PyQt5.QtWebEngineWidgets import QWebEngineSettings, QWebEngineView
from qgis.core import (
    QgsMapLayer,
    QgsProject,
    QgsVectorLayer,
)
from qgis.PyQt import sip
from qgis.PyQt.QtCore import QObject, Qt, QUrl
from qgis.PyQt.QtWidgets import QComboBox

from xmas_plugin.settings_manager import get_normalized_url
from xmas_plugin.util.metadata import PLUGIN_DIR_NAME
from xmas_plugin.util.webengine import detach_channel, wire_page_lifecycle
from xmas_plugin.widgets.handler import AddFeatureInteractionHandler
from xmas_plugin.widgets.page import XMASPluginPage

if TYPE_CHECKING:
    from xmas_plugin.dock import XMASPluginDockWidget  # Only for IDE/type checker

logger = logging.getLogger(PLUGIN_DIR_NAME)


class AssociationsHandler(QObject):
    """
    Loads related ojects in layer
    """

    def __init__(self, parent: "XMASPluginDockWidget") -> None:
        super().__init__(parent)
        self.parent = parent
        self.iface = parent.iface

        self.comboBox_plans: QComboBox = self.parent.findChild(
            QComboBox, "referencesComboBox"
        )
        self.web_view: QWebEngineView = self.parent.findChild(
            QWebEngineView, "referencesView"
        )
        self._base_url = get_normalized_url()
        self._setup_association_view()
        # use generic QgsProject signals to track addition/removal of plugin layers
        QgsProject.instance().layersWillBeRemoved.connect(self.on_layers_removed)
        QgsProject.instance().layersAdded.connect(self.on_layers_added)
        self.comboBox_plans.currentIndexChanged.connect(self.on_current_plan_changed)
        # load/clear web_view depending on tab visibility
        self.parent.tabWidget.currentChanged.connect(self.on_tab_open)
        self.parent.tabWidget.tabCloseRequested.connect(self.on_tab_close)
        self._clear_association_view()

    def on_tab_open(self, index: int) -> None:
        """Load the web view when references tab is opened."""
        if (
            index == self.parent.tabWidget.indexOf(self.parent.references)
            and self.web_view.url().toString() == "about:blank"
        ):
            self.on_current_plan_changed(self.comboBox_plans.currentIndex())

    def on_tab_close(self, index: int) -> None:
        """Reset the web view when the references tab is closed."""
        if self.parent.tabWidget.indexOf(self.parent.references) == index:
            self._clear_association_view()

    def on_current_plan_changed(self, index: int) -> None:
        """Load the plan tree when selection has changed an tab is visible."""
        plan_data = self.comboBox_plans.itemData(index)
        # only load view if it's visibible, otherwise defer to on_tab_open
        if plan_data and self.web_view.isVisible():
            self._load_association_tree(plan_data["plan_id"])

    def on_layers_removed(self, layer_ids: list[str]) -> None:
        """Update combobox items when plan layers are removed and clear web view if currently selected plan is among them."""
        for layer_id in layer_ids:
            layer = QgsProject.instance().mapLayer(layer_id)
            if (
                not layer
                or layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type") != "plan"
            ):
                continue
            plan_data = {"plan_id": layer.customProperty(f"{PLUGIN_DIR_NAME}/plan_id")}
            data_index = self.comboBox_plans.findData(plan_data)
            if data_index == -1:
                continue
            if self.comboBox_plans.currentIndex() == data_index:
                self._clear_association_view()
            self.comboBox_plans.removeItem(data_index)

    def on_layers_added(self, layers: list[QgsMapLayer]) -> None:
        """Update combobox items when plan layers are added."""
        for layer in layers:
            if layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type") != "plan":
                continue
            plan_data = {"plan_id": layer.customProperty(f"{PLUGIN_DIR_NAME}/plan_id")}
            try:
                plan_name = next(layer.getFeatures())["properties"]["name"]
                self.comboBox_plans.addItem(plan_name, plan_data)
            except StopIteration:
                self.comboBox_plans.addItem("Neuer Plan", plan_data)
            # connect signal to track and handle plan name changes
            layer.afterCommitChanges.connect(
                lambda layer=layer: self.on_plan_plan_layer_commit(layer)
            )

    def on_plan_plan_layer_commit(self, layer: QgsVectorLayer):
        """Update plan name in combobox on change."""
        try:
            feature = next(layer.getFeatures())
        except StopIteration:
            return
        attribute_map = feature.attributeMap()
        if not isinstance(
            attribute_map.get("properties"), dict
        ) or not attribute_map.get("id"):
            return
        plan_data = {"plan_id": feature["id"]}
        try:
            data_index = self.comboBox_plans.findData(plan_data)
            if data_index >= 0:
                self.comboBox_plans.setItemText(
                    data_index, feature["properties"].get("name", "Neuer Plan")
                )
        except RuntimeError:
            pass

    def _setup_association_view(self) -> None:
        """Create/prepare the associations QWebEngineView + WebChannel (idempotent)."""
        detach_channel(self.web_view)  # unplug stale channel if any

        # Settings
        self.web_view.settings().setAttribute(
            QWebEngineSettings.ErrorPageEnabled, False
        )

        # Pause health polling during heavier loads (connect once)
        try:
            self.web_view.loadStarted.connect(
                lambda: self.parent._pause_polls(True),
                type=Qt.UniqueConnection,
            )
        except TypeError:
            pass
        try:
            self.web_view.loadFinished.connect(
                lambda _: self.parent._pause_polls(False),
                type=Qt.UniqueConnection,
            )
        except TypeError:
            pass

        # Cleanly unplug channel if the view is destroyed
        try:
            self.web_view.destroyed.connect(
                lambda: detach_channel(self.web_view), type=Qt.UniqueConnection
            )
        except TypeError:
            pass

        # Page
        self.web_view.setPage(XMASPluginPage(self))
        # wire webchannel and handler
        wire_page_lifecycle(self.web_view, self._make_assoc_handler)
        # initial population of combobox
        self.on_layers_added(list(QgsProject.instance().mapLayers().values()))

    def _clear_association_view(self):
        """Reset web view to blank url."""
        self.web_view.load(QUrl("about:blank"))

    def _load_association_tree(self, plan_id: str | None) -> None:
        """Guarded navigation into the associations view."""
        from xmas_plugin.dock import ServerState

        if not plan_id:
            return self._clear_association_view()
        if self.parent.server_state != ServerState.RUNNING:
            logger.info("AssociationHandler: skip load (server not RUNNING)")
            return self._clear_association_view()

        if not self.web_view or sip.isdeleted(self.web_view):
            logger.warning("AssociationHandler: view missing/deleted; not loading")
            return

        if not self._base_url:
            self._base_url = get_normalized_url()

        url = QUrl(f"{self._base_url}/plan-tree/{plan_id}")

        logger.info("AssociationHandler: loading %s", url.toString())
        self.web_view.load(url)

    def _make_assoc_handler(self, parent_channel):
        """Create JS<->Python call handler for the associations view."""
        handler = AddFeatureInteractionHandler(parent_channel)
        return handler
