# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsEditError,
    QgsFeature,
    QgsMapLayer,
    QgsProject,
    QgsVectorLayerUtils,
    edit,
)
from qgis.gui import (
    QgsAttributeEditorContext,
    QgsMapLayerAction,
    QgsMapLayerActionContext,
    QgsMapToolDigitizeFeature,
)
from qgis.PyQt.QtCore import QObject, QTimer, pyqtSlot
from qgis.utils import iface

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME


class AddPO(QgsMapLayerAction):
    def __init__(self, parent: QObject | None = None):
        super().__init__(
            "Präsentationsobjekt hinzufügen",
            parent,
            Qgis.MapLayerActionTarget.SingleFeature,
            QgsApplication.instance().getThemeIcon("title_label.svg"),
            Qgis.MapLayerActionFlags(),
        )
        self._cleaning = False
        self.current_tool = None

        self.dlg = None
        self.digit_tool = None
        self.triggeredForFeatureV2.connect(self._setup)

    def canRunUsingLayer(
        self,
        layer: QgsMapLayer | None,
        _: QgsMapLayerActionContext = QgsMapLayerActionContext(),
    ) -> bool:
        return (
            layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type") == "subject"
            and layer.customProperty(f"{PLUGIN_DIR_NAME}/appschema") == "xplan"
            if layer
            else False
        )

    @pyqtSlot("QgsMapLayer*", QgsFeature, QgsMapLayerActionContext)
    def _setup(
        self,
        layer: QgsMapLayer,
        feature: QgsFeature,
        _: QgsMapLayerActionContext,
    ) -> None:
        self.layer = layer
        self.feature = feature
        target_layer = None
        for map_layer in QgsProject.instance().mapLayers().values():
            layer_type = map_layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type", "")
            if (
                "presentation" == layer_type
                and map_layer.geometryType().name == "Point"
                and map_layer.customProperty(f"{PLUGIN_DIR_NAME}/bereich_id")
                == self.layer.customProperty(f"{PLUGIN_DIR_NAME}/bereich_id")
            ):
                target_layer = map_layer
                break
        if not target_layer:
            return iface.pushCritical(
                "Neues Präsentationsobjekt",
                "kein passender Präsentationsobjekte-Layer gefunden",
            )
        canvas = iface.mapCanvas()
        self.current_tool = canvas.mapTool()
        self.digit_tool = QgsMapToolDigitizeFeature(canvas, iface.cadDockWidget())
        self.digit_tool.setLayer(target_layer)
        self.digit_tool.digitizingCompleted.connect(
            lambda digit_feature: self._handle_digitizing(
                digit_feature=digit_feature,
                source_feature=feature,
                target_layer=target_layer,
            )
        )
        self.digit_tool.digitizingCanceled.connect(lambda: self._cleanup(target_layer))
        self.digit_tool.digitizingFinished.connect(lambda: self._cleanup(target_layer))

        target_layer.startEditing()
        canvas.setMapTool(self.digit_tool)
        iface.messageBar().pushInfo("Neues Präsentationsobjekt", "Geometrie erfassen")

    @pyqtSlot(QgsFeature, QgsFeature, QgsMapLayer)
    def _handle_digitizing(
        self,
        digit_feature: QgsFeature,
        source_feature: QgsFeature,
        target_layer: QgsMapLayer,
    ):
        try:
            if not digit_feature or not digit_feature.hasGeometry():
                return
            target_layer.rollBack()
            feature = QgsVectorLayerUtils.createFeature(
                target_layer,
                digit_feature.geometry(),
                {
                    target_layer.fields().indexFromName("properties"): {
                        "dientZurDarstellungVon": [source_feature["id"]],
                    },
                },
            )
            self.dlg = iface.getFeatureForm(target_layer, feature)
            self.dlg.setMode(QgsAttributeEditorContext.Mode.AddFeatureMode)

            accepted = False
            with edit(target_layer):
                accepted = self.dlg.exec()
            if accepted:
                target_layer.reload()
                iface.messageBar().pushSuccess(
                    "Neues Präsentationsobjekt", "Präsentationsobjekt hinzugefügt"
                )
        except QgsEditError as e:
            iface.messageBar().pushCritical("Neues Präsentationsobjekt", repr(e))
        finally:
            self._cleanup(target_layer=target_layer)

    def _cleanup(self, target_layer: QgsMapLayer):
        if self._cleaning:
            return

        self._cleaning = True

        if self.digit_tool:
            try:
                self.digit_tool.digitizingCompleted.disconnect()
            except TypeError:
                pass
            try:
                self.digit_tool.digitizingCanceled.disconnect()
            except TypeError:
                pass
            try:
                self.digit_tool.digitizingFinished.disconnect()
            except TypeError:
                pass

        if self.current_tool:
            iface.mapCanvas().setMapTool(self.current_tool)
            self.current_tool = None

        if target_layer and target_layer.isEditable():
            QTimer.singleShot(0, target_layer.rollBack)

        dlg = self.dlg
        tool = self.digit_tool
        self.dlg = None
        self.digit_tool = None

        def _delete_later():
            try:
                if dlg:
                    dlg.deleteLater()
                if tool:
                    tool.deleteLater()
            finally:
                self._cleaning = False

        QTimer.singleShot(0, _delete_later)
