# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2

from qgis.analysis import QgsGeometrySnapper
from qgis.core import (
    Qgis,
    QgsApplication,
    QgsFeature,
    QgsMapLayer,
    QgsVectorLayer,
)
from qgis.gui import (
    QgsMapLayerAction,
    QgsMapLayerActionContext,
)
from qgis.PyQt.QtCore import QObject, pyqtSlot
from qgis.PyQt.QtWidgets import QInputDialog
from qgis.utils import iface

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME, PLUGIN_NAME


class SnapToLayer(QgsMapLayerAction):
    def __init__(self, parent: QObject | None = None):
        super().__init__(
            "Snap auf anderen Layer",
            parent,
            Qgis.MapLayerActionTargets(
                Qgis.MapLayerActionTarget.SingleFeature
                | Qgis.MapLayerActionTarget.MultipleFeatures
            ),
            QgsApplication.instance().getThemeIcon("mIconSnappingAllLayers.svg"),
            Qgis.MapLayerActionFlags(),
        )
        self.triggeredForFeatureV2.connect(self._trigger_for_feature)
        self.triggeredForFeaturesV2.connect(self._trigger_for_features)

    def canRunUsingLayer(
        self,
        layer: QgsMapLayer | None,
        _: QgsMapLayerActionContext = QgsMapLayerActionContext(),
    ) -> bool:
        return (
            layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type") is not None
            and layer.isEditable()
            if layer
            else False
        )

    @pyqtSlot("QgsMapLayer*", QgsFeature, QgsMapLayerActionContext)
    def _trigger_for_feature(
        self,
        layer: QgsMapLayer,
        feature: QgsFeature,
        _: QgsMapLayerActionContext,
    ) -> None:
        self._snap_to_layer(layer, [feature])

    @pyqtSlot("QgsMapLayer*", "QList<QgsFeature>", QgsMapLayerActionContext)
    def _trigger_for_features(
        self,
        layer: QgsMapLayer,
        features: list[QgsFeature],
        _: QgsMapLayerActionContext,
    ) -> None:
        self._snap_to_layer(layer, features)

    def _snap_to_layer(
        self,
        layer: QgsMapLayer,
        features: list[QgsFeature],
    ) -> None:
        tree_view = iface.layerTreeView()
        snap_layers = [
            selected_layer
            for selected_layer in tree_view.selectedLayers()
            if isinstance(selected_layer, QgsVectorLayer) and selected_layer != layer
        ]
        if len(snap_layers) != 1:
            iface.messageBar().pushWarning(
                PLUGIN_NAME,
                "zum Snappen genau einen weiteren Vektorlayer im Layerbaum selektieren",
            )
            return
        snap_layer = snap_layers[0]
        snapper = QgsGeometrySnapper(snap_layer)
        try:
            uom = layer.crs().mapUnits().name
            tolerance, ok = QInputDialog().getDouble(
                iface.mainWindow(), PLUGIN_NAME, f"Snapping-Toleranz [{uom}]:", value=1
            )
            if not ok:
                return
            snapped_features = snapper.snapFeatures(features, tolerance)
            snapped = 0
            for index, snapped_feature in enumerate(snapped_features):
                if not features[index].geometry().equals(snapped_feature.geometry()):
                    layer.beginEditCommand("snap to layer")
                    layer.changeGeometry(
                        snapped_feature.id(), snapped_feature.geometry()
                    )
                    layer.endEditCommand()
                    snapped += 1
            iface.messageBar().pushSuccess(
                PLUGIN_NAME,
                f"{snapped} Feature(s) mit Toleranz {tolerance} ({uom}) auf Layer {snap_layer.name()} gesnappt.",
            )
        except Exception as e:
            iface.messageBar().pushError(PLUGIN_NAME, f"Fehler beim Snappen: {e!r}")
