# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsEditError,
    QgsFeature,
    QgsMapLayer,
    QgsProject,
    QgsVectorLayerUtils,
    edit,
)
from qgis.gui import (
    QgsAttributeEditorContext,
    QgsMapLayerAction,
    QgsMapLayerActionContext,
    QgsMapToolDigitizeFeature,
)
from qgis.PyQt.QtCore import QObject, Qt, pyqtSlot
from qgis.utils import iface

from xmas_plugin.plan_manager import PlanManager
from xmas_plugin.util.metadata import PLUGIN_DIR_NAME


class AddSection(QgsMapLayerAction):
    def __init__(self, parent: QObject | None = None):
        super().__init__(
            "Bereich hinzufügen",
            parent,
            Qgis.MapLayerActionTarget.SingleFeature,
            QgsApplication.instance().getThemeIcon("mActionAddLayer.svg"),
            Qgis.MapLayerActionFlags(),
        )
        self.feature = None
        self.layer = None
        self.dlg = None
        self.digit_tool = None
        self.triggeredForFeatureV2.connect(self._setup)

    def canRunUsingLayer(
        self,
        layer: QgsMapLayer | None,
        _: QgsMapLayerActionContext = QgsMapLayerActionContext(),
    ) -> bool:
        return (
            layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type") == "plan"
            and layer.customProperty(f"{PLUGIN_DIR_NAME}/appschema") in ["xplan", "xwp"]
            if layer
            else False
        )

    @pyqtSlot("QgsMapLayer*", QgsFeature, QgsMapLayerActionContext)
    def _setup(
        self,
        layer: QgsMapLayer,
        feature: QgsFeature,
        _: QgsMapLayerActionContext,
    ) -> None:
        self.layer = layer
        self.feature = feature
        self._digitize_section()

    def _digitize_section(self):
        canvas = iface.mapCanvas()
        self.digit_tool = QgsMapToolDigitizeFeature(canvas, iface.cadDockWidget())
        self.digit_tool.setLayer(self.layer)
        try:
            self.digit_tool.digitizingCompleted.connect(
                self._handle_digitizing, type=Qt.UniqueConnection
            )
        except TypeError:
            pass
        try:
            self.digit_tool.digitizingCanceled.connect(
                lambda: self._handle_digitizing(QgsFeature()), type=Qt.UniqueConnection
            )
        except TypeError:
            pass
        self.layer.startEditing()
        canvas.setMapTool(self.digit_tool)
        iface.messageBar().pushInfo(
            "Neuer Bereich", "Geometrie erfassen oder weiter mit ESC"
        )

    @pyqtSlot(QgsFeature)
    def _handle_digitizing(self, digit_feature: QgsFeature):
        self.layer.rollBack()
        prefix = self.feature["featuretype"][:2]
        feature = QgsVectorLayerUtils.createFeature(
            self.layer,
            digit_feature.geometry(),
            {self.layer.fields().indexFromName("featuretype"): f"{prefix}_Bereich"},
        )
        self.dlg = iface.getFeatureForm(self.layer, feature)
        self.dlg.setMode(QgsAttributeEditorContext.Mode.AddFeatureMode)
        try:
            accepted = False
            with edit(self.layer):
                accepted = self.dlg.exec()
        except QgsEditError as e:
            iface.messageBar().pushCritical("Neuer Bereich", repr(e))
        else:
            if accepted:
                feature = self.dlg.attributeForm().currentFormFeature()
                root = QgsProject.instance().layerTreeRoot()
                section_layers = []
                PlanManager().build_section_group(
                    bereich={
                        "id": feature["id"],
                        "nummer": feature["properties"]["nummer"],
                        "geometry": feature.hasGeometry(),
                    },
                    group=root.findLayer(self.layer).parent(),
                    layers=section_layers,
                    plan_id=self.feature["id"],
                    appschema=feature["appschema"],
                    version=feature["version"],
                )
                QgsProject.instance().addMapLayers(
                    section_layers,
                    False,
                )
                iface.messageBar().pushSuccess("Neuer Bereich", "Bereich hinzugefügt")
        finally:
            self._cleanup()

    def _cleanup(self):
        self.feature = None
        self.layer = None
        if self.dlg:
            self.dlg.deleteLater()
            self.dlg = None
        if self.digit_tool:
            self.digit_tool.deleteLater()
            self.digit_tool = None
