# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Anton Jacobsson <anton.jacobsson@init.de>
#
# SPDX-License-Identifier: EUPL-1.2-or-later

import logging
from dataclasses import dataclass
from enum import Enum
from typing import Optional, Set

from qgis.core import (
    Qgis,
    QgsCoordinateTransform,
    QgsFeature,
    QgsGeometry,
    QgsLayerTreeGroup,
    QgsMapLayerStyle,
    QgsMessageLog,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingContext,
    QgsProcessingException,
    QgsProcessingFeedback,
    QgsProcessingOutputString,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterEnum,
    QgsProcessingParameterVectorLayer,
    QgsProject,
    QgsVectorLayer,
    QgsVectorLayerJoinInfo,
    QgsWkbTypes,
)
from qgis.PyQt import QtCore, sip
from qgis.PyQt.QtCore import QCoreApplication, QThread

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME, PLUGIN_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


# Success signal
class SplitFinished(QtCore.QObject):
    split_done = QtCore.pyqtSignal(object)


SPLIT_BUS = SplitFinished()

# Core plugin constants used to define custom property keys in layers.
ORIG_ID_FIELD = "id"
PLUGIN_KEY = PLUGIN_DIR_NAME  # Base key used for namespacing plugin properties
APP_KEY = f"{PLUGIN_KEY}/appschema"
VER_KEY = f"{PLUGIN_KEY}/appschema_version"
MODEL_KEY = f"{PLUGIN_KEY}/layer_type"  # Expected layer model key
PLAN_KEY = f"{PLUGIN_KEY}/plan_id"  # Plan identifier key
BEREICH_KEY = f"{PLUGIN_KEY}/bereich_id"  # Area/sector identifier key


class AppField(Enum):
    TYPE = APP_KEY
    VERSION = VER_KEY


@dataclass(frozen=True)
class AppSchema:
    type: str
    version: str


class SplitPlanByBoundaryAlgorithm(QgsProcessingAlgorithm):
    """
    Implements an algorithm to split a given plan into "inner" and "outer" layers
    based on a set of boundary geometries. The split results are accumulated and sent
    using a formatted payload to downstream processes.

    Key Features:
    - Handles CRS-aware boundary geometries for robust layer splitting.
    - Supports JSON serialization of geometries for integration with external systems.
    - Validates that split geometries intersect source layers before proceeding.

    Major Methods:
    - `processAlgorithm`: Core execution logic triggering the splitting operation.
    - `_new_feature_lists`: Initialize accumulators for collecting split geometry features.
    - `_acc_has_data`: Validation utility to check if accumulators hold any features.
    """

    # ---------------- Parameters / Metadata ----------------
    PLAN_GROUP = "PLAN_GROUP"
    CUT_LAYER = "CUT_LAYER"
    USE_SELECTION = "USE_SELECTION"

    # Improve crash-stability by making sure algorithm runs on the main GUI-thread, and methods like addGroup(...) etc. don't run concurrently with the layer tree
    def flags(self):
        return (
            QgsProcessingAlgorithm.FlagNoThreading
            | QgsProcessingAlgorithm.FlagRequiresProject
        )

    def initAlgorithm(self, config=None) -> None:
        # Build a list of top-level groups for convenience
        root = QgsProject.instance().layerTreeRoot()
        group_names = [g.name() for g in root.children() if g.nodeType() == g.NodeGroup]

        # Keep for later lookup
        self._group_names = ["— Plan-Gruppe wählen —"] + group_names

        self.addParameter(
            QgsProcessingParameterEnum(
                self.PLAN_GROUP,
                "Plan-Gruppe (erforderlich)",
                self._group_names,  # options
                False,  # allowMultiple
                None,  # defaultValue (omit -> None; UI shows index 0)
                False,  # optional=False -> required
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.CUT_LAYER,
                "Schnitt-Layer (Polygon)",
                [QgsProcessing.TypeVectorPolygon],
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.USE_SELECTION, "Nur Auswahl verwenden", defaultValue=False
            )
        )
        self.addOutput(QgsProcessingOutputString("SUMMARY", "Ergebnis-Zusammenfassung"))

    def name(self) -> str:
        return "split_plan_by_boundary"

    def displayName(self) -> str:
        return "Plan nach Schnitt-Layer aufteilen (Innen/Außen)"

    def group(self) -> str:
        return "Plan"

    def groupId(self) -> str:
        return "plan"

    def shortHelpString(self) -> str:
        return (
            "Teilt den Plan in zwei Ausgaben (Innen/Außen) anhand eines Polygon-Schnitt-Layers. "
            "Bei aktiver Option werden nur ausgewählte Features des Schnitt-Layers verwendet."
        )

    def createInstance(self) -> "SplitPlanByBoundaryAlgorithm":
        return SplitPlanByBoundaryAlgorithm()

    # ---------------- Processing entry ----------------

    def processAlgorithm(
        self, params, context: QgsProcessingContext, feedback: QgsProcessingFeedback
    ):
        """
        Main execution entry point for the algorithm.

        Coordinates the splitting of plan layers into 'inner' and 'outer' parts
        and prepares payload for downstream operations.

        Args:
            parameters (dict): Input parameters passed to the algorithm.
            context (QgsProcessingContext): Context containing project-related information.
            feedback (QgsProcessingFeedback): Feedback object for reporting progress/status.

        Returns:
            dict: Results showing success and payload details.
        """
        # ---- Crash-safety harness ----------------------------------------------------
        app = QCoreApplication.instance()
        on_gui = QThread.currentThread() == app.thread()
        if not on_gui:
            logger.warning("[Split] Not on GUI thread (unexpected).")

        self._cancelled = False
        proj = QgsProject.instance()

        def _abort(reason: str):
            if not self._cancelled:
                feedback.reportError(reason)
                self._cancelled = True
                feedback.cancel()

        def _on_layers_will_be_removed(ids):
            _abort(
                "Layers are being removed; aborting split to avoid invalid pointers."
            )

        def _on_project_clearing():
            _abort("Project is being cleared/closed; aborting split.")

        proj.layersWillBeRemoved.connect(_on_layers_will_be_removed)
        proj.cleared.connect(_on_project_clearing)
        # -----------------------------------------------------------------------

        try:
            # --- Resolve required plan group (no auto-detect fallback) ---
            chosen_idx = self.parameterAsEnum(params, self.PLAN_GROUP, context)
            if chosen_idx in (None, 0):
                raise QgsProcessingException(
                    "Bitte wählen Sie die Plan-Gruppe, die aufgeteilt werden soll."
                )

            group_name = self._group_names[chosen_idx]
            plan_group = self._find_group_by_name(group_name)
            if plan_group is None:
                raise QgsProcessingException(
                    f"Ausgewählte Plan-Gruppe '{group_name}' wurde im Projektbaum nicht gefunden."
                )

            # --- Cut layer ---
            cut_layer: QgsVectorLayer = self.parameterAsVectorLayer(
                params, self.CUT_LAYER, context
            )

            if not cut_layer or cut_layer.type() != QgsVectorLayer.VectorLayer:
                raise QgsProcessingException(
                    "Schnitt-Layer fehlt oder ist kein Vektorlayer."
                )
            if (
                QgsWkbTypes.geometryType(cut_layer.wkbType())
                != QgsWkbTypes.PolygonGeometry
            ):
                raise QgsProcessingException(
                    "Schnitt-Layer muss ein Polygon-Layer sein."
                )

            # --- Build boundary features list ---
            all_feats = list(cut_layer.getFeatures())
            sel_feats = list(cut_layer.getSelectedFeatures())
            use_sel: bool = self.parameterAsBoolean(params, self.USE_SELECTION, context)

            # Effective set of boundary polygons used for splitting
            feats = sel_feats if use_sel else all_feats

            # Only treat boundary features as “selected” when the checkbox is ON
            selected_fids: Set[int] = {f.id() for f in sel_feats} if use_sel else set()

            # Guard: empty input
            if not feats:
                total = len(all_feats)
                if use_sel and total > 0:
                    # User chose "selection only" but none are selected
                    raise QgsProcessingException(
                        f"Im Schnitt-Layer '{cut_layer.name()}' sind 0 ausgewählte Polygone, "
                        f"aber {total} Polygone insgesamt vorhanden.\n"
                        f"• Auswahl aufheben oder 'Nur Auswahl verwenden' deaktivieren.\n"
                        f"• Layer-CRS: {cut_layer.crs().authid()}."
                    )
                # Truly no polygons at all
                raise QgsProcessingException(
                    f"Keine Polygone im Schnitt-Layer '{cut_layer.name()}'. "
                    f"Layer-CRS: {cut_layer.crs().authid()}."
                )
            geoms = [
                QgsGeometry(f.geometry())
                for f in feats
                if f.geometry() and not f.geometry().isEmpty()
            ]
            if not geoms:
                raise QgsProcessingException(
                    f"Alle (ausgewählten) Geometrien im Schnitt-Layer '{cut_layer.name()}' sind leer/ungültig."
                )

            boundary_src = QgsGeometry.unaryUnion(geoms)  # in cut_layer.crs()

            # Resolve plan group (explicit selection first, then fallback to autodetect)
            plan_group: Optional[QgsLayerTreeGroup] = None
            chosen_idx = self.parameterAsEnum(params, self.PLAN_GROUP, context)
            if (
                chosen_idx is not None
                and chosen_idx >= 0
                and chosen_idx < len(self._group_names)
            ):
                plan_group_name = self._group_names[chosen_idx]
                plan_group = self._find_group_by_name(plan_group_name)

            if plan_group is None:
                # Use boundary in its own CRS for discovery (ok; just intersects test)
                plan_group = self._find_plan_group_by_geometry(boundary_src)

            if not plan_group:
                raise QgsProcessingException(
                    "Keine Plan-Gruppe gefunden. Bitte wählen Sie eine Plan-Gruppe "
                    "oder verwenden Sie eine Schnitt-Geometrie, die den Geltungsbereich schneidet."
                )

            # Quick eligibility check: require at least one layer whose custom property *contains* "XPlanPlugin"
            if not self._group_has_xplanplugin_fast(plan_group):
                raise QgsProcessingException(
                    f"Die ausgewählte Plan-Gruppe enthält keinen Layer mit einer benutzerdefinierten "
                    f"Eigenschaft, die '{PLUGIN_KEY}' enthält."
                )
            # --- Early overlap diagnostic (helps spot CRS/extent mismatches) ---
            proj_crs = QgsProject.instance().crs()
            plan_ext_proj = self._plan_group_extent_in(plan_group, proj_crs)
            if plan_ext_proj and not plan_ext_proj.isEmpty():
                mask_ext_proj = self._transform_extent(
                    boundary_src.boundingBox(), cut_layer.crs(), proj_crs
                )
                if not mask_ext_proj.intersects(plan_ext_proj):
                    raise QgsProcessingException(
                        "Die Schnitt-Geometrie überlappt den gewählten Plan nicht.\n"
                        f"• Projekt-CRS: {proj_crs.authid()}\n"
                        f"• Plan-Ausdehnung: {plan_ext_proj.toString()}\n"
                        f"• Schnitt-Layer '{cut_layer.name()}' CRS: {cut_layer.crs().authid()}\n"
                        f"• Schnitt-Ausdehnung (in Projekt-CRS): {mask_ext_proj.toString()}\n\n"
                        "Tipps:\n"
                        "  – Prüfen Sie, ob der Schnitt-Layer im richtigen Koordinatensystem vorliegt (Layer-CRS vs. Projekt/Plan-CRS).\n"
                        "  – Reprojizieren Sie den Schnitt-Layer (Vektor → Geometrie → Reprojizieren) oder zeichnen Sie die Maske im Plan-CRS.\n"
                        "  – Vergewissern Sie sich, dass Sie die richtige Plan-Gruppe gewählt haben."
                    )

            feedback.pushInfo(f"[Split] Plan-Gruppe: {plan_group.name()}")
            feedback.pushInfo(
                f"[Split] Schnitt-Layer: {cut_layer.name()} CRS={cut_layer.crs().authid()}"
            )

            # Create output groups at root
            root = QgsProject.instance().layerTreeRoot()
            src_name = plan_group.name()

            # Create top-level destination groups (fresh objects, short-lived handles)
            grp_innen = root.addGroup(f"{src_name} – Innen")
            grp_außen = root.addGroup(f"{src_name} – Außen")
            grp_innen.setExpanded(False)
            grp_außen.setExpanded(False)

            # Snapshot the plan structure to avoid holding live group refs
            plan_snap = self._snapshot_group(plan_group)
            plan_id = self._find_plan_id(plan_group)

            # Initialize payload accumulators, handle splitting, and validate results.
            feature_lists = self._new_feature_lists()  # Ensures eventual payload has valid accumulated data; raises exception otherwise.

            def materialize_groups(
                parent_dst: QgsLayerTreeGroup, snap: dict
            ) -> QgsLayerTreeGroup:
                """
                Ensure a subgroup exists under parent_dst matching snap['name'],
                and return it. Short-lived handle; re-derive as needed.
                """
                sub = parent_dst.addGroup(snap["name"])
                sub.setExpanded(False)
                return sub

            def process_snapshot(
                snap: dict, dst_in: QgsLayerTreeGroup, dst_out: QgsLayerTreeGroup
            ):
                # 1) Process layers by ID (resolve live layer right before use)
                project = QgsProject.instance()
                for lid in snap["layers"]:
                    if feedback.isCanceled() or self._cancelled:
                        return
                    lyr = project.mapLayer(lid)
                    if (
                        not self._is_alive(lyr)
                        or lyr.type() != lyr.VectorLayer
                        or not lyr.customProperty(f"{PLUGIN_DIR_NAME}/appschema")
                    ):
                        continue

                    # Split this resolved layer (short-lived live object)
                    self._split_layer_to_groups(
                        src_layer=lyr,
                        boundary_geom_src=boundary_src,
                        boundary_src_crs=cut_layer.crs(),
                        boundary_src_layer=cut_layer,
                        selected_boundary_fids=selected_fids,
                        grp_innen=dst_in,
                        grp_außen=dst_out,
                        feedback=feedback,
                        feature_lists=feature_lists,
                    )

                # 2) Recurse into subgroups (create fresh destination groups by name)
                for sub_snap in snap["groups"]:
                    sub_in = materialize_groups(dst_in, sub_snap)
                    sub_out = materialize_groups(dst_out, sub_snap)
                    process_snapshot(sub_snap, sub_in, sub_out)

            process_snapshot(plan_snap, grp_innen, grp_außen)
            feedback.pushInfo(f"Plan '{src_name}' in Innen/Außen aufgeteilt.")

            if not (
                self._acc_has_data(feature_lists["inner"])
                or self._acc_has_data(feature_lists["outer"])
            ):
                raise QgsProcessingException(
                    "Schnitt-Geometrie überlappt keine Layer des Plans. Vorgang abgebrochen; nichts zu senden."
                )

            if self._cancelled:
                raise QgsProcessingException(
                    "Vorgang abgebrochen (Projekt/Layers wurden geändert)."
                )

            summary_lines = [
                f"Plan '{src_name}' erfolgreich aufgeteilt.",
                f"Neue Gruppen: '{grp_innen.name()}' und '{grp_außen.name()}'.",
            ]
            summary = " ".join(summary_lines)

            # Show success
            feedback.pushInfo("[Split] ✅ " + summary)  # Processing feedback panel
            feedback.setProgress(100)  # make sure the bar hits 100%
            QgsMessageLog.logMessage(
                summary, PLUGIN_NAME, Qgis.Info
            )  # Log Messages panel

            schema = self._extract_appschema_firsthit_from_group(plan_group)

            logger.info(
                "[Split] INNER ids (sample): %s",
                [it["old_object_id"] for it in feature_lists["inner"][:20]],
            )
            logger.info(
                "[Split] OUTER ids (sample): %s",
                [it["old_object_id"] for it in feature_lists["outer"][:20]],
            )

            payload = {
                "schema_version": "split-import-v1-min",
                "src_plan_group": plan_group.name(),  # Optional
                "src_plan_id": plan_id,
                "crs": proj_crs.authid(),  # Optional (the server currently only needs srid, stored in the features)
                "appschema": {"type": schema.type, "version": schema.version},
                "inner": {
                    "group_name": grp_innen.name(),
                    "items": feature_lists["inner"],
                },
                "outer": {
                    "group_name": grp_außen.name(),
                    "items": feature_lists["outer"],
                },
            }

            # Emit success signal
            SPLIT_BUS.split_done.emit(payload)

            # Return a declared output so the dialog shows it in the results area
            return {"SUMMARY": summary}
        finally:
            # Always disconnect signals to avoid leaks
            try:
                proj.layersWillBeRemoved.disconnect(_on_layers_will_be_removed)
            except Exception:
                pass
            try:
                proj.cleared.disconnect(_on_project_clearing)
            except Exception:
                pass

    # ---------------- Helpers: plan detection ----------------

    def _find_group_by_name(self, name: str) -> Optional[QgsLayerTreeGroup]:
        root = QgsProject.instance().layerTreeRoot()
        for child in root.children():
            if child.nodeType() == child.NodeGroup and child.name() == name:
                return child
        return None

    def _find_plan_group_by_geometry(
        self, split_geom: "QgsGeometry"
    ) -> Optional[QgsLayerTreeGroup]:
        """Return the plan group whose polygon 'Geltungsbereich' intersects the split geometry."""
        root = QgsProject.instance().layerTreeRoot()
        for group in root.children():
            if group.nodeType() != group.NodeGroup:
                continue
            for child in group.children():
                if child.nodeType() != child.NodeLayer:
                    continue
                try:
                    lyr = child.layer()
                except Exception:
                    lyr = None
                if not self._is_alive(lyr) or lyr.type() != lyr.VectorLayer:
                    continue
                if (
                    QgsWkbTypes.geometryType(lyr.wkbType())
                    != QgsWkbTypes.PolygonGeometry
                ):
                    continue
                for feat in lyr.getFeatures():
                    try:
                        if feat.geometry() and feat.geometry().intersects(split_geom):
                            logger.info("[Split] Plan group detected: %s", group.name())
                            return group
                    except Exception:
                        continue
        return None

    # ---------------- Helpers: layer split ----------------

    def _split_layer_to_groups(
        self,
        src_layer: QgsVectorLayer,
        boundary_geom_src: "QgsGeometry",
        boundary_src_crs,  # CRS of boundary_geom_src
        boundary_src_layer: QgsVectorLayer,
        selected_boundary_fids: Set[int],
        grp_innen: QgsLayerTreeGroup,
        grp_außen: QgsLayerTreeGroup,
        feedback: QgsProcessingFeedback,
        feature_lists: list,
    ) -> None:
        """
        Split the source layer into 'inner' and 'outer' layers based on the boundary geometry.

        Args:
            src_layer (QgsVectorLayer): Layer to be split.
            boundary_geom_src (QgsGeometry): Boundary geometry used for splitting.
            boundary_src_crs (CRS): CRS of the boundary geometry.
            boundary_src_layer (QgsVectorLayer): Layer containing the boundary features.
            selected_boundary_fids (Set[int]): IDs of selected boundary features for splitting.
            grp_innen (QgsLayerTreeGroup): Group where 'inner' split layers are added.
            grp_außen (QgsLayerTreeGroup): Group where 'outer' split layers are added.
            feedback (QgsProcessingFeedback): Feedback object for reporting progress.
            feature_lists (list): Accumulator for collecting split features.

        Important Notes:
        - Layers whose geometries do not intersect the boundary will not be split.
        - Features are serialized into JSON for integration with downstream systems.
        """
        # Bail out early for invalid or disappearing layers (runtime safety)
        if not self._is_alive(src_layer):
            return
        # Safety: re-resolve by ID (layer might have been replaced)
        project = QgsProject.instance()
        live = project.mapLayer(src_layer.id()) if hasattr(src_layer, "id") else None
        if not self._is_alive(live):
            return

        src_layer = live  # Ensure we work with updated layer reference
        # Prepare feature lists for POST accumulation (per role, per feature)
        inner_list = feature_lists["inner"]
        outer_list = feature_lists["outer"]

        layer_name = src_layer.name()
        logger.info("[Split] Splitting layer '%s'.", layer_name)

        # Transform boundary into this layer's CRS
        boundary_in_layer = self._ensure_geom_crs(
            boundary_geom_src, boundary_src_crs, src_layer.crs()
        )

        # Validate / repair if needed
        if not boundary_in_layer or boundary_in_layer.isEmpty():
            raise QgsProcessingException(
                f"Masken-Geometrie leer nach Transform ({boundary_src_crs.authid()} → {src_layer.crs().authid()})."
            )
        try:
            if not boundary_in_layer.isGeosValid():
                boundary_in_layer = boundary_in_layer.makeValid()
        except Exception:
            # Don't crash on validator hiccups
            pass

        # Quick bbox overlap test to avoid silent “all Außen”
        if not boundary_in_layer.boundingBox().intersects(src_layer.extent()):
            feedback.pushInfo(
                f"[Split] Hinweis: Maske überlappt Layer '{layer_name}' nicht (nach Transform). Überspringe Split."
            )
            return

        mem_in = self._empty_mem_clone(src_layer, layer_name)
        mem_out = self._empty_mem_clone(src_layer, layer_name)
        prov_in, prov_out = mem_in.dataProvider(), mem_out.dataProvider()
        feats_in, feats_out = [], []

        is_boundary_layer = src_layer.id() == boundary_src_layer.id()
        geom_type = src_layer.geometryType()

        # Extra safety
        if not self._is_alive(grp_innen) or not self._is_alive(grp_außen):
            return

        for f in src_layer.getFeatures():
            if feedback.isCanceled() or self._cancelled:
                break
            g = f.geometry()
            if not g or g.isEmpty():
                continue

            # Boundary features themselves (if on the cut layer & selected) -> whole Innen
            if is_boundary_layer and f.id() in selected_boundary_fids:
                feats_in.append(QgsFeature(f))
                inner_list.append(
                    {
                        "old_object_id": self._old_object_id(f),
                        "wkt": g.asWkt(),
                    }
                )
                continue

            # Robust split via intersection/difference
            try:
                g_in = boundary_in_layer.intersection(g)
                g_out = g.difference(boundary_in_layer)
            except Exception as e:
                logger.warning(
                    "[Split] GEOS op failed on layer '%s', fid=%s: %r",
                    layer_name,
                    f.id(),
                    e,
                )
                feats_out.append(QgsFeature(f))
                continue

            if g_in and not g_in.isEmpty():
                nf = QgsFeature(f)
                nf.setGeometry(g_in)
                feats_in.append(nf)
                inner_list.append(
                    {
                        "old_object_id": self._old_object_id(f),
                        "wkt": g_in.asWkt(),
                    }
                )

            # if 'in' is empty but intersects, push inside; otherwise outside
            if geom_type == QgsWkbTypes.PointGeometry and (not g_in or g_in.isEmpty()):
                try:
                    if boundary_in_layer.intersects(g):
                        feats_in.append(QgsFeature(f))
                        inner_list.append(
                            {
                                "old_object_id": self._old_object_id(f),
                                "wkt": g.asWkt(),
                            }
                        )
                        continue
                except Exception:
                    pass

            if g_out and not g_out.isEmpty():
                nf = QgsFeature(f)
                nf.setGeometry(g_out)
                feats_out.append(nf)
                outer_list.append(
                    {
                        "old_object_id": self._old_object_id(f),
                        "wkt": g_out.asWkt(),
                    }
                )

        if feedback.isCanceled() or self._cancelled:
            return
        # Add non-empty outputs with full style intact
        if feats_in:
            prov_in.addFeatures(feats_in)
            mem_in.updateExtents()
            project.addMapLayer(mem_in, False)
            grp_innen.addLayer(mem_in)
            logger.info(
                "[Split]   → Innen: %d features in '%s'.", len(feats_in), layer_name
            )
        if feats_out:
            prov_out.addFeatures(feats_out)
            mem_out.updateExtents()
            project.addMapLayer(mem_out, False)
            grp_außen.addLayer(mem_out)
            logger.info(
                "[Split]   → Außen: %d features in '%s'.", len(feats_out), layer_name
            )

    # ---------------- Geometry utils ----------------

    def _ensure_geom_crs(self, geom: "QgsGeometry", src_crs, dst_crs) -> "QgsGeometry":
        """Copy/transform geometry to destination CRS."""
        if src_crs == dst_crs:
            return QgsGeometry(geom)
        g = QgsGeometry(geom)
        ct = QgsCoordinateTransform(src_crs, dst_crs, QgsProject.instance())
        g.transform(ct)
        return g

    # ---------------- Layer cloning / helpers ----------------

    def _split_feature_geom(self, feat_geom: "QgsGeometry", boundary: "QgsGeometry"):
        """(Kept for reference; not used in CRS-aware path)"""
        try:
            inter = feat_geom.intersection(boundary)
        except Exception:
            inter = None
        try:
            diff = feat_geom.difference(boundary)
        except Exception:
            diff = None
        inside = inter if (inter and not inter.isEmpty()) else None
        outside = diff if (diff and not diff.isEmpty()) else None
        return inside, outside

    def _empty_mem_clone(self, orig: QgsVectorLayer, new_name: str) -> QgsVectorLayer:
        """
        Create an empty in-memory layer matching schema/CRS/WKB and copy:
        full style, display expression, custom properties, joins.
        """
        crs_auth = orig.crs().authid()
        wkb = orig.wkbType()
        mem = QgsVectorLayer(
            f"{QgsWkbTypes.displayString(wkb)}?crs={crs_auth}",
            new_name,
            "memory",
        )
        prov = mem.dataProvider()
        prov.addAttributes(orig.fields())
        mem.updateFields()

        # Full style (renderer + labeling + symbol settings)
        try:
            style = QgsMapLayerStyle()
            style.readFromLayer(orig)
            style.writeToLayer(mem)
        except Exception:
            if orig.renderer():
                mem.setRenderer(orig.renderer().clone())

        mem.setDisplayExpression(orig.displayExpression())

        # Custom properties (version-safe)
        try:
            props = orig.customProperties()
            keys = (
                list(props.keys())
                if hasattr(props, "keys")
                else list(getattr(props, "propertyKeys")())
            )
            for k in keys:
                mem.setCustomProperty(k, orig.customProperty(k))
        except Exception:
            pass

        # Joins
        try:
            for j in orig.vectorJoins():
                mem.addJoin(QgsVectorLayerJoinInfo(j))
        except Exception:
            pass

        return mem

    def _layer_has_customprop_substr(self, lyr: QgsVectorLayer, needle: str) -> bool:
        """True if ANY custom property key or value contains `needle` (case-insensitive)."""
        try:
            n = needle.lower()
            # QGIS API varies: in some versions you have customPropertyKeys(), in others customProperties()
            if hasattr(lyr, "customPropertyKeys"):
                for k in lyr.customPropertyKeys():
                    v = lyr.customProperty(k)
                    if n in str(k).lower() or n in str(v).lower():
                        return True
            else:
                for k, v in getattr(lyr, "customProperties")().items():
                    if n in str(k).lower() or n in str(v).lower():
                        return True
        except Exception:
            pass
        return False

    def _group_has_xplanplugin_fast(self, grp: QgsLayerTreeGroup) -> bool:
        """Short-circuiting DFS: returns True as soon as a matching layer is found."""
        stack = [grp]
        n = PLUGIN_KEY
        while stack:
            g = stack.pop()
            for child in g.children():
                if not self._is_alive(child):
                    continue
                t = child.nodeType()
                if t == child.NodeLayer:
                    try:
                        lyr = child.layer()
                    except Exception:
                        lyr = None
                    if self._is_alive(lyr) and lyr.type() == lyr.VectorLayer:
                        if self._layer_has_customprop_substr(lyr, n):
                            return True  # exit early to save compute
                elif t == child.NodeGroup:
                    stack.append(child)
        return False

    def _transform_extent(self, extent, src_crs, dst_crs):
        if src_crs == dst_crs:
            return extent
        ct = QgsCoordinateTransform(src_crs, dst_crs, QgsProject.instance())
        return ct.transformBoundingBox(extent)

    def _plan_group_extent_in(self, grp: QgsLayerTreeGroup, target_crs):
        """Union of extents of all vector layers in the group (recursively), transformed to target_crs."""
        ext = None
        stack = [grp]
        while stack:
            g = stack.pop()
            for child in g.children():
                if child.nodeType() == child.NodeLayer:
                    lyr = child.layer()
                    if lyr and lyr.type() == lyr.VectorLayer:
                        try:
                            e = self._transform_extent(
                                lyr.extent(), lyr.crs(), target_crs
                            )
                            ext = e if ext is None else ext.united(e)
                        except Exception:
                            pass
                elif child.nodeType() == child.NodeGroup:
                    stack.append(child)
        return ext

    # Helper to make sure no non-existing C++ object is being touched to avoid ACCESS_VIOLATION_ERROR-crashes
    def _is_alive(self, obj) -> bool:
        try:
            return obj is not None and not sip.isdeleted(obj)
        except Exception:
            return False

    # Use a name/id reference of objects instead of using the objects themselves to avoid long living C++-object references
    def _snapshot_group(self, grp: QgsLayerTreeGroup) -> dict:
        """
        Take a pure-Python snapshot of a group:
        { 'name': str, 'layers': [layer_id, ...], 'groups': [subgroup_dict, ...] }
        """
        snap = {"name": grp.name(), "layers": [], "groups": []}
        for child in grp.children():
            try:
                # Guard in case node vanished mid-iteration
                if not self._is_alive(child):
                    continue
                t = child.nodeType()
                if t == child.NodeLayer:
                    # Store layer ID only; no live layer object
                    lid = (
                        child.layerId()
                        if hasattr(child, "layerId")
                        else (
                            child.layer().id()
                            if self._is_alive(child.layer())
                            else None
                        )
                    )
                    if lid:
                        snap["layers"].append(lid)
                elif t == child.NodeGroup:
                    snap["groups"].append(self._snapshot_group(child))
            except Exception:
                continue
        return snap

    def _first_customprop_value(self, lyr: QgsVectorLayer, key: str):
        try:
            if hasattr(lyr, "customProperty"):
                val = lyr.customProperty(key, None)
                return val if val not in ("", None) else None
        except Exception:
            pass
        return None

    def _find_plan_id(
        self, grp: QgsLayerTreeGroup, key: str = PLAN_KEY
    ) -> Optional[str]:
        """Walk group; return first non-empty custom property `plan_id` found on any layer."""
        stack = [grp]
        while stack:
            g = stack.pop()
            for child in g.children():
                if not self._is_alive(child):
                    continue
                if child.nodeType() == child.NodeLayer:
                    lyr = child.layer()
                    if self._is_alive(lyr) and lyr.type() == lyr.VectorLayer:
                        pid = self._first_customprop_value(lyr, key)
                        if pid:
                            return str(pid)
                elif child.nodeType() == child.NodeGroup:
                    stack.append(child)
        return None

    def _extract_appschema_firsthit_from_group(
        self, group: QgsLayerTreeGroup
    ) -> AppSchema:
        """Depth-first walk through all vector layers in the group.
        Return the first layer that carries both App-Schema custom properties."""
        project = QgsProject.instance()
        group_stack: list[QgsLayerTreeGroup] = [group]

        while group_stack:
            current_group = group_stack.pop()
            for child_item in current_group.children():
                if isinstance(child_item, QgsLayerTreeGroup):
                    group_stack.append(child_item)
                else:  # QgsLayerTreeLayer
                    layer = project.mapLayer(child_item.layerId())
                    if not isinstance(layer, QgsVectorLayer):
                        continue
                    app_name = layer.customProperty(APP_KEY, None)
                    app_vers = layer.customProperty(VER_KEY, None)
                    if app_name and app_vers:
                        return AppSchema(str(app_name), str(app_vers))

        raise QgsProcessingException("Kein Appschema in Plan-Gruppe gefunden.")

    def _new_feature_lists(self):
        """
        Create a fresh accumulator for processing split layer features.

        Returns:
            list: Flat lists for accumulating features segregated by their roles
                  ('inner' and 'outer').
        """
        return {"inner": [], "outer": []}

    def _old_object_id(self, feature: QgsFeature) -> str:
        idx = feature.fields().indexOf(ORIG_ID_FIELD)  # ORIG_ID_FIELD = "id"
        layer_name = (
            feature.layer().name()
            if hasattr(feature, "layer") and feature.layer()
            else "?"
        )
        if idx == -1:
            raise QgsProcessingException(
                f"Layer '{layer_name}' hat kein benötigtes ID-Feld '{ORIG_ID_FIELD}'. "
                "Bitte sicherstellen, dass die Original-UUID im Layer vorhanden ist."
            )
        val = feature[ORIG_ID_FIELD]
        if val in (None, ""):
            raise QgsProcessingException(
                f"Feature in Layer '{layer_name}' hat leeres '{ORIG_ID_FIELD}'."
            )
        return str(val)

    def _acc_has_data(self, role_list: list) -> bool:
        """
        Check whether any layer list under the given role contains features.

        Args:
            role_list (dict): Dictionary representing either the 'inner' or 'outer'
                              role's accumulation data.

        Returns:
            bool: True if any list contains features, False otherwise.
        """
        return bool(role_list)
