# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Anton Jacobsson <anton.jacobsson@init.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2

import json
import logging

from packaging.requirements import Requirement
from packaging.version import Version
from qgis.core import (
    Qgis,
    QgsApplication,
    QgsNetworkAccessManager,
)
from qgis.PyQt.QtCore import QObject, QTimer, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest
from qgis.utils import iface

from xmas_plugin.util.metadata import DEPENDENCIES, PLUGIN_DIR_NAME, PLUGIN_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


class HealthChecker(QObject):
    statusChanged = pyqtSignal(bool)
    error = pyqtSignal(str)

    def __init__(self, parent=None, user_agent=PLUGIN_NAME):
        super().__init__(parent)
        self.nam = QgsNetworkAccessManager(self)
        self.user_agent = user_agent
        self._reply = None
        self._timeout = QTimer(self)
        self._timeout.setSingleShot(True)
        self._timeout.timeout.connect(self._on_timeout)
        self._version_checked_for: str | None = None

    def check(self, url: str, timeout_ms: int = 15000):
        if self._reply is not None:
            self._reply.abort()
            self._cleanup()

        req = QNetworkRequest(QUrl(url))
        req.setRawHeader(b"User-Agent", self.user_agent.encode("utf-8"))
        req.setAttribute(
            QNetworkRequest.CacheLoadControlAttribute, QNetworkRequest.AlwaysNetwork
        )
        req.setAttribute(QNetworkRequest.FollowRedirectsAttribute, True)
        req.setRawHeader(b"Cache-Control", b"no-cache")
        req.setRawHeader(b"Pragma", b"no-cache")
        # Explicitly GET
        self._reply = self.nam.get(req)
        self._reply.finished.connect(self._on_finished)
        self._reply.errorOccurred.connect(self._on_error)
        self._timeout.start(timeout_ms)

    def _check_webapp_version(self) -> None:
        data = bytes(self._reply.readAll())
        if not data:
            raise ValueError("empty response")
        response = json.loads(data.decode("utf-8"))
        if not isinstance(response, dict):
            raise RuntimeError(
                f"unexpected response type for webapp version check: {type(response)!r}"
            )
        if not (version := response.get("version")):
            raise RuntimeError(f"version key not found in response: {response}")
        app_version = Version(version)

        dep = DEPENDENCIES["optional"][0]
        if not dep.lower().startswith("xmas"):
            raise RuntimeError(
                f"unexpected dependency for webapp version check: {dep!r}"
            )
        app_req = Requirement(dep).specifier
        if not app_req.contains(app_version):
            msg = f"Webapp-Version '{app_version}' entspricht nicht nicht der erwarteten Version '{app_req}'"
            iface.messageBar().pushWarning(PLUGIN_NAME, msg)
            QgsApplication.messageLog().logMessage(
                message=f"""{msg}:\n
                Das Plugin funktioniert ggf. nicht wie erwartet, bitte Installation anpassen.""",
                tag=PLUGIN_NAME,
                level=Qgis.MessageLevel.Warning,
            )
        self._version_checked_for = self._reply.url().toString()

    def stop(self):
        if self._reply is not None:
            self._reply.abort()
        self._cleanup()

    def _on_finished(self):
        if self._reply is None:
            return
        code = self._reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)
        err = self._reply.error()
        ok = (err == QNetworkReply.NoError) and (code in (200, 204, 301, 302, 307, 308))

        # Only log when something's wrong
        if not ok:
            logger.warning(
                "[Health] finished bad: code=%s err=%s ok=%s", code, int(err), ok
            )
        # Check webapp version once for any URL (in case it changed via settings)
        elif not self._version_checked_for == self._reply.url().toString():
            try:
                logger.info("[Health] checking webapp version for compatibility")
                self._check_webapp_version()
            except Exception as e:
                logger.exception("[Health] webapp version check failed: %s", e)
                msg = "Überprüfung der Webapp-Version gescheitert"
                iface.messageBar().pushWarning(PLUGIN_NAME, msg)
                QgsApplication.messageLog().logMessage(
                    message=f"{msg}:\n{e!r}",
                    tag=PLUGIN_NAME,
                    level=Qgis.MessageLevel.Warning,
                )

        self.statusChanged.emit(bool(ok))
        self._cleanup()

    def _on_error(self, _code):
        if (
            self._reply is not None
            and self._reply.error() != QNetworkReply.OperationCanceledError
        ):
            self.statusChanged.emit(False)
            self.error.emit(self._reply.errorString())
        self._version_checked_for = None
        self._cleanup()

    def _on_timeout(self):
        if self._reply is not None:
            self._reply.abort()
        self.statusChanged.emit(False)
        self.error.emit("Health check timed out")
        self._cleanup()

    def _cleanup(self):
        self._timeout.stop()
        if self._reply is not None:
            self._reply.deleteLater()
            self._reply = None
