# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2-or-later

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsEditError,
    QgsFeature,
    QgsMapLayer,
    QgsProject,
    QgsVectorLayerUtils,
    edit,
)
from qgis.gui import (
    QgsAttributeEditorContext,
    QgsMapLayerAction,
    QgsMapLayerActionContext,
    QgsMapToolDigitizeFeature,
)
from qgis.PyQt.QtCore import QObject, pyqtSlot
from qgis.utils import iface

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME


class AddPO(QgsMapLayerAction):
    def __init__(self, parent: QObject | None = None):
        super().__init__(
            "Präsentationsobjekt hinzufügen",
            parent,
            Qgis.MapLayerActionTarget.SingleFeature,
            QgsApplication.instance().getThemeIcon("title_label.svg"),
            Qgis.MapLayerActionFlags(),
        )
        self.feature = None
        self.layer = None
        self.target_layer = None
        self.dlg = None
        self.digit_tool = None
        self.triggeredForFeatureV2.connect(self._setup)

    def canRunUsingLayer(
        self,
        layer: QgsMapLayer | None,
        _: QgsMapLayerActionContext = QgsMapLayerActionContext(),
    ) -> bool:
        return (
            layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type") == "subject"
            and layer.customProperty(f"{PLUGIN_DIR_NAME}/appschema") == "xplan"
            if layer
            else False
        )

    @pyqtSlot("QgsMapLayer*", QgsFeature, QgsMapLayerActionContext)
    def _setup(
        self,
        layer: QgsMapLayer,
        feature: QgsFeature,
        _: QgsMapLayerActionContext,
    ) -> None:
        self.layer = layer
        self.feature = feature
        for map_layer in QgsProject.instance().mapLayers().values():
            layer_type = map_layer.customProperty(f"{PLUGIN_DIR_NAME}/layer_type", "")
            if (
                "presentation" == layer_type
                and map_layer.geometryType().name == "Point"
                and map_layer.customProperty(f"{PLUGIN_DIR_NAME}/bereich_id")
                == self.layer.customProperty(f"{PLUGIN_DIR_NAME}/bereich_id")
            ):
                self.target_layer = map_layer
                break
        if not self.target_layer:
            return iface.pushCritical(
                "Neues Präsentationsobjekt",
                "kein passender Präsentationsobjekte-Layer gefunden",
            )
        self._digitize_po()

    def _digitize_po(self):
        canvas = iface.mapCanvas()
        self.digit_tool = QgsMapToolDigitizeFeature(canvas, iface.cadDockWidget())
        self.digit_tool.setLayer(self.target_layer)
        self.digit_tool.digitizingCompleted.connect(self._handle_digitizing)
        self.digit_tool.digitizingCanceled.connect(self._cleanup)
        self.digit_tool.digitizingFinished.connect(self._cleanup)
        self.target_layer.startEditing()
        canvas.setMapTool(self.digit_tool)
        iface.messageBar().pushInfo("Neues Präsentationsobjekt", "Geometrie erfassen")

    @pyqtSlot(QgsFeature)
    def _handle_digitizing(self, digit_feature: QgsFeature):
        self.target_layer.rollBack()
        feature = QgsVectorLayerUtils.createFeature(
            self.target_layer,
            digit_feature.geometry(),
            {
                self.target_layer.fields().indexFromName("properties"): {
                    "dientZurDarstellungVon": [self.feature["id"]],
                },
            },
        )
        self.dlg = iface.getFeatureForm(self.target_layer, feature)
        self.dlg.setMode(QgsAttributeEditorContext.Mode.AddFeatureMode)
        try:
            with edit(self.target_layer):
                self.dlg.exec()
        except QgsEditError as e:
            iface.messageBar().pushCritical("Neues Präsentationsobjekt", repr(e))
        else:
            self.layer.reload()
            iface.messageBar().pushSuccess(
                "Neues Präsentationsobjekt", "Präsentationsobjekt hinzugefügt"
            )
        finally:
            self._cleanup()

    def _cleanup(self):
        self.feature = None
        self.layer = None
        self.target_layer = None
        if self.dlg:
            self.dlg.deleteLater()
            self.dlg = None
        if self.digit_tool:
            self.digit_tool.deleteLater()
            self.digit_tool = None
