# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2

import logging
import os

from qgis.PyQt.QtCore import QObject
from qgis.PyQt.QtWebEngineWidgets import (
    QWebEnginePage,
    QWebEngineProfile,
    QWebEngineSettings,
)

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME, PLUGIN_NAME, PLUGIN_VERSION

logger = logging.getLogger(PLUGIN_DIR_NAME)


class XMASPluginPage(QWebEnginePage):
    """Extends QWebEnginePage with sensible defaults and logging.

    - sets an off-the-record profile with memory cache
    - adds logging for possible error sources and conditionally activates debugging
    - if loading valid URLs fails:
        - activates error pages
        - adds logging
        - sets debug flags
    - ignores certificate errors to allow self-signed certificates in webapp
    """

    def __init__(self, parent: QObject | None = None):
        # Dedicated, off-the-record profile to reduce disk I/O
        profile = QWebEngineProfile(None)
        profile.setHttpCacheType(QWebEngineProfile.MemoryHttpCache)
        profile.setHttpCacheMaximumSize(16 * 1024 * 1024)
        profile.setPersistentCookiesPolicy(QWebEngineProfile.NoPersistentCookies)
        profile.setHttpUserAgent(f"{PLUGIN_NAME}/{PLUGIN_VERSION}")
        super().__init__(profile, parent)
        profile.setParent(self)

        s = self.settings()
        s.setAttribute(QWebEngineSettings.AutoLoadIconsForPage, False)
        s.setAttribute(QWebEngineSettings.ScrollAnimatorEnabled, False)
        s.setAttribute(QWebEngineSettings.JavascriptCanAccessClipboard, False)

        self._debug: bool = False

        self.loadFinished.connect(self._on_load_finished)
        self.renderProcessTerminated.connect(
            lambda status, code: logger.warning(
                "render process for URL %s terminated with exit code %s: %s",
                self.url().toString(),
                code,
                status,
            )
        )

    def _on_load_finished(self, ok: bool):
        if ok:
            return
        url = self.url()
        logger.error(
            "failed loading URL '%s'",
            url.toString(),
        )
        if not self._debug:
            logger.info(
                "reloading %s with debug flags enabled",
                url.toString(),
            )
            flags = os.environ.get("QTWEBENGINE_CHROMIUM_FLAGS", "").split(" ")
            flags.extend(["--enable-logging", "--log-level=0", "--v=1"])
            os.environ["QTWEBENGINE_CHROMIUM_FLAGS"] = " ".join(flags)
            self._debug = True
            self.load(url)

    def certificateError(self, certificateError):
        """Log, but ignore SSL certificate errors."""
        logger.warning("Certificate error: %s", certificateError)
        return True

    def javaScriptConsoleMessage(self, level, message, lineNumber, sourceID):
        """Log js console message according to severity level."""
        # filter out spammy, but irrelevant message
        if "ResizeObserver loop limit exceeded" in message:
            return
        log_level = {0: "debug", 1: "warning", 2: "error"}
        getattr(logger, log_level.get(level, "error"))(
            "[JS] %s:%s: %s", sourceID, lineNumber, message
        )
