# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2
import logging

from qgis.PyQt.QtCore import QObject, QTimer, pyqtSlot
from qgis.utils import iface

from xmas_plugin.util.geom import transform_to_project_crs
from xmas_plugin.util.layer import get_layer_and_feature
from xmas_plugin.util.metadata import PLUGIN_DIR_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


class FeatureInteractionHandler(QObject):
    def __init__(self, parent=None):
        super().__init__(parent)

    def _normalize_feature_payload(self, data) -> dict:
        if isinstance(data, dict):
            return data
        if isinstance(data, str):
            return {"target": data.strip()}
        raise TypeError(f"Unexpected payload type: {type(data)}: {data!r}")

    @pyqtSlot("QVariant")
    def highlight_feature(self, data):
        """Highlights a feature (by UUID or FID) and zooms in on it."""
        logger.info("highlight_feature data type=%s value=%r", type(data), data)

        payload = self._normalize_feature_payload(data)
        target = payload.get("target")

        if not target:
            raise ValueError(f"No target provided in payload: {data!r}")

        target_layer, target_feature = get_layer_and_feature(target)

        if target_layer is None or target_feature is None:
            iface.messageBar().pushCritical(
                "Zum Feature springen", "Feature nicht gefunden"
            )
            logger.debug("No layer/feature found for feature_key=%s", target)
            return

        geom = transform_to_project_crs(target_feature.geometry(), target_layer.crs())

        iface.mapCanvas().setExtent(geom.boundingBox())
        iface.mapCanvas().flashGeometries([geom])
        iface.mapCanvas().refresh()

        logger.debug("Zoom and highlight successful")

    @pyqtSlot("QVariant")
    def show_attribute_form(self, data):
        """Open the QGIS feature form for a specific feature ID."""
        logger.debug(
            "show_attribute_form data is of type: %s, value: %r", type(data), data
        )

        payload = self._normalize_feature_payload(data)
        target = payload.get("target")
        source = payload.get("source")

        if not target:
            raise ValueError(f"No target provided in payload: {data!r}")

        target_layer, target_feature = get_layer_and_feature(target)
        if target_layer is None or target_feature is None:
            iface.messageBar().pushCritical(
                "Attributformular öffnen", "Layer/Feature nicht gefunden"
            )
            return

        if source:
            logger.debug("Retrieving source layer for feature ID %r", source)
            source_layer, _ = get_layer_and_feature(source)

            if not source_layer:
                logger.warning("No source layer for feature ID %r found", source)
            elif source_layer.isEditable() and not target_layer.isEditable():
                success = target_layer.startEditing()
                if not success:
                    logger.warning(
                        "Failed to activate editing for target layer %r",
                        target_layer.id(),
                    )

        QTimer.singleShot(
            0, lambda: iface.openFeatureForm(target_layer, target_feature)
        )
        logger.debug(
            f"attribute form opened for layer={target_layer.id()}, FID={target_feature.id()}, feature_key= {target}"
        )

    @pyqtSlot("QVariant")
    def select_feature(self, data: dict):
        """Open the QGIS feature form for a specific feature ID."""
        target = data["target"]

        target_layer, target_feature = get_layer_and_feature(target)

        if target_layer is None or target_feature is None:
            iface.messageBar().pushCritical(
                "Feature selektieren", "Layer/Feature nicht gefunden"
            )
            return
        logger.debug(
            f"Found: layer={target_layer.id()}, FID={target_feature.id()} for feature_key= {target}"
        )

        # Perform selection
        iface.setActiveLayer(target_layer)
        target_layer.selectByIds([target_feature.id()])
        logger.debug(f"Feature selected by FID: {target_feature.id()}")

        # Jump to feature BBOX if it has a geometry
        if target_feature.hasGeometry():
            geom = transform_to_project_crs(
                target_feature.geometry(), target_layer.crs()
            )
            iface.mapCanvas().setExtent(geom.boundingBox())
            iface.mapCanvas().refresh()
