# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Michael Holzapfel <michael.holzapfel@geocledian.com>
#
# SPDX-License-Identifier: EUPL-1.2-or-later

import logging
from typing import Any, Dict, List, Optional

from qgis.core import (
    QgsFeature,
    QgsGeometry,
    QgsProject,
    QgsVectorLayer,
)
from qgis.gui import QgisInterface, QgsHighlight
from qgis.PyQt.QtCore import (
    QAbstractTableModel,
    QModelIndex,
    QSortFilterProxyModel,
    Qt,
    QVariant,
)

from xmas_plugin.topo_check import TopoCheckerHighlight, TopoTableView
from xmas_plugin.util.geom import transform_to_project_crs
from xmas_plugin.util.metadata import PLUGIN_DIR_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


class OverlapTableModel(QAbstractTableModel):
    def __init__(
        self,
        data: List[QgsFeature | QgsVectorLayer],
        columns: List[str],
        parent: Optional[Any] = None,
    ) -> None:
        super().__init__(parent)
        self._data = data
        self._columns = columns

    def rowCount(self, parent: QModelIndex = QModelIndex()) -> int:
        return len(self._data)

    def columnCount(self, parent: QModelIndex = QModelIndex()) -> int:
        return len(self._columns)

    def data(self, index: QModelIndex, role: int = Qt.DisplayRole) -> Any:
        if index.isValid() and role == Qt.UserRole:
            return self._data[index.row()][index.column()]
        if not index.isValid() or role != Qt.DisplayRole:
            return QVariant()
        row = self._data[index.row()]
        item = row[index.column()]
        if index.column() == 0:
            return f"{self._data[index.row()][0].area():0.2f} m²"
        if isinstance(item, QgsFeature):
            item: QgsFeature
            return item.attribute("id")
        elif isinstance(item, QgsVectorLayer):
            item: QgsVectorLayer
            try:
                return item.name()
            except RuntimeError as e:
                logger.debug(f"qgsvector layer probably removed: {e}")
                return QVariant()
        else:
            return QVariant()

    def headerData(
        self, section: int, orientation: Qt.Orientation, role: int = Qt.DisplayRole
    ) -> Any:
        if role != Qt.DisplayRole:
            return QVariant()
        if orientation == Qt.Horizontal:
            return self._columns[section]
        return section + 1

    def repopulate(self, new_data) -> None:
        # Notify views that model is about to reset
        self.beginResetModel()
        self._data = new_data
        # Notify views that model reset is done
        self.endResetModel()


class OverlapTableView(TopoTableView):
    def __init__(self, iface: QgisInterface, parent=None):
        super().__init__(parent)
        self.iface = iface

        self.model = OverlapTableModel(
            data=[], columns=["Fläche", "Feature1", "Layer1", "Feature2", "Layer2"]
        )
        self.proxymodel = QSortFilterProxyModel()
        self.proxymodel.setSourceModel(self.model)
        self.setModel(self.proxymodel)
        self.setSortingEnabled(True)
        self.resizeColumnsToContents()
        self.highlights: Dict[
            QModelIndex, QgsHighlight
        ] = {}  # TODO: remove highlights on signal layer unload to prevent orphan highlights

        self.clicked.connect(self.on_click)

    def on_click(self, index: QModelIndex) -> None:
        def _select_layer(layer: QgsVectorLayer) -> None:
            """
            Highlights/selects a layer in the QGis Layers Dock,
            if the layer is part of a (nested) group QGis should expand
            thos groups automatically
            :param layer:
            :return:
            """
            if not layer:
                return

            root = QgsProject.instance().layerTreeRoot()
            layer_node = root.findLayer(layer.id())
            if layer_node is None:
                return

            self.iface.layerTreeView().setCurrentLayer(layer)
            return

        self.clear_highlights()

        src_index: QModelIndex = index.model().mapToSource(index)
        item: QgsGeometry | QgsFeature = self.model.data(src_index, Qt.UserRole)

        # select clicked layer in qgis layertree view
        if index.column() in [2, 4]:
            _select_layer(item)
            return

        if isinstance(item, QgsFeature):  # click on feature column
            _geom = item.geometry()
        elif isinstance(item, QgsGeometry):  # click on overlap-geom column
            _geom = item

        geom = transform_to_project_crs(_geom, self.plan_crs)

        # create and apply highlight
        self.highlights[index] = TopoCheckerHighlight(
            self.iface.mapCanvas(), geom, None
        )
        self.highlights[index].setColor(self.highlight_color)
        self.highlights[index].setFillColor(self.highlight_color)
        self.highlights[index].show()
        extent = geom.boundingBox()

        # zoom to geometry
        canvas = self.iface.mapCanvas()
        canvas.setExtent(extent)
        canvas.refresh()

    def clear_highlights(self) -> None:
        for i, h in self.highlights.items():
            if isinstance(h, QgsHighlight):
                h.hide()
        self.highlights.clear()

    def repopulate(self, new_data) -> None:
        self.clear_highlights()
        self.model.repopulate(new_data)
