# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Anton Jacobsson <anton.jacobsson@init.de>
#
# SPDX-License-Identifier: EUPL-1.2-or-later

import logging

import psycopg2
from qgis.core import QgsProviderRegistry

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


def get_db_uri(connection_name) -> str:
    """
    Retrieve the database URI from QGIS Postgres connections.
    """
    provider_metadata = QgsProviderRegistry.instance().providerMetadata("postgres")
    if not provider_metadata:
        raise ValueError("Postgres provider not found in QGIS.")

    connections = provider_metadata.connections()
    if connection_name not in connections:
        raise ValueError(f"Connection '{connection_name}' not found.")

    conn = connections[connection_name]
    return conn.uri()


def test_db_connection(uri) -> bool:
    """
    Test the database connection using the provided URI.
    """
    try:
        conn = psycopg2.connect(uri)
        conn.close()
        return True
    except Exception as e:
        logging.error(f"Database connection test failed: {e}")
        return False


def verify_db_connection_schema(uri) -> bool:
    """
    Verify that the database connection schema contains required tables.
    """
    try:
        conn = psycopg2.connect(uri)
        cur = conn.cursor()
        cur.execute(
            "SELECT tablename FROM pg_catalog.pg_tables WHERE schemaname = 'public';"
        )
        tables = {row[0] for row in cur.fetchall()}
        conn.close()

        required_tables = {"refs", "coretable"}
        if required_tables.issubset(tables):
            return True
        else:
            logging.warning("Required tables are missing in the database.")
            return False
    except Exception as e:
        logging.error(f"Database schema verification failed: {e}")
        return False


def execute_query(uri: str, query: str, params: tuple = None) -> list[tuple]:
    """
    Execute a parameterized SQL query against a PostgreSQL database.

    :param uri: Database connection URI.
    :type uri: str

    :param query: SQL query string, with placeholders for parameters (if any).
    :type query: str

    :param params: Optional tuple of parameters to use with the SQL query.
    :type params: tuple or None

    :return: List of tuples containing the query results. Each tuple represents a row.
    :rtype: list[tuple]

    :raises ValueError: If the query fails to execute.
    """
    try:
        with psycopg2.connect(uri) as conn:
            with conn.cursor() as cursor:
                cursor.execute(query, params)
                results = cursor.fetchall()
                if results:
                    logging.info(f"Query successful. Results: {results}")
                else:
                    logging.info("Query successful but returned no results.")
        return results
    except Exception as e:
        logging.error(f"Query execution failed: {e}")
        raise ValueError(f"Failed to execute query: {e}")
