# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Anton Jacobsson <anton.jacobsson@init.de>
#
# SPDX-License-Identifier: EUPL-1.2-or-later

import logging

from qgis.PyQt.QtCore import QObject, QTimer, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkAccessManager, QNetworkReply, QNetworkRequest

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME, PLUGIN_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


class HealthChecker(QObject):
    statusChanged = pyqtSignal(bool)
    error = pyqtSignal(str)

    def __init__(self, parent=None, user_agent=PLUGIN_NAME):
        super().__init__(parent)
        self.nam = QNetworkAccessManager(self)
        self.user_agent = user_agent
        self._reply = None
        self._timeout = QTimer(self)
        self._timeout.setSingleShot(True)
        self._timeout.timeout.connect(self._on_timeout)

    def check(self, url: str, timeout_ms: int = 15000):
        if self._reply is not None:
            self._reply.abort()
            self._cleanup()

        req = QNetworkRequest(QUrl(url))
        req.setRawHeader(b"User-Agent", self.user_agent.encode("utf-8"))
        req.setAttribute(
            QNetworkRequest.CacheLoadControlAttribute, QNetworkRequest.AlwaysNetwork
        )
        req.setAttribute(QNetworkRequest.FollowRedirectsAttribute, True)
        req.setRawHeader(b"Cache-Control", b"no-cache")
        req.setRawHeader(b"Pragma", b"no-cache")
        # Explicitly GET
        self._reply = self.nam.get(req)
        self._reply.finished.connect(self._on_finished)
        self._reply.errorOccurred.connect(self._on_error)
        self._timeout.start(timeout_ms)

    def stop(self):
        if self._reply is not None:
            self._reply.abort()
        self._cleanup()

    def _on_finished(self):
        if self._reply is None:
            return
        code = self._reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)
        err = self._reply.error()
        ok = (err == QNetworkReply.NoError) and (code in (200, 204, 301, 302, 307, 308))

        # Only log when something's wrong
        if not ok:
            logger.warning(
                "[Health] finished bad: code=%s err=%s ok=%s", code, int(err), ok
            )

        self.statusChanged.emit(bool(ok))
        self._cleanup()

    def _on_error(self, _code):
        if (
            self._reply is not None
            and self._reply.error() != QNetworkReply.OperationCanceledError
        ):
            self.statusChanged.emit(False)
            self.error.emit(self._reply.errorString())
        self._cleanup()

    def _on_timeout(self):
        if self._reply is not None:
            self._reply.abort()
        self.statusChanged.emit(False)
        self.error.emit("Health check timed out")
        self._cleanup()

    def _cleanup(self):
        self._timeout.stop()
        if self._reply is not None:
            self._reply.deleteLater()
            self._reply = None
