# SPDX-FileCopyrightText: 2025 XLeitstelle Planen und Bauen <xleitstelle@gv.hamburg.de>
# SPDX-FileContributor: Tobias Kraft <tobias.kraft@gv.hamburg.de>
#
# SPDX-License-Identifier: EUPL-1.2-or-later

import logging

from qgis.core import (
    Qgis,
    QgsAttributeEditorField,
    QgsDefaultValue,
    QgsEditorWidgetSetup,
    QgsFieldConstraints,
    QgsVectorLayer,
)

from xmas_plugin.util.metadata import PLUGIN_DIR_NAME

logger = logging.getLogger(PLUGIN_DIR_NAME)


def configure_layer_form(layer: QgsVectorLayer, appschema: str, version: str):
    """Configures edit form, widgets and fields for the plugin."""
    form = layer.editFormConfig()
    form.setLayout(Qgis.AttributeFormLayout.DragAndDrop)
    root = form.invisibleRootContainer()
    root.clear()
    for index, field in enumerate(layer.fields()):
        match name := field.name():
            case "id":
                layer.setDefaultValueDefinition(
                    index, QgsDefaultValue("uuid('WithoutBraces')")
                )
                form.setReadOnly(index, True)
                # generate new UUID on splitting, duplication and merging
                field.setDuplicatePolicy(Qgis.FieldDuplicatePolicy.DefaultValue)
                field.setSplitPolicy(Qgis.FieldDomainSplitPolicy.DefaultValue)
                if Qgis.versionInt() >= 34400:
                    field.setMergePolicy(Qgis.FieldDomainMergePolicy.DefaultValue)
                editor_field = QgsAttributeEditorField(name, index, root)
                editor_field.setShowLabel(False)
                root.addChildElement(editor_field)
            case "properties":
                layer.setEditorWidgetSetup(
                    index, QgsEditorWidgetSetup(PLUGIN_DIR_NAME, {})
                )
                editor_field = QgsAttributeEditorField(name, index, root)
                editor_field.setShowLabel(False)
                editor_field.setVerticalStretch(2)
                root.addChildElement(editor_field)
                layer.setFieldConstraint(
                    index,
                    QgsFieldConstraints.ConstraintNotNull,
                    QgsFieldConstraints.ConstraintStrengthHard,
                )
            case "appschema" | "version":
                layer.setDefaultValueDefinition(
                    index,
                    QgsDefaultValue(
                        f"'{appschema if name == 'appschema' else version}'"
                    ),
                )
                layer.setEditorWidgetSetup(index, QgsEditorWidgetSetup("Hidden", {}))
            case _:
                layer.setEditorWidgetSetup(index, QgsEditorWidgetSetup("Hidden", {}))
    layer.setEditFormConfig(form)
    layer.setDisplayExpression("featuretype || ' '  || left(id, 8)")
    logger.debug("layer form config successful")
