from typing import Any, Dict, List, Optional, Tuple

from qgis.core import QgsGeometry, QgsProject, QgsVectorLayer
from qgis.gui import QgisInterface, QgsHighlight
from qgis.PyQt.QtCore import (
    QAbstractTableModel,
    QModelIndex,
    QSortFilterProxyModel,
    Qt,
    QVariant,
)

from xmas_plugin.topo_check import TopoCheckerHighlight, TopoTableView
from xmas_plugin.util.geom import transform_to_project_crs


class BoundsTableModel(QAbstractTableModel):
    def __init__(
        self,
        data: List[Dict[str, Any]],
        columns: List[str],
        parent: Optional[Any] = None,
    ) -> None:
        super().__init__(parent)
        self._data = data
        self._columns = columns

    def rowCount(self, parent: QModelIndex = QModelIndex()) -> int:
        return len(self._data)

    def columnCount(self, parent: QModelIndex = QModelIndex()) -> int:
        return len(self._columns)

    def data(self, index: QModelIndex, role: int = Qt.DisplayRole) -> Any:
        if index.isValid() and role == Qt.UserRole:
            item = self._data[index.row()]
            return item
        if not index.isValid() or role != Qt.DisplayRole:
            return QVariant()
        try:
            if index.column() == 0:
                return str(self._data[index.row()]["feature"].attribute("id"))
            elif index.column() == 1:
                return str(self._data[index.row()]["layer"].name())
            elif index.column() == 2:
                return str(self._data[index.row()]["Typ"])
            else:
                return QVariant()
        except RuntimeError:
            # TODO: logger message that probably a layer was deleted
            # "wrapped C/C++ object of type QgsVectorLayer has been deleted"
            return QVariant()

    def headerData(
        self, section: int, orientation: Qt.Orientation, role: int = Qt.DisplayRole
    ) -> Any:
        if role != Qt.DisplayRole:
            return QVariant()
        if orientation == Qt.Horizontal:
            return self._columns[section]
        return section + 1

    def repopulate(self, new_data) -> None:
        # Notify views that model is about to reset
        self.beginResetModel()
        self._data = new_data
        # Notify views that model reset is done
        self.endResetModel()


class BoundsTableView(TopoTableView):
    def __init__(self, iface: QgisInterface, parent=None):
        super().__init__(parent)
        self.iface = iface

        self.gap_features: List[Tuple[QgsGeometry, QgsVectorLayer]] = []

        self.model = BoundsTableModel([], ["Feature", "Layer", "Typ"])
        self.proxymodel = QSortFilterProxyModel()
        self.proxymodel.setSourceModel(self.model)
        self.setModel(self.proxymodel)
        self.setSortingEnabled(True)
        self.resizeColumnsToContents()
        self.highlights: Dict[
            QModelIndex, QgsHighlight
        ] = {}  # TODO: remove highlights on signal layer unload to prevent orphan highlights

        self.clicked.connect(self.on_click)

    def repopulate(self, new_data) -> None:
        self.clear_highlights()
        self.model.repopulate(new_data)

    def clear_highlights(self) -> None:
        for i, h in self.highlights.items():
            if isinstance(h, QgsHighlight):
                h.hide()
        self.highlights.clear()

    def on_click(self, index: QModelIndex) -> None:
        def _select_layer(layer: QgsVectorLayer) -> None:
            """
            Highlights/selects a layer in the QGis Layers Dock,
            if the layer is part of a (nested) group QGis should expand
            thos groups automatically
            :param layer:
            :return:
            """
            if not layer:
                return

            root = QgsProject.instance().layerTreeRoot()
            layer_node = root.findLayer(layer.id())
            if layer_node is None:
                return

            self.iface.layerTreeView().setCurrentLayer(layer)
            return

        def _select_gap_layers(
            gap_geom: QgsGeometry,
            gap_features: List[Tuple[QgsGeometry, QgsVectorLayer]],
        ) -> None:
            touching_layers = set()
            for gap, layer in gap_features:
                if gap.geometry().touches(gap_geom):
                    touching_layers.add(layer)
            for layer in touching_layers:
                _select_layer(layer)

        # collect geometry from data
        # noinspection PyUnresolvedReferences
        src_index: QModelIndex = index.model().mapToSource(index)
        row_data = self.model.data(src_index, Qt.UserRole)

        if index.column() == 0:
            # highlight feature
            geom = transform_to_project_crs(
                row_data["feature"].geometry(), self.plan_crs
            )

            self.clear_highlights()
            # create and show highlight
            self.highlights[index] = TopoCheckerHighlight(
                self.iface.mapCanvas(), geom, None
            )  # No layer needed
            # _color = QColor(255, 0, 100, 100)
            self.highlights[index].setColor(self.highlight_color)
            self.highlights[index].setFillColor(self.highlight_color)
            self.highlights[index].show()

            # zoom to geometry
            extent = geom.boundingBox()
            canvas = self.iface.mapCanvas()
            canvas.setExtent(extent)
            canvas.refresh()

        elif index.column() == 1:
            # highlight layer
            _select_layer(row_data["layer"])
        else:
            return
