import re
from pathlib import Path
from typing import TYPE_CHECKING, Any, Dict, List, Set, Tuple

from qgis.core import (
    Qgis,
    QgsFeature,
    QgsGeometry,
    QgsLayerTreeGroup,
    QgsLayerTreeLayer,
    QgsMapLayer,
    QgsProject,
    QgsVectorLayer,
    QgsWkbTypes,
)
from qgis.PyQt import uic
from qgis.PyQt.QtCore import pyqtSignal
from qgis.PyQt.QtWidgets import (
    QComboBox,
    QMessageBox,
    QPushButton,
    QTableView,
    QTabWidget,
    QWidget,
)

from xmas_plugin.topo_check.table_bounds import BoundsTableView
from xmas_plugin.topo_check.table_gaps import GapTableView
from xmas_plugin.topo_check.table_overlaps import OverlapTableView, logger
from xmas_plugin.util.metadata import PLUGIN_DIR_NAME

if TYPE_CHECKING:
    from xmas_plugin.dock import XMASPluginDockWidget  # Only for IDE/type checker


class TopologyChecker(QWidget):
    plan_layer_changed = pyqtSignal(name="plan_layer_changed")

    def __init__(self, parent: "XMASPluginDockWidget") -> None:
        super().__init__(parent)
        self.parent = parent
        self.iface = parent.qgis_iface
        self._topology_errors: Dict[str, List[Any]] | None = None

        # noinspection PyUnresolvedReferences
        ui_file = Path(__file__).parent.parent / "ui" / "topo_check_ui.ui"
        uic.loadUi(str(ui_file), self)

        self.comboBox_plans: QComboBox

        # initialize gaps-tableview
        _placeholder = self.findChild(QTableView, "table_gaps")
        _layout = _placeholder.parentWidget().layout()
        self.table_gaps = GapTableView(self.iface)
        _layout.replaceWidget(_placeholder, self.table_gaps)
        _placeholder.deleteLater()
        del _placeholder

        # initialize overlaps-tableview
        _placeholder = self.findChild(QTableView, "table_overlaps")
        _layout = _placeholder.parentWidget().layout()
        self.table_overlaps = OverlapTableView(self.iface)
        _layout.replaceWidget(_placeholder, self.table_overlaps)
        _placeholder.deleteLater()
        del _placeholder

        # initialize bounds-tableview
        _placeholder = self.findChild(QTableView, "table_bounds")
        _layout = _placeholder.parentWidget().layout()
        self.table_bounds = BoundsTableView(self.iface)
        _layout.replaceWidget(_placeholder, self.table_bounds)
        _placeholder.deleteLater()
        del _placeholder

        self.findChild(QPushButton, "pb_run_check").clicked.connect(self.run_topo_check)
        self.findChild(QPushButton, "pb_gap_rm_highlight").clicked.connect(
            self.table_gaps.clear_highlights
        )
        self.findChild(QPushButton, "pb_overlap_rm_highlight").clicked.connect(
            self.table_overlaps.clear_highlights
        )
        self.findChild(QPushButton, "pb_bounds_rm_highlight").clicked.connect(
            self.table_bounds.clear_highlights
        )

        self.plan_layers = {}
        self.parent.plan_manager.xplan_layer_added.connect(self.on_xplan_layer_added)
        QgsProject.instance().layerWillBeRemoved.connect(self.on_layer_removed)
        self.plan_layer_changed.connect(self.on_plan_layer_changed)
        self.comboBox_plans.currentIndexChanged.connect(self.on_current_plan_changed)

    def on_current_plan_changed(self, index: int):
        # reset all tables
        self.topology_errors = {
            "bounds": [],
            "overlaps": [],
            "gaps": [],
            "gap_features": [],
            "crs": None,
        }

    def on_plan_layer_changed(self) -> None:
        # keep the combobox with selection of loaded plans up to date
        previous_index = self.comboBox_plans.currentIndex()
        previous_item = self.comboBox_plans.itemData(previous_index)

        # clear combobox
        self.comboBox_plans.clear()

        # add items
        root = QgsProject.instance().layerTreeRoot()
        for k, v in self.plan_layers.items():
            try:
                layer: QgsVectorLayer = v["layer"]
                layer_node = root.findLayer(layer.id())
                parent_group = layer_node.parent()
                plan_name = parent_group.name()
                self.comboBox_plans.addItem(
                    plan_name, {"plan_id": v["plan_id"], "qgis_layerid": layer.id()}
                )
            except AttributeError as e:
                logger.debug(e)

        # set previous item if applicable
        if previous_item:
            prev_index_current = self.comboBox_plans.findData(previous_item)
            if prev_index_current != -1:
                self.comboBox_plans.setCurrentIndex(prev_index_current)
                return

        # invalidate topologie check if applicable
        try:
            # this will also clear the tableviews because topology_errors has a setter method
            self.topology_errors["gaps"].clear()
            self.topology_errors["overlaps"].clear()
        except TypeError:
            pass  # no need to clear topology result (prob. has not run yet)

    def on_xplan_layer_added(self, layer_info: Dict) -> None:
        # watch if a xplan-plan layer was added -> a plan has been loaded via plugin
        layer = layer_info["layer"]
        # check if layer is a plan-layer
        layer_type = layer.customProperties().value(
            f"{PLUGIN_DIR_NAME}/layer_type", None
        )
        if layer_type == "plan":
            self.plan_layers[layer_info["layer"].id()] = layer_info
            self.plan_layer_changed.emit()

    def on_layer_removed(self, layername: str) -> None:
        # watch if a plan has been removed -> layer with xplan type plan has been removed
        try:
            layer = QgsProject.instance().mapLayers()[layername]
            layer_type = layer.customProperties().value(
                f"{PLUGIN_DIR_NAME}/layer_type", None
            )
            if layer_type == "plan":
                del self.plan_layers[layer.id()]
                self.plan_layer_changed.emit()
        except KeyError:
            pass

    @property
    def topology_errors(self) -> Dict[str, List[Any]] | None:
        return self._topology_errors

    @topology_errors.setter
    def topology_errors(self, value: Dict[str, List[Any]] | None):
        self._topology_errors = value

        # send new data to tables
        self.table_bounds.repopulate(self.topology_errors["bounds"])
        self.table_overlaps.repopulate(self.topology_errors["overlaps"])
        self.table_gaps.repopulate(self.topology_errors["gaps"])
        self.table_gaps.gap_features = self.topology_errors["gap_features"]

        # set crs for table views
        self.table_bounds.plan_crs = self._topology_errors["crs"]
        self.table_gaps.plan_crs = self._topology_errors["crs"]
        self.table_overlaps.plan_crs = self._topology_errors["crs"]

        # apply number of violations to tab header
        tabwidget: QTabWidget = self.findChild(QTabWidget, "tabWidget")
        for i in range(tabwidget.count()):
            _tab_text = tabwidget.tabText(i)
            if _tab_text.startswith("Bereichsverletzungen"):
                if (_nr := len(self.topology_errors["bounds"])) > 0:
                    tabwidget.setTabText(i, f"Bereichsverletzungen [{_nr}]")
                else:
                    tabwidget.setTabText(i, "Bereichsverletzungen")
            elif _tab_text.startswith("Überlappungen"):
                if (_nr := len(self.topology_errors["overlaps"])) > 0:
                    tabwidget.setTabText(i, f"Überlappungen [{_nr}]")
                else:
                    tabwidget.setTabText(i, "Überlappungen")
            elif _tab_text.startswith("Lücken"):
                if (_nr := len(self.topology_errors["gaps"])) > 0:
                    tabwidget.setTabText(i, f"Lücken [{_nr}]")
                else:
                    tabwidget.setTabText(i, "Lücken")

    def extractc_polygons_from_collection(self, geom: QgsGeometry) -> QgsGeometry:
        """
        Extracts all polygon and multipolygon parts from a QgsGeometry (including GeometryCollections)
        and returns them as a single MultiPolygon QgsGeometry.
        Returns an empty QgsGeometry if no polygons are found.
        """
        # Convert to geometry collection (list of QgsGeometry)
        parts = geom.asGeometryCollection()
        polygons = []

        for part in parts:
            wkb_type = part.wkbType()
            if QgsWkbTypes.geometryType(wkb_type) == QgsWkbTypes.PolygonGeometry:
                # If it's a multipolygon, decompose into polygons
                if QgsWkbTypes.isMultiType(wkb_type):
                    polygons.extend(
                        [QgsGeometry.fromPolygonXY(p) for p in part.asMultiPolygon()]
                    )
                else:
                    polygons.append(QgsGeometry.fromPolygonXY(part.asPolygon()))

        if polygons:
            # Combine all polygons into a single MultiPolygon geometry
            return QgsGeometry.collectGeometry(polygons)
        else:
            # Return an empty geometry if no polygons found
            return QgsGeometry()

    def get_overlaps_and_gaps(
        self,
        overlap_features: List[Tuple[QgsFeature, QgsVectorLayer]],
        gap_features: List[Tuple[QgsFeature, QgsVectorLayer]],
        bounding_geom: QgsGeometry,
    ) -> Dict[str, List[Any]]:
        """
        :param gap_features:
        :param overlap_features:
        :param bounding_geom:
        :return:
        """

        topology_errors = {"bounds": [], "overlaps": [], "gaps": [], "gap_features": []}

        # Check for overlaps
        for i, (feat1, layer1) in enumerate(overlap_features):
            geom1: QgsGeometry = feat1.geometry()
            for j, (feat2, layer2) in enumerate(overlap_features):
                # Avoid double-checking and self-comparison
                if i >= j:
                    continue
                geom2: QgsGeometry = feat2.geometry()
                # Check if the two geometries overlap
                if geom1.overlaps(geom2):
                    geom_intersect_collection = geom2.intersection(geom1)
                    geom_intersect = self.extractc_polygons_from_collection(
                        geom_intersect_collection
                    )
                    if geom_intersect_collection.area() < 0:
                        # this is an edge case that can occur while editing gaps and overlaps
                        continue
                    topology_errors["overlaps"].append(
                        (geom_intersect, feat1, layer1, feat2, layer2)
                    )

        # Check for gaps
        union_geom = QgsGeometry().fromWkt(wkt="POLYGON EMPTY")
        for feat, _ in gap_features:
            if union_geom.isEmpty():
                union_geom = feat.geometry()
            else:
                union_geom = union_geom.combine(feat.geometry())
        gap_geom = bounding_geom.difference(union_geom)

        if not gap_geom.isEmpty():
            if gap_geom.isMultipart():
                for p in gap_geom.constParts():
                    topology_errors["gaps"].append(QgsGeometry.fromWkt(p.asWkt()))
            else:
                topology_errors["gaps"].append(gap_geom)

        topology_errors["gap_features"] = gap_features
        return topology_errors

    def run_topo_check(self) -> None:
        """
        Funktion die eine Prüfung der Topolgie aller Flächenschlußobjekte durchführt
        :return:
        """

        # TODO: check that section-geometries do not intersect or overlap each other
        # TODO: check that plan-geometry covers all section-geometries

        def _is_vector_poly(layer) -> bool:
            # test if a layer is vector and has polygon-like geometry
            return (
                isinstance(layer, QgsVectorLayer)
                and QgsWkbTypes.geometryType(layer.wkbType())
                == QgsWkbTypes.PolygonGeometry
            )

        def _collect_layers(group) -> List[QgsVectorLayer]:
            # filter all polygon layers from a layergroup recursive
            result = []
            for child in group.findLayers():
                layer = child.layer()
                if _is_vector_poly(layer):
                    result.append(layer)
            return result

        def _search_layer_feature_props(
            layer: QgsVectorLayer, propkey: str
        ) -> List[str | int | None]:
            """
            get a list of values for a given xplan property key of all features of a layer
            :param layer: QgsVectorlayer which features are searched
            :param propkey: name of the xplan property to look for
            :return:
            """
            ret = []
            for feature in layer.getFeatures():
                props = feature.attributeMap().get("properties", {})
                try:
                    ret.append(props[propkey])
                except KeyError:
                    pass
            return ret

        def _layer_features_contained_by_geom(
            layers_list: List[QgsMapLayer] | Set[QgsMapLayer],
            reference_geom: QgsGeometry,
        ) -> Dict[QgsMapLayer, List[Any]]:
            """
            check whether geometries in a list of layers are outside a given reference polygon.

            :param layers_list (list): List of QgsVectorLayer objects to process.
            :param reference_geom (QgsGeometry): Reference polygon as a QgsGeometry.

            :returns dict: Dictionary mapping layer to a list of dicts with feature and xplan IDs.
            """
            # Check reference polygon validity and emptiness
            wkb_type = reference_geom.wkbType()
            if not QgsWkbTypes.geometryType(wkb_type) == QgsWkbTypes.PolygonGeometry:
                raise ValueError("Reference geometry is not of type Polygon")
            if reference_geom is None or reference_geom.isEmpty():
                raise ValueError("Reference polygon geometry is empty or None.")
            if not reference_geom.isGeosValid():
                raise ValueError(
                    "Reference polygon geometry is invalid. Please fix geometry before proceeding."
                )

            # GEOS geometry engine for performance
            geometry_engine = QgsGeometry.createGeometryEngine(
                reference_geom.constGet()
            )
            geometry_engine.prepareGeometry()

            results = []

            for layer in layers_list:
                if layer is None or not layer.isValid():
                    continue

                for feature in layer.getFeatures():
                    feat_geom = feature.geometry()
                    if feat_geom is None or feat_geom.isEmpty():
                        continue
                    if not feat_geom.isGeosValid():
                        continue

                    if geometry_engine.contains(feat_geom.constGet()):
                        continue
                    else:
                        results.append({"feature": feature, "layer": layer})

            return results

        def _get_layer_group_names(layer: QgsMapLayer) -> List[str]:
            """
            Returns the list of group names that the given QgsMapLayer belongs to.
            """
            groups = []
            root = QgsProject.instance().layerTreeRoot()
            layer_node = root.findLayer(layer.id())
            if not layer_node:
                return groups  # Layer not found in tree

            # parse layer tree upwards
            parent = layer_node.parent()
            while parent and isinstance(parent, QgsLayerTreeGroup):
                groups.append(parent.name())
                parent = parent.parent()

            return [g for g in groups if g]  # skip empty root group

        # get information about selected plan from combo box
        cb_item = self.comboBox_plans.currentData()
        if not cb_item:
            return
        selected_xplan_id = cb_item["plan_id"]
        selected_qgis_layerid = cb_item["qgis_layerid"]

        # make sure the selected plan is still present
        plan_layer = QgsProject.instance().mapLayer(selected_qgis_layerid)
        if not plan_layer:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                "Plan-Layer nicht geladen. "
                "Die Topologie-Prüfung kann nicht ausgeführt werden.",
                level=Qgis.MessageLevel.Warning,
            )
            return

        # check that the plan-layer has only one feature
        if plan_layer.featureCount() == 0:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                f"Keine Geometrie im Plan Layer: '{plan_layer.name()}'. "
                f"Die Topologie-Prüfung kann nicht ausgeführt werden.",
                level=Qgis.MessageLevel.Warning,
            )
            return
        elif plan_layer.featureCount() > 1:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                f"Mehr als eine Geometrie im Plan Layer: '{plan_layer.name()}'. "
                f"Die Topologie-Prüfung kann nicht ausgeführt werden.",
                level=Qgis.MessageLevel.Warning,
            )
            return

        # check that there is no "flächenschlussobjekt" in plan layer
        plan_feature: QgsFeature = next(
            plan_layer.getFeatures()
        )  # checked that this layer has only one feature
        plan_feature_properties = plan_feature.attributeMap().get("properties", {})
        plan_geom = plan_feature.geometry()
        if plan_feature_properties.get("flaechenschluss", None) == "true":
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                "Plan-Layer enthält ein Flächenschlußobjekt. "
                "Die Topologie-Prüfung kann nicht ausgeführt werden.",
                level=Qgis.MessageLevel.Warning,
            )
            return

        # get all sections and all applicable layers
        sections: Dict[QgsLayerTreeGroup, List[QgsVectorLayer]] = {}
        rex_sections = re.compile(r"^Bereich \d+$")
        plan_layergroup: QgsLayerTreeGroup = (
            QgsProject.instance().layerTreeRoot().findLayer(plan_layer).parent()
        )
        if not plan_layergroup:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                "Plan-Layer nicht Teil einer Gruppe. "
                "Die Topologie-Prüfung kann nicht ausgeführt werden.",
                level=Qgis.MessageLevel.Warning,
            )
            return

        # make sure all layers are in the same crs
        _crss: List[str] = [plan_layer.crs().toWkt()]
        for layer_node in plan_layergroup.findLayers():
            _wkt = layer_node.layer().crs().toWkt()
            if _wkt != "":  # layers without geometry might not have a crs set
                _crss.append(layer_node.layer().crs().toWkt())

        if len(set(_crss)) != 1:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                "Layer mit unterschiedlichen Koordinatenreferenzsystemen. "
                "Topologie-Prüfung kann nicht durchgeführt werden",
                level=Qgis.MessageLevel.Warning,
            )
            return

        # collect layers of type section
        tree_nodes = [plan_layergroup]
        while tree_nodes:
            node = tree_nodes.pop()  # get next element to check
            for child in node.children():
                if isinstance(child, QgsLayerTreeGroup):
                    if rex_sections.search(child.name()):
                        sections[child] = _collect_layers(child)
                    else:
                        tree_nodes.append(
                            child
                        )  # add back to stack for recursive traversing

        # section layer features must not have attribute flächenschluss == true
        for section_node, layers in sections.items():
            for child in section_node.children():
                if (
                    isinstance(child, QgsLayerTreeLayer)
                    and child.layer()
                    .customProperties()
                    .value(f"{PLUGIN_DIR_NAME}/layer_type")
                    == "section"
                ):
                    if _fs := list(child.layer().getFeatures()):
                        for _f in _fs:
                            _f: QgsFeature
                            props = _f.attributeMap().get("properties", {})
                            if props.get("flaechenschluss", None):
                                self.iface.messageBar().pushMessage(
                                    "XPlan Topologie-Prüfung",
                                    f"Bereichsobjekt in '{section_node.name()}' enthält ein "
                                    f"Flächenschlußobjekt. Die Topologie-Prüfung kann nicht "
                                    f"ausgeführt werden.",
                                    level=Qgis.MessageLevel.Warning,
                                )
                                return

        # test that every geometry object is inside Plan
        _bound_errors = []
        layers_to_check = [
            layer.layer()
            for layer in plan_layergroup.findLayers()
            if layer.layer() != plan_layer and isinstance(layer.layer(), QgsMapLayer)
        ]
        try:
            res = _layer_features_contained_by_geom(layers_to_check, plan_geom)
        except ValueError as e:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung",
                f"interner Fehler: {e}",
                level=Qgis.MessageLevel.Error,
            )
            return

        if res:
            for i, _e in enumerate(res):
                res[i].update({"Typ": "Plan"})
            _bound_errors += res

        # for sections with geometry test that all section features are inside section
        for section_node in sections.keys():
            # divide section-layer and data-layers
            section_layers = [_l.layer() for _l in section_node.findLayers()]
            section_layer = None
            subject_layers = set()

            for layer in section_layers:
                layer_type = layer.customProperties().value(
                    f"{PLUGIN_DIR_NAME}/layer_type"
                )
                if layer_type == "section":
                    if not section_layer:
                        section_layer = layer
                    else:
                        self.iface.messageBar().pushMessage(
                            "XPlan Topologie-Prüfung",
                            f"Der Bereich {section_node.name()} enthält mehr als ein "
                            f"'Bereichsobjekt'-Layer.\n"
                            f"Die Topologie-Prüfung kann nicht durchgeführt werden",
                            level=Qgis.MessageLevel.Error,
                        )
                        return
                elif layer_type:
                    subject_layers.add(layer)

            # check if section-layer has a geom
            if not section_layer or section_layer.featureCount() != 1:
                continue

            # check features inside section-geom
            section_geom = section_layer.getFeature(0).geometry()
            if section_geom.isEmpty():
                continue

            try:
                res = _layer_features_contained_by_geom(subject_layers, section_geom)
            except ValueError as e:
                self.iface.messageBar().pushMessage(
                    "XPlan Topologie-Prüfung",
                    f"interner Fehler: {e}",
                    level=Qgis.MessageLevel.Error,
                )
                return
            if res:
                if res:
                    for i, _e in enumerate(res):
                        res[i].update({"Typ": "Bereich"})
                    _bound_errors += res

        # collect all features that must fulfill the topology rules
        _layers: List[QgsVectorLayer] = [x for k, v in sections.items() for x in v]
        topology_features: List[Tuple[QgsFeature, QgsVectorLayer]] = []
        for _layer in _layers:
            _layer_props = _layer.customProperties()
            _layer_plan_id = _layer_props.value(f"{PLUGIN_DIR_NAME}/plan_id")
            if not _layer_plan_id == selected_xplan_id:
                continue

            for f in _layer.getFeatures():
                f: QgsFeature
                feature_props = f.attributeMap().get("properties", {})
                if feature_props.get("ebene", 0) != 0:
                    continue
                if not feature_props.get("flaechenschluss", None):
                    continue
                topology_features.append((f, _layer))

        # collect all features that must fullfill the overlap rules
        overlap_features = []
        for f, layer in topology_features:
            feature_props = f.attributeMap().get("properties", {})
            if (
                "startBedingung" in feature_props.keys()
                or "endeBedingung" in feature_props.keys()
            ):
                self.iface.messageBar().pushMessage(
                    "XPlan Topologie-Prüfung",
                    "Fläche mit 'startBedingung' oder 'endeBedingung' gefunden. "
                    "Die Topologie-Prüfung kann nicht ausgeführt werden.",
                    level=Qgis.MessageLevel.Warning,
                )
                return
            overlap_features.append((f, layer))

        # collect all features and bounding_geoms for the gap checking
        skip_gap_check = False
        skip_gap_msg = "Überprüfung auf Flächenschluß wird nicht durchgeführt: "

        # skip gap check if type is Landschaftsplan
        if plan_feature.attributeMap().get("featuretype", None) == "LP_Plan":
            skip_gap_check = True
            skip_gap_msg += "Landschaftsplan."
            logger.debug(
                "skipped topology gap-checks because plan is of type LP (Landschaftsplan"
            )

        # skip gap check if plan or section has relation aendertPlan | aendertPlanBereich
        node_stack = [plan_layergroup]
        try:
            while node_stack:
                node = node_stack.pop()
                for child in node.children():
                    if isinstance(child, QgsLayerTreeGroup):
                        node_stack.append(child)
                    elif (
                        isinstance(child, QgsLayerTreeLayer)
                        and isinstance(child.layer(), QgsVectorLayer)
                        and child.layer()
                        .customProperties()
                        .value(f"{PLUGIN_DIR_NAME}/plan_id")
                        == selected_xplan_id
                        and child.layer()
                        .customProperties()
                        .value(f"{PLUGIN_DIR_NAME}/layer_type")
                        in ["plan", "section"]
                    ):
                        _features = list(child.layer().getFeatures())
                        for _f in _features:
                            feature_props = _f.attributeMap().get("properties", {})
                            if (
                                "aendertPlan" in feature_props.keys()
                                or "aendertPlanBereich" in feature_props.keys()
                            ):
                                raise AttributeError
        except AttributeError:
            skip_gap_msg += (
                'Feature mit Attribut "aendertPlan" bzw. "aenderPlanBereich" vorhanden.'
            )
            skip_gap_check = True

        # exclude features which are part of a section with bedeutung 1800|2000
        _flat_section_layers = [layer for k, v in sections.items() for layer in v]
        _flat_section_layers = [
            layer
            for layer in _flat_section_layers
            if layer.customProperties().value(f"{PLUGIN_DIR_NAME}/layer_type")
            == "section"
        ]
        _skip_layers_ = [
            layer
            for layer in _flat_section_layers
            if 1800 in _search_layer_feature_props(layer, "bedeutung")
            or 2000 in _search_layer_feature_props(layer, "bedeutung")
        ]

        gap_features = []
        if not skip_gap_check:
            for f, layer in topology_features:
                if layer in _skip_layers_:
                    continue
                gap_features.append((f, layer))
        else:
            self.iface.messageBar().pushMessage(
                "XPlan Topologie-Prüfung", skip_gap_msg, level=Qgis.MessageLevel.Info
            )

        _topology_errors = self.get_overlaps_and_gaps(
            overlap_features, gap_features, plan_geom
        )
        _topology_errors["bounds"] = _bound_errors
        _topology_errors["crs"] = plan_layer.crs()
        self.topology_errors = _topology_errors

        # message to user when no errors found
        if (
            len(self.topology_errors["bounds"]) == 0
            and len(self.topology_errors["gaps"]) == 0
            and len(self.topology_errors["overlaps"]) == 0
        ):
            msg = QMessageBox()
            msg.setWindowTitle("XPlan Topologie-Prüfung")
            msg.setText("Topologie Prüfung abgeschlossen. Keine Beanstandungen.")
            msg.setIcon(
                QMessageBox.Information
            )  # Shows an info icon, can be replaced with custom icon
            msg.setStandardButtons(QMessageBox.Ok)
            msg.exec_()
        return
