import importlib
import importlib.metadata as md
import importlib.util
import logging
import os
import platform
import socket
import sys
import threading
from pathlib import Path
from urllib.parse import urlparse

import requests
from packaging import requirements
from qgis.core import QgsDataSourceUri, QgsProviderRegistry
from qgis.PyQt.QtCore import QObject, QProcess, QSize, QTimer
from qgis.PyQt.QtGui import QFontDatabase
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QDialogButtonBox,
    QHBoxLayout,
    QLabel,
    QMessageBox,
    QPlainTextEdit,
    QProgressBar,
    QPushButton,
    QVBoxLayout,
    QWidget,
)

from xmas_plugin import settings_manager
from xmas_plugin.settings_manager import get_appschema, load_setting
from xmas_plugin.util.helpers import get_main_window, get_plugin_root
from xmas_plugin.util.metadata import PLUGIN_DIR_NAME
from xmas_plugin.util.requirements import REQUIREMENTS

logger = logging.getLogger(PLUGIN_DIR_NAME)


def python_interpreter_for_qgis() -> str | None:
    """Locate the python interpreter used by QGIS."""
    try:
        # non-Linux
        path_py = os.environ["PYTHONHOME"]
    except Exception:
        # Linux
        path_py = sys.executable

    # convert to Path for easier processing
    path_py = Path(path_py)

    # pre-defined paths for python executable
    dict_pybin = {
        "Darwin": path_py / "bin" / "python3",
        "Windows": path_py / ("../../bin/pythonw.exe"),
        "Linux": path_py,
    }

    # python executable
    path_pybin = dict_pybin[platform.system()]

    if path_pybin.exists():
        return str(path_pybin)
    else:
        return None


def _pip_install_interactive(pkgs: list[str], *, parent=None) -> bool:
    py = python_interpreter_for_qgis()
    if not py:
        QMessageBox.critical(
            parent,
            "Python-Interpreter nicht gefunden",
            "Der Pfad des QGIS-Python-Interpreters konnte nicht ermittelt werden.",
        )
        return False
    args = ["-m", "pip", "install", "--upgrade", "--no-input"]
    args += pkgs

    # Pre-confirm
    ret = QMessageBox.question(
        parent or QApplication.activeWindow(),
        "XMAS-Plugin",
        "Webapp wurde nicht gefunden.\n\nMöchten Sie die Installation jetzt starten?",
        QMessageBox.Yes | QMessageBox.No,
        QMessageBox.Yes,
    )
    if ret != QMessageBox.Yes:
        return False

    # Cancel-able dialog
    dlg = PipInstallDialog(
        "Webapp wird installiert", parent=parent or QApplication.activeWindow()
    )
    dlg.start(py, args)
    dlg.exec_()
    ok = dlg.ok()

    # Post-success message
    if ok:
        QMessageBox.information(
            parent or QApplication.activeWindow(),
            "XMAS-Plugin",
            "Installation erfolgreich. Die Webapp wird jetzt gestartet.\n"
            "Dies kann insbesondere bei der ersten Initialisierung kurz dauern.",
        )
    return ok


def verify_nice_gui_app_installation(parent=None) -> bool:
    missing = []

    for req in REQUIREMENTS:
        requirement = requirements.Requirement(req)
        try:
            installed_version = md.version(requirement.name)
            if installed_version not in requirement.specifier:
                missing.append(str(requirement))
        except Exception:
            missing.append(str(requirement))

    if missing:
        if not _pip_install_interactive(missing, parent=parent):
            logger.error("installation of webapp failed")
            return False

    # ensure the module exists afterwards (fast guard)
    if not importlib.util.find_spec("xmas_app.main"):
        logger.error("xmas_app.main not found after installation")
        return False
    return True


def is_port_in_use(port: int, host: str) -> bool:
    """Return True if (host, port) is already bound by another process."""
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    s.settimeout(1.0)  # just in case
    try:
        # If bind fails, it means the port is in use
        s.bind((host, port))
    except OSError:
        return True
    finally:
        s.close()
    return False


class ServerManager(QObject):
    """Manage the embedded XPlan web-server.

    Responsible for launching and stopping the server process, tracking its
    running state, and emitting `status_changed` whenever the server goes up
    or down.
    """

    def __init__(self, iface) -> None:
        """
        :param python_executable: The path to QGIS's python.exe (or whichever interpreter).
        :param module_name: The python module to run, e.g. 'xplan_gui.main'.
        """
        super().__init__()
        self.iface = iface
        self.server_url = settings_manager.get_normalized_url()
        if not self.server_url:
            parent = get_main_window()
            if parent is not None:
                QMessageBox.warning(
                    parent,
                    "Fehlende URL",
                    "Kein URL zum Webserver in den Plugin-Einstellungen gefunden. Bitte URL eingeben.",
                )
        # if not self.server_url:
        #     return
        self.server_host = urlparse(self.server_url).hostname
        self.server_port = urlparse(self.server_url).port
        logger.info(
            f"server_url: {self.server_url} host: {self.server_host} port: {self.server_port}"
        )
        self.server_status = False
        self.server = None
        self.thread = None
        logger.debug("ServerManager initialised")

    def start_server(self) -> None:
        """
        Spawns the NiceGUI server in a separate thread, forwarding the settings from the QGIS project
        as uvicorn server config.
        """
        if self.thread and self.thread.is_alive():
            logger.info("Webapp server already running.")
            return

        if not self.server_status:
            logger.info("No running server found..")
            if not verify_nice_gui_app_installation(parent=get_main_window()):
                logger.error("Installation of webapp failed")
                raise RuntimeError(
                    "Installation der Webapp abgebrochen oder fehlgeschlagen"
                )

            if is_port_in_use(self.server_port, self.server_host):
                logger.error("Port %s already in use", self.server_port)
                raise RuntimeError(f"Port {self.server_port} ist bereits belegt")

            # Get DB credentials from QGIS's project
            try:
                creds = self._get_qgis_db_creds()
                logger.debug("DB credentials found")
            except ValueError:
                logger.exception("Error reading db credentials")
                raise RuntimeError(
                    "DB-Verbindungsparameter konnten nicht ermittelt werden"
                )

            schema_type, schema_version = get_appschema()

            app_mode = "prod"  # launched from plugin

            env_vars = {
                # pydantic-v2-friendly lower-case
                "appschema": schema_type,
                "appschema_version": schema_version,
                "app_mode": app_mode,
                "app_port": str(self.server_port),
                # uppercase mirrors just in case
                "APPSCHEMA": schema_type,
                "APPSCHEMA_VERSION": schema_version,
                "APP_MODE": app_mode,
                "APP_PORT": str(self.server_port),
                "DB_TYPE": "postgres",
                "XMAS_APP_FROM_PLUGIN": "1",  # switch for xmas-app log file dir
                "XMAS_APP_PLUGIN_DIR": get_plugin_root(),
            }
            os.environ.update(env_vars)
            for k, v in env_vars.items():
                os.putenv(k, str(v))

            if "service" in creds:
                env_vars["PGSERVICE"] = creds["service"]
            else:
                env_vars.update(
                    {
                        "PGUSER": creds["user"],
                        "PGPASSWORD": creds["password"],
                        "PGHOST": creds["host"],
                        "PGPORT": str(creds["port"]),
                        "PGDATABASE": creds["dbname"],
                    }
                )

            # Inject settings into environment
            os.environ.update(env_vars)

            # Import webapp only after setting the environment.
            # Settings() executes already at import and fails without the values.
            try:
                import uvicorn
                from xmas_app.main import create_app
            except ImportError:
                logger.exception("Error on webapp import")
                raise RuntimeError("Webapp konnte nicht initialisiert werden")

            # Start the NiceGUI-app in a thread; webapp will pick up variables from the environment
            config = uvicorn.Config(
                app=create_app,  # factory returns the ASGI app
                factory=True,
                host=self.server_host,
                port=self.server_port,
                reload=False,  # dev reload only in standalone
                log_config=None,  # disable Uvicorn's own logging,
                # which causes errors since it uses sys.stdout which is None in qgis
            )
            logger.debug("Preparing uvicorn Server config")
            server = uvicorn.Server(config)
            self.server = server
            self.thread = threading.Thread(
                target=self._run_server, daemon=True, name="XMAS-Uvicorn-Thread"
            )
            try:
                logger.info("Starting NiceGUI server thread.")
                self.thread.start()
                self.server_status = True
            except Exception as e:
                logger.error(f"Thread couldn't be started: {e}")

    def _run_server(self):
        logger.debug("_run_server entered")
        try:
            self.server.run()
        except Exception as e:
            logger.exception(f"Uvicorn server crashed: {e}")
        finally:
            logger.info("_run_server exited – cleaning handles")
            self.server_status = False
            self.thread = None  # clear stale handle

    def stop_server(self) -> None:
        """
        Stopping webserver process to enable graceful exit.
        """
        if self.server and self.thread and self.thread.is_alive():
            self.server.should_exit = True
            self.thread.join()  # Wait until thread really dies
            # thread cleanup handled in _run_server's finally-block
            return

    def server_running(self) -> bool:
        url = f"{self.server_url}/health_check"
        try:
            response = requests.get(url, timeout=15)
            if response.status_code != 200:
                logger.warning(f"Health-check HTTP {response.status_code} at {url}")
                return False
            return response.status_code == 200
        except Exception:
            logger.exception("Health-check request failed")
            return False

    def _get_qgis_db_creds(self) -> dict:
        """
        Retrieves the QGIS Postgres connection from the plugin settings,
        uses QgsDataSourceUri to handle both service= and host= expansions.

        Returns a dict with either:
          { "service": "some_service_name" }
        or
          {
            "user": "xxx",
            "password": "yyy",
            "host": "zzz",
            "port": 5432,
            "dbname": "abc"
          }
        """
        db_conn_name = load_setting("db_connection", "")
        if not db_conn_name:
            raise ValueError("No 'db_connection' found in QGIS plugin settings.")

        provider_metadata = QgsProviderRegistry.instance().providerMetadata("postgres")
        connections = provider_metadata.connections()
        if db_conn_name not in connections:
            raise ValueError(f"QGIS has no Postgres connection named '{db_conn_name}'")

        conn_info = connections[db_conn_name]
        ds_uri_str = conn_info.uri()

        dsUri = QgsDataSourceUri(ds_uri_str)

        # If dsUri.service() is non-empty, QGIS is storing a named service reference
        if dsUri.service():
            return {"service": dsUri.service()}
        else:
            return {
                "user": dsUri.username(),
                "password": dsUri.password(),
                "host": dsUri.host(),
                "port": dsUri.port(),
                "dbname": dsUri.database(),
            }


class PipInstallDialog(QDialog):
    def __init__(self, title: str, parent=None):
        super().__init__(parent)
        self.setWindowTitle(title)
        self.setModal(True)
        # Console-like proportions
        self.resize(900, 520)  # wider than tall
        self.setMinimumSize(QSize(720, 420))

        self.proc = QProcess(self)
        self.proc.setProcessChannelMode(QProcess.MergedChannels)
        self.proc.readyReadStandardOutput.connect(self._on_ready)
        self.proc.finished.connect(self._on_finished)
        self.proc.errorOccurred.connect(self._on_error)

        self.label = QLabel("Installation läuft… (dies kann einige Minuten dauern)")
        self.out = QPlainTextEdit(readOnly=True)
        # monospaced, no wrapping → proper “console”
        self.out.setLineWrapMode(QPlainTextEdit.NoWrap)
        self.out.setWordWrapMode(0)
        self.out.setUndoRedoEnabled(False)
        fixed = QFontDatabase.systemFont(QFontDatabase.FixedFont)
        self.out.setFont(fixed)

        self.progress = QProgressBar()
        self.progress.setRange(0, 0)  # indeterminate

        self.buttons = QDialogButtonBox()
        self.btn_cancel = self.buttons.addButton(QDialogButtonBox.Cancel)
        self.btn_cancel.clicked.connect(self._on_cancel)

        # small actions row: copy log, clear
        actions = QWidget()
        ah = QHBoxLayout(actions)
        ah.setContentsMargins(0, 0, 0, 0)
        self.btn_copy = QPushButton("Ausgabe kopieren")
        self.btn_clear = QPushButton("Leeren")
        self.btn_copy.clicked.connect(lambda: self._copy_all())
        self.btn_clear.clicked.connect(lambda: self.out.setPlainText(""))
        ah.addWidget(self.btn_copy)
        ah.addWidget(self.btn_clear)
        ah.addStretch()

        lay = QVBoxLayout(self)
        lay.addWidget(self.label)
        lay.addWidget(self.out)
        lay.addWidget(actions)
        lay.addWidget(self.progress)
        lay.addWidget(self.buttons)

        self._result_ok = False

    # API
    def start(self, program: str, arguments: list[str]) -> None:
        self.out.appendPlainText(f"$ {program} " + " ".join(arguments))
        self.proc.start(program, arguments)
        self.show()

    def ok(self) -> bool:
        return self._result_ok

    # slots
    def _on_ready(self):
        data = bytes(self.proc.readAllStandardOutput()).decode(errors="replace")
        if data:
            self.out.appendPlainText(data.rstrip())

    def _on_finished(self, exitCode: int, exitStatus: QProcess.ExitStatus):
        self.progress.setRange(0, 1)
        self.progress.setValue(1)
        self._result_ok = exitStatus == QProcess.NormalExit and exitCode == 0

        if self._result_ok:
            self.label.setText("✅ Installation erfolgreich abgeschlossen.")
            self.progress.setStyleSheet(
                "QProgressBar::chunk { background-color: #4CAF50; }"
            )
            self.out.appendPlainText(
                "\n[Erfolg] Installation erfolgreich abgeschlossen.\n"
            )
        else:
            self.label.setText("❌ Installation fehlgeschlagen oder abgebrochen.")
            self.progress.setStyleSheet(
                "QProgressBar::chunk { background-color: #E53935; }"
            )
            self.out.appendPlainText(f"\n[Fehler] Beendet mit Code {exitCode}.\n")

        self.btn_cancel.setEnabled(True)
        self.btn_cancel.setText("Schließen")

    def _on_error(self, err: QProcess.ProcessError):
        self.out.appendPlainText(f"\n[Fehler] QProcess error: {err}")

    def _on_cancel(self):
        if self.proc.state() != QProcess.NotRunning:
            self.out.appendPlainText("\nAbbruch angefordert…")
            self.btn_cancel.setEnabled(False)
            self.proc.terminate()
            QTimer.singleShot(
                2500,
                lambda: self.proc.kill()
                if self.proc.state() != QProcess.NotRunning
                else None,
            )
        else:
            self.reject()

    def _copy_all(self):
        self.out.selectAll()
        self.out.copy()
