from __future__ import annotations

from enum import IntEnum
from pathlib import Path

from qgis.core import Qgis
from qgis.PyQt import uic
from qgis.PyQt.QtCore import pyqtProperty, pyqtSignal
from qgis.PyQt.QtWidgets import QWidget

from woody.toolbelt import PlgLogger


class CropButtonWidget(QWidget):
    class Mode(IntEnum):
        """Defines which bound is used as default crop value."""

        MIN = 0
        MAX = 1

    valueChanged = pyqtSignal(int)
    activeChanged = pyqtSignal(bool)

    def __init__(self, parent: QWidget = None):
        """Widget to set a crop value along an axis

        It contains a button and a spinbox. The spinbox is only visible
        when the button is pressed. It allows to set a crop value.
        If the widget is not active (button not pressed), the crop value
        is ignored.
        """
        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self.__mode = self.Mode.MIN
        self._log = PlgLogger().log

        self.__init_gui()

    def __init_gui(self) -> None:
        self._button.clicked.connect(lambda: self.setActive(self._button.isChecked()))
        self._spin_box.valueChanged.connect(lambda: self.setValue(self._spin_box.value()))

    def getActive(self) -> bool:
        return self._spin_box.isVisible()

    def setActive(self, active: bool) -> None:
        self._button.blockSignals(True)
        self._button.setChecked(active)
        self._button.blockSignals(False)

        self._spin_box.setVisible(active)
        self.activeChanged.emit(self.getActive())

    def getText(self) -> str:
        return self._button.text()

    def setText(self, text: str) -> None:
        self._button.setText(text)

    def getValue(self) -> int:
        if not self.active:
            if self.__mode == CropButtonWidget.Mode.MIN:
                return self._spin_box.minimum()
            elif self.__mode == CropButtonWidget.Mode.MAX:
                return self._spin_box.maximum()
            else:
                self._log(
                    "Wrong mode value, this should not happen",
                    log_level=Qgis.MessageLevel.Critical,
                    push=False,
                )
                return 0

        return self._spin_box.value()

    def setValue(self, value: int) -> None:
        self._spin_box.blockSignals(True)
        self._spin_box.setValue(value)
        self._spin_box.blockSignals(False)
        self.valueChanged.emit(self.getValue())

    def setRange(self, min: int, max: int) -> None:
        """Sets the interval of the spinbox"""
        self._spin_box.setMinimum(min)
        self._spin_box.setMaximum(max)

    def getMode(self) -> CropButtonWidget.Mode:
        return self.__mode

    def setMode(self, mode: CropButtonWidget.Mode) -> None:
        self.__mode = mode

    active = pyqtProperty(bool, fget=getActive, fset=setActive, notify=activeChanged)
    text = pyqtProperty(str, fget=getText, fset=setText)
    value = pyqtProperty(int, fget=getValue, fset=setValue, notify=valueChanged)
    mode = pyqtProperty(IntEnum, fget=getMode, fset=setMode)
