from pathlib import Path
from typing import Optional

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateReferenceSystem,
    QgsDoubleRange,
    QgsGeometry,
    QgsProject,
)
from qgis.gui import QgisInterface, QgsMapTool
from qgis.PyQt import uic
from qgis.PyQt.QtWidgets import QDialog, QDialogButtonBox, QFileDialog, QWidget

from woody.gui.curve_rubberband import CurveRubberband
from woody.gui.map_tool_draw_curve import MapToolDrawCurve
from woody.gui.map_tool_select_curve import MapToolSelectCurve
from woody.layer.image_layer import ImageLayer
from woody.toolbelt import PlgLogger


class AddImageDialog(QDialog):
    def __init__(self, iface: QgisInterface, parent: QWidget = None):
        """Dialog to select an image and its location (horizontal and vertical)"""

        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self._iface = iface
        self._log = PlgLogger().log

        self._curve_rubberband: Optional[CurveRubberband] = None
        self._xy_curve: Optional[QgsGeometry] = None
        self._current_tool: Optional[QgsMapTool] = None
        self._vertical_range: Optional[QgsDoubleRange] = None
        self._crs: Optional[QgsCoordinateReferenceSystem] = None
        self._image: Optional[Path] = None

        self.__init_gui()
        self.__init_tools()

        self.finished.connect(self.__on_finished)

    def __init_gui(self) -> None:
        # add image
        self._add_image_button.setIcon(QgsApplication.getThemeIcon("mLayoutItemPicture.svg"))
        self._add_image_button.clicked.connect(self.__on_select_image)

        # crop image widget
        self._crop_image_widget.changed.connect(self.__check_crop_values)

        # buttons for tools
        self._draw_reference_curve_button.setIcon(
            QgsApplication.getThemeIcon("mActionCaptureLine.svg")
        )
        self._draw_reference_curve_button.clicked.connect(self.__draw_reference_line)

        self._select_reference_curve_button.setIcon(
            QgsApplication.getThemeIcon("mActionCaptureCurveFromFeature.svg")
        )
        self._select_reference_curve_button.clicked.connect(self.__select_reference_line)

        # ok and cancel buttons
        self._add_layer_button = self.dialog_button_box.button(QDialogButtonBox.StandardButton.Ok)
        self._add_layer_button.setText(self.tr("Add layout"))
        self._add_layer_button.setIcon(QgsApplication.getThemeIcon("mLayoutItem.svg"))
        self._add_layer_button.clicked.connect(self.__create_image_layer)
        self.dialog_button_box.rejected.connect(self.on_cancel)

        # vertical range
        self._vertical_range_upper_spinbox.valueChanged.connect(self.__on_vertical_range_changed)
        self._vertical_range_lower_spinbox.valueChanged.connect(self.__on_vertical_range_changed)
        self.__on_vertical_range_changed()

        self.__update_main_button_enabled()

    def __init_tools(self) -> None:
        # draw curve tool
        self._map_tool_draw_curve = MapToolDrawCurve(self._iface)
        self._map_tool_draw_curve.curveCaptured.connect(self.__on_curve_captured)
        self._map_tool_draw_curve.captureStarted.connect(self.__on_curve_draw_started)
        self._map_tool_draw_curve.captureCanceled.connect(self.__on_curve_draw_canceled)

        # select curve from feature tool
        self._map_tool_select_curve = MapToolSelectCurve(self._iface)
        self._map_tool_select_curve.curveCaptured.connect(self.__on_curve_captured)

    def closeEvent(self, event):
        self.reset_fields()

    def __update_main_button_enabled(self) -> None:
        curve_defined = self._xy_curve is not None
        crs_defined = self._crs is not None
        range_defined = self._vertical_range is not None
        image_path_defined = self._image is not None
        bounds_valid = self._crop_image_widget.isValid()
        self._add_layer_button.setEnabled(
            curve_defined and crs_defined and range_defined and image_path_defined and bounds_valid
        )

    def __check_crop_values(self) -> None:
        crop_rect = self._crop_image_widget.getBounds()
        has_errors = False
        if crop_rect.left() >= crop_rect.right():
            has_errors = True
            self._log(
                message="Crop left value must be less than the right value.",
                log_level=Qgis.MessageLevel.Critical,
                push=True,
                parent_location=self,
            )
        if crop_rect.top() >= crop_rect.bottom():
            has_errors = True
            self._log(
                message="Crop top value must be less than the bottom value.",
                log_level=Qgis.MessageLevel.Critical,
                push=True,
                parent_location=self,
            )
        if not has_errors:
            self._msg_bar.clearWidgets()

    def __draw_reference_line(self) -> None:
        self._iface.mapCanvas().setMapTool(self._map_tool_draw_curve)
        self._current_tool = self._map_tool_draw_curve

    def __select_reference_line(self) -> None:
        self._iface.mapCanvas().setMapTool(self._map_tool_select_curve)
        self._current_tool = self._map_tool_select_curve

    def __on_curve_draw_started(self) -> None:
        # When starting a new draw, hide the previous one
        if self._curve_rubberband:
            self._curve_rubberband.hide()

    def __on_curve_draw_canceled(self) -> None:
        # If the draw is canceled, restore the previous one
        if self._curve_rubberband:
            self._curve_rubberband.show()

        self._draw_reference_curve_button.setChecked(False)

    def __on_curve_captured(self, curve: QgsGeometry) -> None:
        if self._curve_rubberband is None:
            self._curve_rubberband = CurveRubberband(self._iface.mapCanvas())

        self._curve_rubberband.setToGeometry(curve)
        self._curve_rubberband.show()

        self._xy_curve = curve
        self._crs = QgsProject.instance().crs()
        self.__update_main_button_enabled()

    def __on_vertical_range_changed(self) -> None:
        lower_value = self._vertical_range_lower_spinbox.value()
        upper_value = self._vertical_range_upper_spinbox.value()
        if lower_value >= upper_value:
            self._log(
                message="The Upper range value must be greater than the lower one",
                log_level=Qgis.MessageLevel.Critical,
                push=True,
                parent_location=self,
            )
            self._vertical_range = None
        else:
            self._msg_bar.clearWidgets()
            self._vertical_range = QgsDoubleRange(lower_value, upper_value)

        self.__update_main_button_enabled()

    def __on_select_image(self) -> None:
        filters = "Images raster (*.png *.jpg *.jpeg *.tif *.tiff)"
        filename, _ = QFileDialog.getOpenFileName(
            parent=self._iface.mainWindow(),
            caption="Add image to upload",
            filter=filters,
        )

        if filename:
            self._image = Path(filename)
            self._crop_image_widget.setImage(self._image)
            self.__update_main_button_enabled()
            file_stem = self._image.stem

            font = self._name_line_edit.font()
            font.setItalic(True)
            self._name_line_edit.setFont(font)

            self._name_line_edit.setText(file_stem)

    def __create_image_layer(self) -> None:
        layer_name = self._name_line_edit.text()
        image_layer = ImageLayer(layer_name)

        if self._crs is not None:
            image_layer.setCrs(self._crs)

        if self._curve_rubberband:
            self._curve_rubberband.hide()
            image_layer.setGeometry(self._curve_rubberband.asGeometry())

        if self._image:
            image_layer.setImagePath(self._image)
            crop_rect = self._crop_image_widget.getBounds(False)
            if crop_rect.isValid():
                image_layer.setImageBounds(crop_rect)

        if self._vertical_range:
            image_layer.setImageZ(self._vertical_range)

        QgsProject.instance().addMapLayer(image_layer)
        self.accept()
        self.reset_fields()

    def __on_finished(self) -> None:
        if self._curve_rubberband:
            self._curve_rubberband.hide()

        if self._current_tool:
            self._iface.mapCanvas().unsetMapTool(self._current_tool)

    def on_cancel(self):
        self.reject()

        self.reset_fields()

    def reset_fields(self):
        """
        Reset all UI fields and internal variables to their default state.

        - Resets vertical range spinboxes to default values (0.0 to 1.0).
        - Clears references to the selected XY curve, CRS, and vertical range.
        - Hides the curve rubberband if present.
        - Temporarily disables button group exclusivity to allow both
        reference curve buttons to be deselected, then re-enables it.
        - Disables the "add layer" button.
        - Unsets the current map tool if one is active.
        """
        self._vertical_range_lower_spinbox.setValue(0.0)
        self._vertical_range_upper_spinbox.setValue(1.0)

        self._draw_reference_curve_button.clearFocus()
        self._select_reference_curve_button.clearFocus()

        self._xy_curve = None
        self._crs = None
        if getattr(self, "_curve_rubberband", None) is not None:
            self._curve_rubberband.hide()

        # Temporarily disable exclusivity to allow both buttons to be deselected
        self._add_layer_button.setEnabled(False)
        if hasattr(self, "_tools_button_group"):
            self._tools_button_group.setExclusive(False)

        self._draw_reference_curve_button.setChecked(False)
        self._select_reference_curve_button.setChecked(False)

        # Reactivate exclusivity (setExclusive(True))
        if hasattr(self, "_tools_button_group"):
            self._tools_button_group.setExclusive(True)

        if getattr(self, "_current_tool", None) is not None:
            try:
                self.iface.mapCanvas().unsetMapTool(self._current_tool)
            except Exception:
                pass
            self._current_tool = None

        # clear the crop widget
        self._crop_image_widget.reset()
