from qgis.core import Qgis, QgsApplication, QgsException, QgsGeometry
from qgis.gui import QgisInterface, QgsIdentifyMenu, QgsMapMouseEvent, QgsMapTool
from qgis.PyQt.QtCore import QPoint, Qt, pyqtSignal

from woody.toolbelt import PlgLogger


class MapToolSelectCurve(QgsMapTool):
    curveCaptured = pyqtSignal(QgsGeometry)

    def __init__(self, iface: QgisInterface) -> None:
        """This tool allows to capture a curve by selecting a feature
        The curveCaptured signal is sent once the capture is finished.

        This is similar to the tool used by the profile tool:
        QgsMapToolProfileCurveFromFeature

        :param iface: QgsInterface
        """
        super().__init__(iface.mapCanvas())

        self._log = PlgLogger().log

        self.setCursor(QgsApplication.getThemeCursor(QgsApplication.Cursor.CrossHair))

    def flags(self) -> QgsMapTool.Flags:
        return QgsMapTool.Flag.AllowZoomRect | QgsMapTool.Flag.ShowContextMenu

    def canvasPressEvent(self, event: QgsMapMouseEvent) -> None:
        if event.button() != Qt.MouseButton.LeftButton:
            return

        event.ignore()
        results = QgsIdentifyMenu.findFeaturesOnCanvas(
            event, self.canvas(), [Qgis.GeometryType.Line]
        )
        if not results:
            return

        menu = QgsIdentifyMenu(self.canvas())
        menu.setAllowMultipleReturn(False)
        menu.setExecWithSingleResult(False)

        global_pos = self.canvas().mapToGlobal(QPoint(event.pos().x() + 5, event.pos().y() + 5))
        selected_features = menu.exec(results, global_pos)
        if selected_features and selected_features[0].mFeature.hasGeometry():
            first_feature = selected_features[0]
            transform = self.canvas().mapSettings().layerTransform(first_feature.mLayer)
            geometry = first_feature.mFeature.geometry()
            try:
                geometry.transform(transform)
                self.curveCaptured.emit(geometry)
            except QgsException:
                self._log(
                    message="Could not transform geometry from layer CRS",
                    log_level=Qgis.MessageLevel.Critical,
                    push=True,
                )
