#! python3  # noqa: E265

"""Main plugin module."""

# standard
from functools import partial
from pathlib import Path
from typing import Optional

# PyQGIS
from qgis.core import QgsApplication, QgsSettings
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import QCoreApplication, QLocale, QTranslator, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction, QMenu

# project
from woody.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __title__,
    __uri_homepage__,
)
from woody.gui.dlg_add_image import AddImageDialog
from woody.gui.dlg_settings import PlgOptionsFactory
from woody.layer.image_layer import ImageLayer, ImageLayerType
from woody.toolbelt import PlgLogger

# ############################################################################
# ########## Classes ###############
# ##################################


class WoodyPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self._iface = iface
        self._log = PlgLogger().log

        # Register custom layer type to handle images
        QgsApplication.pluginLayerRegistry().addPluginLayerType(ImageLayerType())

        # Customize layer tree view for ImageLayer
        if (layer_tree_view := self._iface.layerTreeView()) is not None:
            layer_tree_view.contextMenuAboutToShow.connect(self.customLayerTreeMenu)

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[0:2]
        locale_path: Path = (
            DIR_PLUGIN_ROOT / "resources" / "i18n" / f"{__title__.lower()}_{self.locale}.qm"
        )
        self._log(message=f"Translation: {self.locale}, {locale_path}", log_level=4)
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

        self._dlg_add_image: Optional[AddImageDialog] = None

    def customLayerTreeMenu(self, menu: QMenu):
        """
        Customize the layer context menu in the main layer tree panel: add specific actions
        for ImageLayer layers.
        """
        if (
            (layer_tree_view := self._iface.layerTreeView()) is None
            or (layer := layer_tree_view.currentLayer()) is None
            or not isinstance(layer, ImageLayer)
            or (action_flip := menu.addAction(self.tr("Image flipped"))) is None
        ):
            return

        action_flip.setCheckable(True)
        action_flip.setChecked(layer.isImageFlipped())
        action_flip.triggered.connect(lambda checked: layer.setImageFlipped(checked))

    def initGui(self):
        """Set up plugin UI elements."""

        # settings page within the QGIS preferences menu
        self._options_factory = PlgOptionsFactory()
        self._iface.registerOptionsWidgetFactory(self._options_factory)

        # -- Actions
        self._action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self._iface.mainWindow(),
        )
        self._action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self._action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self._iface.mainWindow(),
        )
        self._action_settings.triggered.connect(
            lambda: self._iface.showOptionsDialog(currentPage="mOptionsPage{}".format(__title__))
        )

        self._action_add_image = QAction(
            QgsApplication.getThemeIcon("mActionAddLayer.svg"),
            self.tr("Add image"),
            self._iface.mainWindow(),
        )
        self._action_add_image.triggered.connect(self.__add_image)

        # -- Toolbar
        self._iface.addToolBarIcon(self._action_add_image)

        # -- Menu
        self._iface.addPluginToMenu(__title__, self._action_settings)
        self._iface.addPluginToMenu(__title__, self._action_help)
        self._iface.addPluginToMenu(__title__, self._action_add_image)

        # -- Help menu

        # documentation
        self._iface.pluginHelpMenu().addSeparator()
        self._action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self._iface.mainWindow(),
        )
        self._action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self._iface.pluginHelpMenu().addAction(self._action_help_plugin_menu_documentation)

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Remove image layer type
        QgsApplication.pluginLayerRegistry().removePluginLayerType(ImageLayer.LAYER_TYPE)
        # Refresh the canvas to ensure the layers and rubberband are
        # removed if necessary
        if (map_canvas := self._iface.mapCanvas()) is not None:
            map_canvas.refresh()

        # -- Clean up menu
        self._iface.removePluginMenu(__title__, self._action_help)
        self._iface.removePluginMenu(__title__, self._action_settings)
        self._iface.removePluginMenu(__title__, self._action_add_image)

        # -- Clean up toolbar
        self._iface.pluginToolBar().removeAction(self._action_add_image)

        # -- Clean up preferences panel in QGIS settings
        self._iface.unregisterOptionsWidgetFactory(self._options_factory)

        # remove from QGIS help/extensions menu
        if self._action_help_plugin_menu_documentation:
            self._iface.pluginHelpMenu().removeAction(self._action_help_plugin_menu_documentation)

        # Remove layer tree view customization for ImageLayer
        if (layer_tree_view := self._iface.layerTreeView()) is not None:
            layer_tree_view.contextMenuAboutToShow.disconnect(self.customLayerTreeMenu)

        # remove actions
        del self._action_settings
        del self._action_help
        del self._action_add_image

    def __add_image(self):
        if self._dlg_add_image is None:
            self._dlg_add_image = AddImageDialog(self._iface, self._iface.mainWindow())

        self._dlg_add_image.show()
