from pathlib import Path
from typing import Optional

from qgis.PyQt import uic
from qgis.PyQt.QtCore import QPoint, QRect, Qt, pyqtSignal
from qgis.PyQt.QtGui import QImage, QPainter, QPen, QPixmap
from qgis.PyQt.QtWidgets import QWidget

from woody.gui.widget_crop_button import CropButtonWidget


class CropImageWidget(QWidget):
    changed = pyqtSignal()

    IMG_DEFAULT_SIZE = 500

    def __init__(self, parent: QWidget = None):
        """Widget for cropping an image.

        This widget displays a preview of the selected image inside a
        label. It provides four crop buttons, each positioned at one
        corner of the image. These buttons allow cropping the image
        along the corresponding axis: top, bottom, left, or right.

        When a crop button is activated, a line is drawn over the image
        to indicate the cropping boundary.
        """

        # init module and ui
        super().__init__(parent)
        uic.loadUi(Path(__file__).parent / f"{Path(__file__).stem}.ui", self)

        self.__image: Optional[QImage] = None

        self.__init_gui()

    def __init_gui(self) -> None:
        # Ensure to redraw the image every time a crop parameter
        # has changed
        for btn in [
            self._btn_crop_bottom,
            self._btn_crop_left,
            self._btn_crop_right,
            self._btn_crop_top,
        ]:
            btn.valueChanged.connect(lambda: self.__draw_image())
            btn.activeChanged.connect(lambda: self.__draw_image())

    def setImage(self, image_path: Optional[Path]) -> bool:
        self.__image = QImage(str(image_path))
        if not self.__image.isNull():
            self._btn_crop_bottom.setRange(0, self.__image.height())
            self._btn_crop_bottom.setValue(self.__image.height())
            self._btn_crop_bottom.mode = CropButtonWidget.Mode.MAX
            self._btn_crop_top.setRange(0, self.__image.height())
            self._btn_crop_top.setValue(0)

            self._btn_crop_left.setRange(0, self.__image.width())
            self._btn_crop_left.setValue(0)
            self._btn_crop_right.setRange(0, self.__image.width())
            self._btn_crop_right.setValue(self.__image.width())
            self._btn_crop_right.mode = CropButtonWidget.Mode.MAX
        else:
            for btn in [
                self._btn_crop_bottom,
                self._btn_crop_left,
                self._btn_crop_right,
                self._btn_crop_top,
            ]:
                btn.setRange(0, 0)

        self.__draw_image()
        return not self.__image.isNull()

    def __draw_image(self) -> None:
        if not self.__image or self.__image.isNull():
            self._label_image.setText(self.tr("No Image"))
            return
        else:
            img_width = self.__image.width()
            img_height = self.__image.height()

            # Ensure that the image has an alpha channel to be able to draw lines
            pixmap = QPixmap.fromImage(self.__image.convertToFormat(QImage.Format.Format_ARGB32))
            painter = QPainter(pixmap)
            pen = QPen(Qt.GlobalColor.red)

            # the bigger the image is, the less visible, the lines become
            # this adaptive size ensures that the lines are always visible
            pen_width = 2 + max(img_width, img_height) // self.IMG_DEFAULT_SIZE
            pen.setWidth(pen_width)
            pen.setStyle(Qt.PenStyle.CustomDashLine)
            pen.setDashPattern([5, 5])
            painter.setPen(pen)

            # draw the cropping lines
            if self._btn_crop_left.value > 0 and self._btn_crop_left.value < img_width:
                painter.drawLine(
                    self._btn_crop_left.value, 0, self._btn_crop_left.value, img_height
                )
            if self._btn_crop_right.value > 0 and self._btn_crop_right.value < img_width:
                painter.drawLine(
                    self._btn_crop_right.value, 0, self._btn_crop_right.value, img_height
                )
            if self._btn_crop_bottom.value > 0 and self._btn_crop_bottom.value < img_height:
                painter.drawLine(
                    0, self._btn_crop_bottom.value, img_width, self._btn_crop_bottom.value
                )
            if self._btn_crop_top.value > 0 and self._btn_crop_top.value < img_height:
                painter.drawLine(0, self._btn_crop_top.value, img_width, self._btn_crop_top.value)

            painter.end()

            # Resize the pixmap to fit within the label
            # Ensure that max size is IMG_DEFAULT_SIZE
            p_width = min(img_width, self.IMG_DEFAULT_SIZE)
            p_height = min(img_height, self.IMG_DEFAULT_SIZE)
            scaled_pixmap = pixmap.scaled(
                p_width,
                p_height,
                Qt.AspectRatioMode.KeepAspectRatio,
                Qt.TransformationMode.SmoothTransformation,
            )
            self._label_image.setPixmap(scaled_pixmap)
            self._label_image.setFixedSize(scaled_pixmap.width(), scaled_pixmap.height())

        self.changed.emit()

    def isValid(self) -> bool:
        return self.getBounds().isValid()

    def getBounds(self, strict: bool = True) -> QRect:
        """Returns the cropping rectangle of the image.

        strict (bool):
            If True (default), always returns the rectangle defined by the
            current crop button values.
            If False and the widget is inactive, returns an empty rectangle.

        Returns:
        QRect: The rectangle representing the crop bounds.
        """
        if not strict and not self.isActive():
            return QRect()

        top_left = QPoint(self._btn_crop_left.value, self._btn_crop_top.value)
        bottom_right = QPoint(self._btn_crop_right.value, self._btn_crop_bottom.value)
        return QRect(top_left, bottom_right)

    def isActive(self) -> bool:
        return (
            self._btn_crop_left.active
            or self._btn_crop_right.active
            or self._btn_crop_bottom.active
            or self._btn_crop_top.active
        )

    def reset(self) -> None:
        self.setImage(None)
        for btn in [
            self._btn_crop_bottom,
            self._btn_crop_left,
            self._btn_crop_right,
            self._btn_crop_top,
        ]:
            btn.active = False
