from typing import Optional

from qgis.core import QgsAbstractGeometry, QgsGeometry, QgsMapLayer
from qgis.gui import QgisInterface, QgsMapMouseEvent, QgsMapToolCapture
from qgis.PyQt.QtCore import Qt, pyqtSignal
from qgis.PyQt.QtGui import QKeyEvent


class MapToolDrawCurve(QgsMapToolCapture):
    captureStarted = pyqtSignal()
    captureCanceled = pyqtSignal()
    curveCaptured = pyqtSignal(QgsGeometry)

    def __init__(self, iface: QgisInterface) -> None:
        """This tool allows to capture a curve by drawing it on the canvas
        The curveCaptured signal is sent once the capture is finished.

        This is similar to the tool used by the profile tool: QgsMapToolProfileCurve

        :param iface: QgsInterface
        """
        super().__init__(
            iface.mapCanvas(),
            iface.cadDockWidget(),
            QgsMapToolCapture.CaptureMode.CaptureLine,
        )

    def capabilities(self) -> QgsMapToolCapture.Capabilities:
        """QgsMapToolCapture override"""
        return QgsMapToolCapture.Capability.SupportsCurves

    def supportsTechnique(self, technique: QgsMapToolCapture.CaptureTechnique) -> bool:
        """QgsMapToolCapture override"""
        if technique in [
            QgsMapToolCapture.CaptureTechnique.StraightSegments,
            QgsMapToolCapture.CaptureTechnique.CircularString,
            QgsMapToolCapture.CaptureTechnique.Streaming,
        ]:
            return True

        return False

    def layer(self) -> Optional[QgsMapLayer]:
        """QgsMapToolCapture override
        This tool is not associated with any layer
        """
        return None

    def cadCanvasReleaseEvent(self, event: QgsMapMouseEvent) -> None:
        """QgsMapToolCapture override
        Emit a signal when the capture starts
        """
        was_capturing = self.isCapturing()
        super().cadCanvasReleaseEvent(event)
        if not was_capturing and self.isCapturing():
            self.captureStarted.emit()

    def keyPressEvent(self, event: QKeyEvent) -> None:
        """QgsMapToolCapture override
        Restore the previous tool if the capture is cancelled
        """
        if event.key() == Qt.Key.Key_Escape:
            self.clean()
            self.captureCanceled.emit()

    def lineCaptured(self, line: QgsAbstractGeometry) -> None:
        """QgsMapToolCapture override
        Emitted once the line is captured
        """
        self.curveCaptured.emit(QgsGeometry(line.clone()))
