# Treat all annotations as strings to avoid circular imports.
# When PEP 649 is implemented (Python 3.14) we can
# remove this, and the annotations will be deferred
from __future__ import annotations

from typing import TYPE_CHECKING, Optional

from qgis.core import Qgis, QgsLineSymbol
from qgis.gui import QgsFontButton, QgsSymbolButton
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import (
    QCheckBox,
    QDialog,
    QDialogButtonBox,
    QFormLayout,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QSizePolicy,
    QSlider,
    QSpacerItem,
    QWidget,
)

if TYPE_CHECKING:
    from woody.layer.image_layer import ImageLayerProperties


class ImageLayerPropertiesDialog(QDialog):
    """This class defines the layer properties dialog"""

    def __init__(self, properties: ImageLayerProperties, parent: Optional[QWidget] = None) -> None:
        super().__init__(parent)
        self.properties = properties

        layout = QFormLayout(self)
        self.setLayout(layout)

        self.ok_cancel_buttons = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel, self
        )
        self.symbol_button = QgsSymbolButton(self)
        self.symbol_button.setSymbolType(Qgis.SymbolType.Line)
        self.symbol_button.setSymbol(self.properties.map_symbol.clone())
        self.symbol_button.setFixedSize(200, 30)

        self.label_format = QgsFontButton()
        self.label_format.setTextFormat(self.properties.map_label_format)
        self.label_format.setFixedSize(200, 30)

        self.label = QLineEdit(self)
        self.label.setText(self.properties.map_label)
        self.label.setFixedWidth(200)

        self.image_flipped = QCheckBox()
        self.image_flipped.setChecked(self.properties.image_flipped)

        self.image_opacity = QSlider(Qt.Orientation.Horizontal, self)
        self.image_opacity.setMinimum(0)
        self.image_opacity.setMaximum(100)
        self.image_opacity.setValue(self.properties.image_opacity)
        self.image_opacity.setFixedWidth(200)
        self.label_opacity = QLabel(f"{self.properties.image_opacity}%", self)

        self.image_opacity.valueChanged.connect(self.update_image_opacity)
        opacity_layout = QHBoxLayout()
        opacity_layout.addWidget(self.image_opacity)
        opacity_layout.addWidget(self.label_opacity)

        self.ok_cancel_buttons.accepted.connect(self.accept)
        self.ok_cancel_buttons.rejected.connect(self.reject)

        layout.addRow(self.tr("Line symbol"), self.symbol_button)
        layout.addRow(self.tr("Label text format"), self.label_format)
        layout.addRow(self.tr("Label"), self.label)
        layout.addRow(self.tr("Image flipped"), self.image_flipped)
        layout.addRow(self.tr("Image opacity"), opacity_layout)

        layout.addItem(QSpacerItem(0, 0, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Expanding))
        layout.addWidget(self.ok_cancel_buttons)

    def update_image_opacity(self, value):
        return self.label_opacity.setText(f"{value}%")

    def accept(self) -> None:
        new_symbol = self.symbol_button.symbol()
        if new_symbol is not None:
            new_symbol = new_symbol.clone()
            if new_symbol is None or not isinstance(new_symbol, QgsLineSymbol):
                raise RuntimeError("Error when setting new symbol")
            self.properties.map_symbol = new_symbol

        self.properties.map_label = self.label.text()
        self.properties.map_label_format = self.label_format.textFormat()
        self.properties.image_flipped = self.image_flipped.isChecked()
        self.properties.image_opacity = self.image_opacity.value()

        super().accept()
