# -*- coding: utf-8 -*-

"""
/***************************************************************************
 WhiteboxWorkflows
                                 A QGIS plugin
 Provides access to Whitebox Workflows within QGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-03-09
        copyright            : (C) 2023 by John Lindsay, Whitebox Geospatial Inc.
        email                : support@whiteboxgeo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'John Lindsay, Whitebox Geospatial Inc.'
__date__ = '2024-01-01'
__copyright__ = '(C) 2024 by Whitebox Geospatial Inc.'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os, re, platform
from qgis.utils import showPluginHelp
from qgis.PyQt.QtCore import QProcess
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QUrl
from qgis.core import (Qgis,
                       QgsProcessingAlgorithm,
                       QgsProcessingFeedback,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterString,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterMultipleLayers,
                       QgsProcessingParameterField,
                       QgsProcessingParameterFile,
                       QgsProcessing,
                       QgsProcessingUtils,
                       QgsProcessingException,
                       QgsRunProcess,
                       QgsBlockingProcess,
                       QgsMessageLog)
from processing.core.parameters import getParameterFromString
from processing.core.ProcessingConfig import ProcessingConfig
# from contextlib import redirect_stdout
# import whitebox_workflows

pluginPath = os.path.dirname(__file__)

progressRegex = re.compile(r'\d+')

def helpPath():
    return os.path.normpath(os.path.join(os.path.dirname(__file__), 'help'))

def scriptPath():
    return os.path.normpath(os.path.join(os.path.dirname(__file__), 'function_code_files'))

class WhiteboxWorkflowsAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    def __init__(self, descriptionFile):
        super().__init__()

        self.descriptionFile = descriptionFile
        self._name = ''
        self._displayName = ''
        self._group = ''
        self._groupId = ''
        self._shortHelp = ''
        self._helpUrl = ''
        self.params = []

        self.defineCharacteristicsFromFile()

        scriptFile = os.path.join(scriptPath(), f"{self._name}.py")
        self.script = ''
        with open(scriptFile) as f:
            self.script = f.read()

    def defineCharacteristicsFromFile(self):
        with open(self.descriptionFile) as lines:
            line = lines.readline().strip('\n').strip()
            self._name = line

            line = lines.readline().strip('\n').strip()
            self._displayName = line

            line = lines.readline().strip('\n').strip()
            if "None" not in line:
                self._groupId = line
            else:
                self._groupId = None

            line = lines.readline().strip('\n').strip()
            self._helpUrl = line

            line = lines.readline().strip('\n').strip()
            while line != '':
                self.params.append(getParameterFromString(line, 'WbtAlgorithm'))
                line = lines.readline().strip('\n').strip()
    
    def initAlgorithm(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """
        for p in self.params:
            self.addParameter(p, True)

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        if feedback is None:
            feedback = QgsProcessingFeedback()
        
        numOutputs = 0
        params = {}
        for param in self.parameterDefinitions():
            if param.isDestination():
                outputName = os.path.normpath(self.parameterAsOutputLayer(parameters, param.name(), context))
                params[param.name()] = outputName
                numOutputs += 1
                continue

            if param.name() not in parameters: # or parameters[param.name()] is None:
                continue

            if isinstance(param, QgsProcessingParameterRasterLayer):
                layer = self.parameterAsRasterLayer(parameters, param.name(), context)
                if layer is not None:
                    params[param.name()] = str(os.path.normpath(layer.source()))
                    params['wk_dir'] = os.path.dirname(os.path.abspath(params[param.name()]))
                else: # Likely optional parameter
                    params[param.name()] = 'None'
            elif isinstance(param, QgsProcessingParameterFeatureSource):
                layer = self.parameterAsVectorLayer(parameters, param.name(), context)
                if layer is not None:
                    params[param.name()] = str(os.path.normpath(layer.source()))
                    params['wk_dir'] = os.path.dirname(os.path.abspath(params[param.name()]))
                else: # Likely optional parameter
                    params[param.name()] = 'None'
            elif isinstance(param, QgsProcessingParameterFile):
                layer = self.parameterAsString(parameters, param.name(), context)
                if layer is not None:
                    params[param.name()] = layer
                    params['wk_dir'] = os.path.dirname(os.path.abspath(params[param.name()]))
                else: # Likely optional parameter
                    params[param.name()] = 'None'
            elif isinstance(param, QgsProcessingParameterBoolean):
                params[param.name()] = str(self.parameterAsBool(parameters, param.name(), context))
            elif isinstance(param, (QgsProcessingParameterString)):
                if "List" in param.description():
                    s = self.parameterAsString(parameters, param.name(), context)
                    if s is None or len(s) == 0:
                        s = 'None'
                    else:
                        if s[0] == '(':
                            s = s[1:-1]

                        if s[-1] == ')':
                            s = s[0:-2]

                        if s[0] != '[':
                            s = f"[{s}"

                        if s[-1] != ']':
                            s = f"{s}]"

                    params[param.name()] = s
                elif "Tuple" in param.description():
                    s = self.parameterAsString(parameters, param.name(), context)
                    if s is None or len(s) == 0:
                        s = 'None'
                    else:
                        if s[0] == '[':
                            s = s[1:-1]

                        if s[-1] == ']':
                            s = s[0:-2]

                        if s[0] != '(':
                            s = f"({s}"

                        if s[-1] != ')':
                            s = f"{s})"

                    params[param.name()] = s
                else: 
                    s = self.parameterAsString(parameters, param.name(), context).replace("'", "\"")
                    params[param.name()] = f"'{s}'"
            elif isinstance(param, QgsProcessingParameterNumber):
                if param.dataType() == QgsProcessingParameterNumber.Integer:
                    params[param.name()] = str(self.parameterAsInt(parameters, param.name(), context))
                else:
                    params[param.name()] = str(self.parameterAsDouble(parameters, param.name(), context))
            elif isinstance(param, QgsProcessingParameterEnum):
                idx = self.parameterAsEnum(parameters, param.name(), context)
                params[param.name()] = f"{param.options()[idx]}"
            elif isinstance(param, QgsProcessingParameterMultipleLayers):
                layers = self.parameterAsLayerList(parameters, param.name(), context)
                if layers is None or len(layers) == 0:
                    continue
                files = []
                if param.layerType() == QgsProcessing.TypeFile or param.layerType() == QgsProcessing.TypeRaster:
                    files = [os.path.normpath(layer.source()) for layer in layers]
                else:
                    files = []
                    for i, layer in enumerate(layers):
                        tmp  = QgsProcessingUtils.convertToCompatibleFormat(layer, False, 'exported-{}'.format(i), ['shp'], 'shp', context, feedback)
                        files.append(os.path.normpath(tmp))
                
                params[param.name()] = f"{files}"
                # feedback.pushInfo(params[param.name()])
                params[param.name()] = params[param.name()].replace("['", "'").replace("']", "'")
                # feedback.pushInfo(params[param.name()])
                params['wk_dir'] = os.path.dirname(os.path.abspath(files[0]))
            elif isinstance(param, (QgsProcessingParameterField)):
                params[param.name()] = f"'{self.parameterAsString(parameters, param.name(), context)}'"
            else:
                feedback.pushInfo(f"Unrecognized Parameter: {param.name()} {parameters[param.name()]}")
        
        scriptString = self.script
        
        max_threads = ProcessingConfig.getSetting('WBW_MAX_THREADS')
        if max_threads == None or int(max_threads) < 1:
            max_threads = '-1' # All available threads
        scriptString = scriptString.replace("max_threads", max_threads)
        compress_raster = ProcessingConfig.getSetting('WBW_COMPRESS_RASTERS')
        scriptString = scriptString.replace("compress_raster", str(compress_raster))
        scriptString = scriptString.replace("plugin_path", pluginPath)

        # Make sure that the working directory string ends with a path separator
        if 'wk_dir' in params and not params['wk_dir'].endswith(os.sep):
            params['wk_dir'] += os.sep

        for p in params:
            if p != "wk_dir" and 'wk_dir' in params: # replace the long working directory in file names
                params[p] = params[p].replace(params['wk_dir'], "")
            #     params[p] = os.path.basename(params[p])

            scriptString = scriptString.replace(p, params[p]).replace("'None'", "None")
            # Likely optional parameters
            scriptString = scriptString.replace("wbe.read_raster(None)", "None # Optional parameter")
            scriptString = scriptString.replace("wbe.read_vector(None)", "None # Optional parameter")
            scriptString = scriptString.replace("wbe.read_lidar(None)", "None # Optional parameter")
            
            # if "'[" in scriptString and "]'" in scriptString:
            #     scriptString = scriptString.replace("'[", "[").replace("]'", "]")
            # if "'(" in scriptString and ")'" in scriptString:
            #     scriptString = scriptString.replace("'(", "(").replace(")'", ")")

        scriptString = scriptString.replace("''", "'") # This happens with some string parameters

        # This comes after we print the script to feedback so that we don't expose
        # the license ID in the tool dialog output.
        license_id = ProcessingConfig.getSetting('FLOATING_LICENSE_ID')
        if len(license_id) == 0:
            license_id = None
            scriptString = scriptString.replace("wbe.check_in_license('license_id')\n", "").replace("'license_id'", "")
        else:
            scriptString = scriptString.replace("license_id", license_id)

        feedback.pushInfo("WbW Script:")
        if license_id != None:
            feedback.pushInfo(scriptString.replace(license_id, "license_id")) # Don't expose the user's license_id on the output.
        else:
            feedback.pushInfo(scriptString)

        command = "python3"
        if platform.system() == 'Windows':
            command = "py"

        fused_command = f"{command} -c " + scriptString
        QgsMessageLog.logMessage(fused_command, 'Processing', Qgis.Info)

        def onStdOut(ba):
            val = ba.data().decode('utf-8')
            if '%' in val:
                onStdOut.progress = int(progressRegex.search(val).group(0))
                feedback.setProgress(onStdOut.progress)
            else:
                onStdOut.buffer += val

            if onStdOut.buffer.endswith(('\n', '\r')):
                feedback.pushConsoleInfo(onStdOut.buffer.rstrip())
                onStdOut.buffer = ''

        onStdOut.progress = 0
        onStdOut.buffer = ''

        def onStdErr(ba):
            val = ba.data().decode('utf-8')
            onStdErr.buffer += val

            if onStdErr.buffer.endswith(('\n', '\r')):
                feedback.reportError(onStdErr.buffer.rstrip())
                onStdErr.buffer = ''

        onStdErr.buffer = ''

        arguments = ['-c', scriptString]
        proc = QgsBlockingProcess(command, arguments)
        proc.setStdOutHandler(onStdOut)
        proc.setStdErrHandler(onStdErr)

        res = proc.run(feedback)
        if feedback.isCanceled() and res != 0:
            feedback.pushInfo('Process was canceled and did not complete.')
        elif not feedback.isCanceled() and proc.exitStatus() == QProcess.CrashExit:
            raise QgsProcessingException('Process was unexpectedly terminated.')
        elif res == 0:
            feedback.pushInfo('Process completed successfully.')
        elif proc.processError() == QProcess.FailedToStart:
            raise QgsProcessingException('Process "{}" failed to start. Either "{}" is missing, or you may have insufficient permissions to run the program.'.format(command, command))
        else:
            feedback.reportError('Process returned error code {}'.format(res))

        results = {}
        for output in self.outputDefinitions():
            outputName = output.name()
            if outputName in parameters:
                results[outputName] = parameters[outputName]

        return results

    def createInstance(self):
        return self.__class__(self.descriptionFile)

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return self._name
    
    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        # return self.tr(self.name())
        return self._displayName

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    # def group(self):
    #     """
    #     Returns the name of the group this algorithm belongs to. This string
    #     should be localised.
    #     """
    #     return self._group

    def groupId(self):
        return self._groupId

    def shortHelpString(self):
        helpFile = os.path.join(helpPath(), f"{self._name}.html")
        help = ""
        with open(helpFile) as f:
            help = f.read()

        return help

    # def shortHelpString(self):
    #     return self._shortHelp

    def helpUrl(self):
        return self._helpUrl
    
    # def helpUrl(self):
    #     file = os.path.realpath(__file__)
    #     file = os.path.join(
    #         os.path.dirname(file), "help", "index.html"
    #     )
    #     if not os.path.exists(file):
    #         return ""
    #     return QUrl.fromLocalFile(file).toString(QUrl.FullyEncoded)

    def icon(self):
        return QIcon(os.path.join(pluginPath, 'icons', 'WbW.svg'))

    # def tr(self, text):
    #     return QCoreApplication.translate(self.__class__.__name__, text)
    
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)


