# -*- coding: utf-8 -*-
"""
/***************************************************************************
 whisp_analysis
                                 A QGIS plugin
 This plugin analyzes your geometries for deforestation risk through the OpenForis Whisp API

 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
       
        copyright            : (C) 2025 by Jonas Spekker (UN-FAO)
        email                : jospek.dev@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   MIT License                                                           *
 *                                                                         *
 *   Copyright (c) 2025 by Jonas Spekker                                   *
 *                                                                         *
 *   Permission is hereby granted, free of charge, to any person obtaining *
 *   a copy of this software and associated documentation files (the       *
 *   "Software"), to deal in the Software without restriction, including   *
 *   without limitation the rights to use, copy, modify, merge, publish,   *
 *   distribute, sublicense, and/or sell copies of the Software, and to    *
 *   permit persons to whom the Software is furnished to do so, subject to *
 *   the following conditions:                                             *
 *                                                                         *
 *   The above copyright notice and this permission notice shall be        *
 *   included in all copies or substantial portions of the Software.       *
 *                                                                         *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
 *   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
 *   IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY  *
 *   CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,  *
 *   TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE     *
 *   SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                *
 *                                                                         *
 ***************************************************************************/

"""

def is_connected(url="https://whisp.openforis.org", timeout=5):
    # Check if connection to API can be established
    try:
        import requests
        requests.get(url, timeout=timeout)
        return True
    except Exception:
        return False


import subprocess
import sys
import os
import requests
import re
import json
import tempfile
import math

def check_and_install(package):
    #Check if a package is installed, and install it if not
    try:
        __import__(package)
    except ImportError:
        subprocess.check_call([sys.executable, "-m", "pip", "install", package])

# Ensure required packages are installed
check_and_install("requests")

from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QComboBox, QLabel, QRadioButton, QApplication,
    QLineEdit, QPushButton, QDialogButtonBox, QFileDialog, QProgressBar, QScrollArea, QWidget, QCheckBox, QMessageBox
)

from qgis import processing
from qgis.core import QgsProject, QgsMapLayer, QgsVectorFileWriter, QgsVectorLayer, QgsMessageLog, Qgis, QgsField, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsFeature, QgsWkbTypes, QgsDistanceArea, QgsGeometry, QgsPointXY, QgsFields
from qgis.PyQt.QtCore import QThread, pyqtSignal, QObject, QSettings, QTranslator, QCoreApplication, QVariant, Qt

from qgis.PyQt.QtGui import QIcon, QPixmap
from qgis.PyQt.QtWidgets import QAction, QProgressBar, QMessageBox, QInputDialog, QLineEdit, QDialog, QVBoxLayout, QTextBrowser, QDialogButtonBox, QCheckBox, QGridLayout

from .resources import *  # Qt resources
from .whisp_analysis_dialog import whisp_analysisDialog
import os.path
from PyQt5.QtCore import QVariant, QThread, pyqtSignal, QObject, Qt, QTimer



# # ─────────────────────────────────────────────────────────────────────────────
# # OpenForis Whisp API key (required as of June 2025)
# API_KEY = "9232ee12-5f84-4b64-a034-a202a2289c6d"
# # ─────────────────────────────────────────────────────────────────────────────


class InitializationDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowIcon(QIcon(":/plugins/whisp_analysis/icon.png"))
        self.setWindowTitle("Whisp")
        layout = QVBoxLayout()
        self.label = QLabel("Initializing OpenForis Whisp, please wait...")
        layout.addWidget(self.label)
        self.progress_bar = QProgressBar()
        self.progress_bar.setRange(0, 0)  # Indeterminate progress
        layout.addWidget(self.progress_bar)
        self.setLayout(layout)

# Worker to send the test geometry
class InitializationWorker(QObject):
    finished = pyqtSignal(dict)
    progress = pyqtSignal(str)

    def __init__(self, api_key, parent=None):
        super().__init__(parent)
        self.api_key = api_key

    def run(self):
        # Check connectivity.
        if not is_connected("https://whisp.openforis.org", timeout=5):
            error_msg = "Can't connect to Whisp API. \n\nMissing internet connection or port issue. \n\nMake sure to be connected and that port 443 can be accessed."
            self.progress.emit(error_msg)
            self.finished.emit({"error": error_msg})
            return

        # Proceed with test geometry API call.
        test_geojson = {
            "type": "FeatureCollection",
            "features": [
                {"type": "Feature", "geometry": {"type": "Point", "coordinates": [0, 0]}, "properties": {}} #Sends a point at 0°N/S and 0°E/W to retrieve Whisp columns currently available.
            ]
        }
        url = "https://whisp.openforis.org/api/submit/geojson"
        headers = {
            "Content-Type": "application/json",
            "x-api-key": self.api_key
        }
        try:
            response = requests.post(url, json=test_geojson, headers=headers)
            if response.status_code == 200:
                result = response.json()
                self.progress.emit("Initialization complete.")
                self.finished.emit(result)
            else:
                error_msg = f"Error {response.status_code}: {response.text}"
                self.progress.emit(error_msg)
                self.finished.emit({"error": error_msg})
        except Exception as e:
            error_msg = f"Request failed: {str(e)}"
            self.progress.emit(error_msg)
            self.finished.emit({"error": error_msg})





class LayerSelectionDialog(QDialog):
    def __init__(self, columns_mapping, parent=None, default_layer=None):
        super().__init__(parent)
        self.columns_mapping = columns_mapping  # store for later use
        self.setWindowTitle("Whisp")
        self.setWindowIcon(QIcon(":/plugins/whisp_analysis/icon.png"))
        layout = QVBoxLayout(self)

        # Header with Title and Whisp Logo
        headerLayout = QHBoxLayout()

        # Left part: Title and description.
        textLayout = QVBoxLayout()
        titleLabel = QLabel("Whisp")
        font = titleLabel.font()
        font.setPointSize(font.pointSize() * 2)
        font.setBold(True)
        titleLabel.setFont(font)
        descriptionLabel = QLabel("Analyze your geometries for deforestation risk through the OpenForis Whisp API and output them as GeoJSON.")
        descriptionLabel.setWordWrap(True)
        textLayout.addWidget(titleLabel)
        textLayout.addWidget(descriptionLabel)
        headerLayout.addLayout(textLayout)

        headerLayout.addStretch()

        # Right part: the icon.
        logoLabel = QLabel()
        logoPixmap = QPixmap(":/plugins/whisp_analysis/icon.png")
        if not logoPixmap.isNull():
            scaledLogo = logoPixmap.scaled(
                int(logoPixmap.width() * 0.2),
                int(logoPixmap.height() * 0.2),
                Qt.KeepAspectRatio,
                Qt.SmoothTransformation
            )
            logoLabel.setPixmap(scaledLogo)
        headerLayout.addWidget(logoLabel)

        layout.addLayout(headerLayout)
        layout.addSpacing(20)

        # Input Layer Selection
        layout.addWidget(QLabel("Select Input Layer:"))
        inputLayout = QHBoxLayout()
        self.inputCombo = QComboBox()
        layers = [layer for layer in QgsProject.instance().mapLayers().values()
                if layer.type() == QgsMapLayer.VectorLayer]
        for layer in layers:
            self.inputCombo.addItem(layer.name(), layer)
        inputLayout.addWidget(self.inputCombo)

        # Add "Browse..." button for input layers.
        self.browseInputButton = QPushButton("Browse...")
        self.browseInputButton.setFixedWidth(100)
        inputLayout.addWidget(self.browseInputButton)
        layout.addLayout(inputLayout)

        # Use the passed default_layer, or fallback to the active layer from iface.
        if default_layer is None and hasattr(self, "iface"):
            default_layer = self.iface.activeLayer()
        if default_layer is not None:
            for index in range(self.inputCombo.count()):
                if self.inputCombo.itemData(index) == default_layer:
                    self.inputCombo.setCurrentIndex(index)
                    break

        self.inputCombo.currentIndexChanged.connect(self.updateOkButtonState)
        self.browseInputButton.clicked.connect(self.browseInputLayer)

        # Warning label for too many geometries.
        self.inputWarningLabel = QLabel("")
        self.inputWarningLabel.setStyleSheet("font-style: italic; color: red;")
        self.inputWarningLabel.setVisible(False)
        layout.addWidget(self.inputWarningLabel)

        # Output File Selection
        layout.addWidget(QLabel("Output File Name:"))
        fileLayout = QHBoxLayout()
        self.newFileLineEdit = QLineEdit()
        self.newFileBrowseButton = QPushButton("Browse...")
        self.newFileBrowseButton.setFixedWidth(100)
        fileLayout.addWidget(self.newFileLineEdit)
        fileLayout.addWidget(self.newFileBrowseButton)
        layout.addLayout(fileLayout)
        self.newFileBrowseButton.clicked.connect(self.browseNewFile)
        self.newFileLineEdit.textChanged.connect(self.updateOkButtonState)

        # Output Columns Selection
        layout.addWidget(QLabel("Select Output Columns:"))
        self.scroll_area = QScrollArea()
        self.scroll_area.setWidgetResizable(True)
        self.checkbox_widget = QWidget()
        self.checkbox_widget.setObjectName("checkboxWidget")
        self.checkbox_layout = QVBoxLayout(self.checkbox_widget)
        self.checkboxes = {}
        for column in columns_mapping:
            checkbox = QCheckBox(column)
            checkbox.setChecked(True)
            checkbox.stateChanged.connect(self.updateOkButtonState)
            self.checkboxes[column] = checkbox
            self.checkbox_layout.addWidget(checkbox)
        self.scroll_area.setWidget(self.checkbox_widget)
        layout.addWidget(self.scroll_area)

        # Quick Selection Buttons
        btnLayout = QHBoxLayout()
        self.btnDeselectAll = QPushButton("Deselect all")
        self.btnSelectAll = QPushButton("Select all")
        # self.btnSelectEUDR = QPushButton("Reduced Selection")
        btnLayout.addWidget(self.btnDeselectAll)
        btnLayout.addWidget(self.btnSelectAll)
        # btnLayout.addWidget(self.btnSelectEUDR)
        layout.addLayout(btnLayout)
        self.btnDeselectAll.clicked.connect(self.deselectAll)
        self.btnSelectAll.clicked.connect(self.selectAll)
        # self.btnSelectEUDR.clicked.connect(self.selectEUDRRelevant)

        # Dialog Buttons (OK/Cancel)
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        layout.addWidget(self.buttonBox)

        self.updateOkButtonState()
        self.applyCustomStyleSheet()



    def accept(self):

        QgsMessageLog.logMessage(">>> ENTERED LayerSelectionDialog.accept() <<<", "WhispAnalysis", Qgis.Info)

        # First, check internet connectivity.
        if not is_connected("https://whisp.openforis.org", timeout=5):
            msgBox = QMessageBox(self)
            msgBox.setIcon(QMessageBox.Critical)
            msgBox.setWindowTitle("Connectivity Error")
            msgBox.setText("Can't connect to Whisp API. \n\nMissing internet connection or port issue. \n\nMake sure to be connected and that port 443 can be accessed.")
            msgBox.exec_()
            return
        
        # 0) Abort on mixed GeoJSON geometry types right away
        input_layer = self.inputCombo.currentData()
        if input_layer and input_layer.providerType() == "ogr":
            uri = input_layer.dataProvider().dataSourceUri()
            path = uri.split("|")[0]
            try:
                import json
                with open(path, "r", encoding="utf-8") as f:
                    gj = json.load(f)
                geom_types = {
                    feat.get("geometry", {}).get("type")
                    for feat in gj.get("features", [])
                }
            except Exception as e:
                QgsMessageLog.logMessage(
                    f"Could not inspect GeoJSON: {e}",
                    "WhispAnalysis", Qgis.Warning
                )
            else:
                pts  = {"Point", "MultiPoint"}
                polys = {"Polygon", "MultiPolygon"}
                if geom_types & pts and geom_types & polys:
                    dlg = QMessageBox(self)
                    dlg.setIcon(QMessageBox.Warning)
                    dlg.setWindowTitle("Unsupported Mixed Geometry")
                    dlg.setText(
                        "This GeoJSON contains both Point and Polygon features.\n\n"
                        "QGIS can not process such files.\n\n"
                        "Please add the file directly to QGIS to split into separate layers."
                    )
                    dlg.setStandardButtons(QMessageBox.Ok)
                    dlg.exec_()
                    return

        # Next, check CRS of the selected input layer.
        input_layer = self.inputCombo.currentData()
        if input_layer:
            source_crs = input_layer.crs()
            # Construct a detailed CRS string, e.g. "EPSG:3857 - Google Maps CRS"
            source_crs_str = f"{source_crs.authid()} - {source_crs.description()}"
            if source_crs.authid() != "EPSG:4326":
                msgBox = QMessageBox(self)
                msgBox.setIcon(QMessageBox.Warning)
                msgBox.setWindowTitle("CRS Conversion Warning")
                # Build HTML message with extra spacing and bold formatting for the CRS information.
                msg = (
                    f"<p>Your input file's CRS is <b>{source_crs_str}</b>.</p>"
                    f"<p>The output will be in CRS <b>EPSG:4326 - WGS84</b>.</p>"
                    f"<p>Do you wish to proceed?</p>"
                )
                msgBox.setTextFormat(Qt.RichText)
                msgBox.setText(msg)
                proceed_button = msgBox.addButton("Proceed", QMessageBox.AcceptRole)
                cancel_button = msgBox.addButton("Cancel", QMessageBox.RejectRole)
                msgBox.setDefaultButton(cancel_button)
                msgBox.exec_()
                if msgBox.clickedButton() == cancel_button:
                    # User cancelled the CRS warning; keep the dialog open.
                    return
                
        
        # Perform a re‑whisp check.
        if not getattr(self, 'allow_rewhisp', False):
            analysis_fields = set(self.columns_mapping.keys())
            input_layer = self.inputCombo.currentData()
            input_field_names = set(field.name() for field in input_layer.fields())
            missing_fields = analysis_fields.difference(input_field_names)
            # If five or fewer expected analysis fields are missing, assume it’s been whisped.
            if len(missing_fields) <= 5:
                msgBox = QMessageBox(self)
                msgBox.setIcon(QMessageBox.Warning)
                msgBox.setWindowTitle("Re‑whisp?")
                msgBox.setText("It seems you have previously whisped this input layer. Do you want to whisp it again?")
                rewhisp_button = msgBox.addButton("Re‑whisp", QMessageBox.AcceptRole)
                cancel_button = msgBox.addButton("Cancel", QMessageBox.RejectRole)
                msgBox.setDefaultButton(cancel_button)
                msgBox.exec_()
                if msgBox.clickedButton() == cancel_button:
                    # User chose to cancel; do not close the dialog.
                    return
                else:
                    # User chose to re‑whisp.
                    self.allow_rewhisp = True
                    try:
                        self.buttonBox.accepted.disconnect(self.accept)
                    except Exception:
                        pass
                    super().accept()
                    return
                
        # 4) Multi-geometry warning
        # -- grab the same layer from the combo each time --
        input_layer = self.inputCombo.currentData()
        if input_layer:
            QgsMessageLog.logMessage(
                "Checking for MultiPoint/MultiPolygon in accept()…",
                "WhispAnalysis", Qgis.Info
            )
            for feat in input_layer.getFeatures():
                wkb = feat.geometry().wkbType()
                QgsMessageLog.logMessage(
                    f"  Feature WKB → {QgsWkbTypes.displayString(wkb)}",
                    "WhispAnalysis", Qgis.Info
                )
                if (QgsWkbTypes.isMultiType(wkb)
                    and QgsWkbTypes.geometryType(wkb) in (
                        QgsWkbTypes.PointGeometry,
                        QgsWkbTypes.PolygonGeometry
                    )
                ):
                    reply = QMessageBox(self)
                    reply.setIcon(QMessageBox.Warning)
                    reply.setWindowTitle("Warning")
                    reply.setText(
                        "Whisp API does not support MultiPoint / MultiPolygon features.\n"
                        "Multi-features will be split into single features for the analysis."
                    )
                    reply.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)
                    reply.button(QMessageBox.Ok).setText("OK")
                    reply.button(QMessageBox.Cancel).setText("Cancel")
                    reply.setDefaultButton(reply.button(QMessageBox.Cancel))
                    if reply.exec_() != QMessageBox.Ok:
                        return
                    break

        # If all checks pass, accept normally.
        super().accept()



    def applyCustomStyleSheet(self):
        """
        Applies a style sheet that:
        - Forces the scroll area viewport to have a white background
        - Adds thinner horizontal lines (0.5px) under each checkbox that span the full width
        - Styles the scrollbar for a 3D-like look
        """

        # Make sure the scroll area viewport is white.
        self.scroll_area.viewport().setStyleSheet("background-color: white;")

        # Give the checkbox layout zero margins and spacing so lines can stretch edge to edge.
        self.checkbox_layout.setContentsMargins(0, 0, 0, 0)
        self.checkbox_layout.setSpacing(0)

        # Style the scrollbar with a simple 3D-like gradient.
        self.scroll_area.setStyleSheet("""
            QScrollBar:vertical {
                border: 1px solid #999999;
                background: #F0F0F0;
                width: 12px;
                margin: 0px 0px 0px 0px;
            }
            QScrollBar::handle:vertical {
                background: qlineargradient(
                    x1:0, y1:0, x2:0, y2:1,
                    stop:0 #E4E4E4,
                    stop:0.5 #D1D1D1,
                    stop:0.5 #C7C7C7,
                    stop:1 #BFBFBF
                );
                min-height: 20px;
                border: 1px solid #AAA;
                border-radius: 4px;
            }
            QScrollBar::add-line:vertical {
                height: 14px;
                subcontrol-position: bottom;
                subcontrol-origin: margin;
                border: 1px solid #999;
                background: #C2CCDF;
            }
            QScrollBar::sub-line:vertical {
                height: 14px;
                subcontrol-position: top;
                subcontrol-origin: margin;
                border: 1px solid #999;
                background: #C2CCDF;
            }
            QScrollBar::add-page:vertical,
            QScrollBar::sub-page:vertical {
                background: none;
            }
        """)

        # Thinner lines (0.5px) between checkboxes that span from left to right.
        # Removing margins ensures the line extends fully across the layout width.
        self.checkbox_widget.setStyleSheet("""
            QCheckBox {
                border-bottom: 0.1px solid #CCC;
                margin: 0;
                padding: 4px 0;
            }
        """)


    def updateOkButtonState(self):
        ok_button = self.buttonBox.button(QDialogButtonBox.Ok)
        file_ok = bool(self.newFileLineEdit.text().strip())
        # Check if at least one checkbox is selected.
        column_ok = any(cb.isChecked() for cb in self.checkboxes.values())
        ok_button.setEnabled(file_ok and column_ok)
    
    def browseInputLayer(self):
        file_path, _ = QFileDialog.getOpenFileName(
            self,
            "Select Geometry File",
            "",
            "Vector Files (*.shp *.geojson *.gpkg);;All Files (*)"
        )
        if file_path:
            new_layer = QgsVectorLayer(file_path, os.path.basename(file_path), "ogr")
            if not new_layer.isValid():
                QMessageBox.critical(self, "Error", "The selected file could not be loaded as a vector layer.")
            else:
                # Keep a reference so it isn’t garbage-collected.
                if not hasattr(self, 'extraInputLayers'):
                    self.extraInputLayers = []
                self.extraInputLayers.append(new_layer)
                
                self.inputCombo.addItem(new_layer.name(), new_layer)
                self.inputCombo.setCurrentIndex(self.inputCombo.count() - 1)





    def browseNewFile(self):
        # Start at the directory of the currently selected input layer (if any).
        selected_layer = self.inputCombo.currentData()
        default_dir = ""
        src = ""  # initialize src so it's always defined
        if selected_layer is not None:
            src = selected_layer.source()
            if src:
                default_dir = os.path.dirname(src)
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Select Output File",
            default_dir,
            "GeoJSON Files (*.geojson)"
        )
        if file_path:
            self.newFileLineEdit.setText(file_path)



    def deselectAll(self):
        for checkbox in self.checkboxes.values():
            checkbox.setChecked(False)

    def selectAll(self):
        for checkbox in self.checkboxes.values():
            checkbox.setChecked(True)

    def selectEUDRRelevant(self):
        # ADJUST THESE NAMES IF NEEDED IN PLUGIN-UPDATE!
        for col, checkbox in self.checkboxes.items():
            if col in {"Area", "ProducerCountry", "EUDR_risk"}:
                checkbox.setChecked(True)
            else:
                checkbox.setChecked(False)

    def getSelections(self):
        # Return the chosen input layer, output file path, and list of selected columns
        input_layer = self.inputCombo.currentData()
        output_file = self.newFileLineEdit.text()
        selected_columns = [col for col, cb in self.checkboxes.items() if cb.isChecked()]
        return input_layer, output_file, selected_columns







class WhispWorker(QObject):
    """
    Handles the main Whisp API request in a background thread.
    """
    finished = pyqtSignal(dict)   # Signal to send back the API response
    progress = pyqtSignal(str)    # Signal to update progress messages

    def __init__(self, geojson, api_key, parent=None):
        super().__init__(parent)
        self.geojson = geojson
        self.api_key = api_key

    def run(self):
        # 1) Connectivity check
        if not is_connected("https://whisp.openforis.org", timeout=5):
            error_msg = (
                "Can't connect to Whisp API.\n\n"
                "Missing internet connection or port issue.\n\n"
                "Make sure to be connected and that port 443 can be accessed."
            )
            self.progress.emit(error_msg)
            self.finished.emit({"error": error_msg})
            return

        # 2) Send the request
        self.progress.emit("Sending request to Whisp API...")
        url = "https://whisp.openforis.org/api/submit/geojson"
        headers = {
            "Content-Type": "application/json",
            "x-api-key": self.api_key
        }

        try:
            response = requests.post(url, json=self.geojson, headers=headers)
            if response.status_code == 200:
                result = response.json()
                self.progress.emit("Whisp Analysis completed successfully.")
                self.finished.emit(result)
            else:
                error_msg = f"Error {response.status_code}: {response.text}"
                self.progress.emit(error_msg)
                self.finished.emit({"error": error_msg})

        except Exception as e:
            error_msg = f"Request failed: {str(e)}"
            self.progress.emit(error_msg)
            self.finished.emit({"error": error_msg})






class whisp_analysis:
    #Main plugin implementation code

    def __init__(self, iface):
        #Constructor
        self.iface = iface

        # —————— API key setup ——————
        self.settings = QSettings()
        self.api_key = self.settings.value("WhispAnalysis/api_key", "")

        if not self.api_key:
            # Ask user for their key
            dlg = QInputDialog(self.iface.mainWindow())
            dlg.setWindowTitle("OpenForis Whisp API Key")
            # Multi-line label with explanatory text
            dlg.setLabelText(
                "Please enter your OpenForis Whisp API key:\n\n"
                "If you do not have an API key, please register for OpenForis Whisp\n"
                "(https://whisp.openforis.org/) and create one for your account."
            )
            dlg.setTextValue("")              # start with empty field
            dlg.setOkButtonText("Submit")
            dlg.setCancelButtonText("Cancel")
            dlg.setOption(QInputDialog.UsePlainTextEditForTextInput, True)
            # Optionally resize so the text wraps nicely
            dlg.resize(400, 200)

            if dlg.exec_() == QDialog.Accepted:
                key = dlg.textValue()
                ok = True
            else:
                key = ""
                ok = False

            if not ok or not key:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "API Key Required",
                    "An API key is required to use the Whisp plugin."
                )
                # Abort loading the plugin
                raise Exception("API key not provided")
            # Save for next time
            self.settings.setValue("WhispAnalysis/api_key", key)
            self.api_key = key


            

        
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr(u"&Whisp Analysis")
        self.first_start = None
        self.whisp_columns_file = None  # Store path to temp file
        self.whisp_columns_mapping = {}

    def tr(self, message):
        #Translate a string
        return QCoreApplication.translate("whisp_analysis", message)

    def initGui(self):
        
        # Add toolbar / menu actions
        icon_path = ":/plugins/whisp_analysis/icon.png"
        self.add_action(
            icon_path=icon_path,
            text=self.tr("Start OpenForis Whisp"),
            callback=self.on_submit_geojson,
            status_tip=self.tr("Whisping..."),
            add_to_toolbar=True,
            add_to_menu=True,
            parent=self.iface.mainWindow()
        )
        self.first_start = True

    def add_action(self, icon_path, text, callback, status_tip=None, add_to_toolbar=True, add_to_menu=True, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setStatusTip(status_tip or "")
        self.actions.append(action)
        if add_to_toolbar:
            self.iface.addToolBarIcon(action)
        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)
        return action

    def initialize_whisp_columns(self):
        init_dialog = InitializationDialog(self.iface.mainWindow())
        # Set up progress bar as percentage bar.
        init_dialog.progress_bar.setRange(0, 100)
        init_dialog.progress_bar.setValue(0)

        # Create timer to increment progress bar value.
        timer = QTimer(init_dialog)
        timer.setInterval(300)  # 100 ms intervals -> 100 steps for 10 seconds.
        timer.timeout.connect(lambda: init_dialog.progress_bar.setValue(
            min(init_dialog.progress_bar.value() + 1, 100)))
        timer.start()

        thread = QThread()
        worker = InitializationWorker(self.api_key)
        worker.moveToThread(thread)

        def on_worker_finished(result):
            timer.stop()  # Stop progress timer.
            init_dialog.progress_bar.setValue(100)
            self.on_initialization_finished(result, init_dialog)

        worker.finished.connect(on_worker_finished)
        worker.progress.connect(lambda msg: init_dialog.label.setText(msg))
        worker.finished.connect(thread.quit)
        worker.finished.connect(worker.deleteLater)
        thread.finished.connect(thread.deleteLater)
        thread.started.connect(worker.run)

        self.init_thread = thread  # Keep a reference.
        thread.start()
        init_dialog.exec_()


    
    def on_initialization_finished(self, result, init_dialog):
        """
        Callback once the test-point probe returns.
        If we see an auth error, ask for a new API key and retry once.
        Otherwise build whisp_columns_mapping as before.
        """
        init_dialog.progress_bar.setValue(100)

        # 1) Handle any API-returned errors
        if "error" in result:
            err = result["error"]
            if "Invalid or expired API key" in err:
                # 1) Ensure they've re-acknowledged the ToS/GEE
                if not self.show_acknowledgement_dialog():
                    # they cancelled or didn’t check both boxes: abort
                    init_dialog.close()
                    return  
                # Create a custom input dialog with extra explanation
                dlg = QDialog(self.iface.mainWindow())
                dlg.setWindowTitle("API Key")

                layout = QVBoxLayout(dlg)

                # 1) Prompt label
                prompt_label = QLabel(
                    "Your Whisp API key is inexistent, invalid, or expired.\n\n"
                    "Please enter a valid API key:"
                )
                prompt_label.setWordWrap(True)
                layout.addWidget(prompt_label)

                # 2) Single-line input
                line_edit = QLineEdit(dlg)
                layout.addWidget(line_edit)

                # 3) Explanation below the input, with clickable link
                explain_label = QLabel(
                    'If you do not have an API key, please register at '
                    '<a href="https://whisp.openforis.org/">OpenForis Whisp</a> '
                    'and create one for your account.'
                )
                explain_label.setTextFormat(Qt.RichText)
                explain_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
                explain_label.setOpenExternalLinks(True)
                explain_label.setWordWrap(True)
                layout.addWidget(explain_label)

                # 4) OK / Cancel buttons
                buttons = QDialogButtonBox(
                    QDialogButtonBox.Ok | QDialogButtonBox.Cancel,
                    parent=dlg
                )
                buttons.accepted.connect(dlg.accept)
                buttons.rejected.connect(dlg.reject)
                layout.addWidget(buttons)

                # 5) Execute and fetch result
                if dlg.exec_() == QDialog.Accepted:
                    new_key = line_edit.text().strip()
                    ok = bool(new_key)
                else:
                    new_key = ""
                    ok = False

                if ok:
                    # Save and retry initialization
                    QSettings().setValue("WhispAnalysis/api_key", new_key)
                    self.api_key = new_key
                    init_dialog.close()
                    self.initialize_whisp_columns()
                    return
                else:
                    QMessageBox.critical(
                        self.iface.mainWindow(),
                        "API Key Required",
                        "A valid API key is required to use the Whisp plugin."
                    )
                    init_dialog.close()
                    return
            else:
                # Some other initialization error
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "Initialization Error",
                    f"Unexpected response from Whisp API:\n{err}"
                )
                init_dialog.close()
                return

        # 2) Normal mapping of columns from the returned GeoJSON
        try:
            features = result.get("data", {}).get("features") or result.get("features")
            first_props = features[0]["properties"]
        except Exception:
            QMessageBox.critical(
                self.iface.mainWindow(),
                "Initialization Error",
                "Unexpected response from Whisp API:\n" + json.dumps(result)
            )
            init_dialog.close()
            return

        mapping = {}
        for key, value in first_props.items():
            if key == "plotId":
                mapping[key] = "int"
            elif isinstance(value, (int, float)):
                mapping[key] = "double"
            else:
                mapping[key] = "string"

        self.whisp_columns_mapping = mapping
        QgsMessageLog.logMessage(
            f"Whisp columns mapping: {self.whisp_columns_mapping}",
            "WhispAnalysis", Qgis.Info
        )
        init_dialog.accept()




    def unload(self):
        #Remove the plugin menu item and toolbar icon
        for action in self.actions:
            self.iface.removeToolBarIcon(action)
            self.iface.removePluginMenu(self.menu, action)

    def export_layer_with_properties(self, layer):
        """
        Exports the given layer to a temporary GeoJSON file,
        flattening multipart geometries into singletons and
        tagging each feature with unique plotId and extractId.
        Returns the path to the temp GeoJSON.
        """
        features = []

        for feature in layer.getFeatures():
            # 1) Build original properties dict (ensure plotId key)
            if layer.fields().count() > 0:
                orig_props = {f.name(): feature[f.name()] for f in layer.fields()}
            else:
                orig_props = {}
            if "plotId" not in orig_props:
                orig_props["plotId"] = None
            orig_id = orig_props["plotId"]

            geom = feature.geometry()

            def make_feature(subgeom, idx):
                # Convert sub-geometry to GeoJSON geometry dict
                geom_json = subgeom.asJson()
                try:
                    geom_obj = json.loads(geom_json)
                except Exception as e:
                    QgsMessageLog.logMessage(
                        f"Error parsing sub-geometry JSON for plot {orig_id}_{idx}: {e}",
                        "WhispAnalysis",
                        Qgis.Warning
                    )
                    return None

                # Copy & update properties
                props = orig_props.copy()
                props["plotId"] = f"{orig_id}_{idx}"
                geom_type = QgsWkbTypes.displayString(subgeom.wkbType()).replace("Multi", "")
                props["extractId"] = f"{geom_type}_{orig_id}_extract_{idx}"

                return {
                    "type": "Feature",
                    "geometry": geom_obj,
                    "properties": props
                }

            # 2) Flatten multipart vs singlepart
            if geom.isMultipart():
                flat_type = QgsWkbTypes.flatType(geom.wkbType())
                if flat_type == QgsWkbTypes.PolygonGeometry:
                    parts = geom.asMultiPolygon()
                    maker = lambda poly: QgsGeometry.fromPolygonXY(poly)
                elif flat_type == QgsWkbTypes.PointGeometry:
                    parts = geom.asMultiPoint()
                    maker = lambda pt: QgsGeometry.fromPointXY(pt)
                else:
                    parts = geom.asGeometryCollection()
                    maker = lambda g: g

                for i, part in enumerate(parts, start=1):
                    subgeom = maker(part)
                    feat = make_feature(subgeom, i)
                    if feat:
                        features.append(feat)
            else:
                single = make_feature(geom, 1)
                if single:
                    features.append(single)

        # 3) Dump out FeatureCollection
        fc = {
            "type": "FeatureCollection",
            "features": features
        }
        temp_file = tempfile.NamedTemporaryFile(delete=False, suffix=".geojson")
        try:
            with open(temp_file.name, "w", encoding="utf-8") as f:
                json.dump(fc, f)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Failed writing flattened GeoJSON: {e}",
                "WhispAnalysis",
                Qgis.Critical
            )
            raise

        QgsMessageLog.logMessage(
            f"Exported flattened GeoJSON: {temp_file.name}",
            "WhispAnalysis",
            Qgis.Info
        )
        return temp_file.name
    
    # def show_terms_of_service(self):
    #     """
    #     Fetch the full Terms of Service page, then display only the
    #     section from the “Terms of Service for Whisp API” heading onward.
    #     """
    #     tos_url = "https://openforis.org/whisp-terms-of-service/"
    #     try:
    #         resp = requests.get(tos_url, timeout=5)
    #         resp.raise_for_status()
    #         page = resp.text
    #     except Exception as e:
    #         # Fallback if fetch fails
    #         page = (
    #             "<h2>Terms of Service for Whisp API</h2>"
    #             "<p>Unable to fetch the live Terms of Service.</p>"
    #             f'<p>Please review them online: <a href="{tos_url}">{tos_url}</a></p>'
    #         )

    #     # Find the heading and take everything from there to the end
    #     marker = "Terms of Service for Whisp API"
    #     idx = page.find(marker)
    #     if idx != -1:
    #         # Back up to the start of the <h1> tag
    #         h1_start = page.rfind("<h1", 0, idx)
    #         fragment = page[h1_start:]
    #     else:
    #         # If no marker found, just show the whole fetched page
    #         fragment = page

    #     # Wrap in minimal HTML so QTextBrowser renders properly
    #     html = f"<html><body>{fragment}</body></html>"

    #     # Build and show the dialog
    #     dlg = QDialog(self.iface.mainWindow())
    #     dlg.setWindowTitle("OpenForis Whisp Terms of Service")
    #     dlg.resize(600, 500)

    #     layout = QVBoxLayout(dlg)
    #     browser = QTextBrowser(dlg)
    #     browser.setOpenExternalLinks(True)
    #     browser.setHtml(html)
    #     layout.addWidget(browser)

    #     buttons = QDialogButtonBox(QDialogButtonBox.Close, parent=dlg)
    #     buttons.rejected.connect(dlg.reject)
    #     layout.addWidget(buttons)

    #     dlg.exec_()


    def show_acknowledgement_dialog(self):
        dlg = QDialog(self.iface.mainWindow())
        dlg.setWindowTitle("Terms of Service")
        dlg.setModal(True)
        dlg.resize(400, 120)

        main_layout = QVBoxLayout(dlg)
        main_layout.setContentsMargins(8, 8, 8, 8)
        main_layout.setSpacing(4)

        # Grid for two rows: label in col 0, checkbox in col 1
        grid = QGridLayout()
        grid.setColumnStretch(0, 1)
        grid.setColumnStretch(1, 0)
        grid.setHorizontalSpacing(6)
        grid.setVerticalSpacing(4)

        # First row
        lbl1 = QLabel(
            'I have read and understood the '
            '<a href="https://openforis.org/whisp-terms-of-service/">'
            'Terms of Service</a>.'
        )
        lbl1.setTextFormat(Qt.RichText)
        lbl1.setTextInteractionFlags(Qt.TextBrowserInteraction)
        lbl1.setOpenExternalLinks(True)
        lbl1.setWordWrap(True)
        cb1 = QCheckBox()

        grid.addWidget(lbl1, 0, 0)
        grid.addWidget(cb1,  0, 1, Qt.AlignVCenter | Qt.AlignRight)

        # Second row
        lbl2 = QLabel(
            "I understand and agree that the Whisp API processes my data "
            "through Google Earth Engine (GEE)."
        )
        lbl2.setWordWrap(True)
        cb2 = QCheckBox()

        grid.addWidget(lbl2, 1, 0)
        grid.addWidget(cb2,  1, 1, Qt.AlignVCenter | Qt.AlignRight)

        main_layout.addLayout(grid)

        # OK / Cancel
        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, dlg)
        ok_btn = buttons.button(QDialogButtonBox.Ok)
        ok_btn.setEnabled(False)
        buttons.accepted.connect(dlg.accept)
        buttons.rejected.connect(dlg.reject)
        main_layout.addWidget(buttons)

        # Enable OK only when both boxes are checked
        def update_ok():
            ok_btn.setEnabled(cb1.isChecked() and cb2.isChecked())
        cb1.stateChanged.connect(update_ok)
        cb2.stateChanged.connect(update_ok)

        if dlg.exec_() != QDialog.Accepted:
            QMessageBox.warning(
                self.iface.mainWindow(),
                "Acknowledgement Required",
                "You must agree to the Terms of Service and GEE processing to proceed."
            )
            return False
        return True

    
    def on_submit_geojson(self):

        # self.show_terms_of_service()
                
        # First, check connectivity.
        if not is_connected("https://whisp.openforis.org", timeout=5):
            QMessageBox.critical(self.iface.mainWindow(),
                "Connectivity Error",
                "Can't connect to Whisp API. \n\nMissing internet connection or port issue. \n\nMake sure to be connected and that port 443 can be accessed.")
            return
             
        # If the column mapping isn't set or is empty, trigger initialization.
        if not self.whisp_columns_mapping:
            self.initialize_whisp_columns()
            # initialize_whisp_columns() blocks on its dialog, so once it returns
            # we either have mapping entries, or we bail out:
            if not self.whisp_columns_mapping:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "Initialization Error",
                    "Unable to fetch Whisp columns. Check connectivity and API key."
                )
                return

        # Show the main layer selection dialog.
        dialog = LayerSelectionDialog(self.whisp_columns_mapping, self.iface.mainWindow(), self.iface.activeLayer())
        if not dialog.exec_():
            return

        input_layer, output_file, selected_columns = dialog.getSelections()
        if not input_layer or not output_file or not selected_columns:
            QgsMessageLog.logMessage("Invalid selection. Ensure an input layer, output file, and at least one column are selected.",
                                    "WhispAnalysis", Qgis.Warning)
            return
        
        # Immediately convert the input layer to EPSG:4326
        input_layer = self.convert_to_epsg4326(input_layer)

        # ——— Manual flatten + backlink ID ———
        # 1) Copy original fields into a new QgsFields and add 'link_id'
        orig_fields = input_layer.fields()
        fields = QgsFields()
        for f in orig_fields:
            fields.append(f)
        # only add link_id if not already present
        if 'link_id' not in [f.name() for f in fields]:
            fields.append(QgsField('link_id', QVariant.String))

        # 2) Create an in-memory layer with those fields
        mem = QgsVectorLayer(
            f"{QgsWkbTypes.displayString(input_layer.wkbType())}?crs=EPSG:4326",
            "flattened",
            "memory"
        )
        mem.dataProvider().addAttributes(fields)
        mem.updateFields()

        # 3) Iterate, explode and back-link
        for orig_idx, feat in enumerate(input_layer.getFeatures(), start=1):
            geom = feat.geometry()
            wkb = geom.wkbType()
            # decide how to split
            if QgsWkbTypes.isMultiType(wkb):
                flat = QgsWkbTypes.flatType(wkb)
                if flat == QgsWkbTypes.PolygonGeometry:
                    parts = geom.asMultiPolygon()
                    maker = lambda poly: QgsGeometry.fromPolygonXY(poly)
                elif flat == QgsWkbTypes.PointGeometry:
                    parts = geom.asMultiPoint()
                    maker = lambda pt: QgsGeometry.fromPointXY(pt)
                else:
                    parts = geom.asGeometryCollection()
                    maker = lambda g: g
            else:
                # single‐part: wrap into a list
                parts = [geom]
                maker = lambda g: g

            for part_idx, part in enumerate(parts, start=1):
                subgeom = maker(part)
                new_feat = QgsFeature(fields)
                # copy original attributes + space for link_id
                attrs = feat.attributes() + [None]
                new_feat.setAttributes(attrs)
                new_feat.setGeometry(subgeom)
                new_feat['link_id'] = f"geom{orig_idx}_feat{part_idx}"
                mem.dataProvider().addFeatures([new_feat])

        input_layer = mem
        QgsMessageLog.logMessage("Manual flatten complete; created link_id field.", "WhispAnalysis", Qgis.Info)
        # ——————————————————————————————


        # Force normalization if the layer's attribute schema is empty
        # (Even if the layer reports fields, check if the attributes are effectively empty.)
        features = list(input_layer.getFeatures())
        if input_layer.fields().count() == 0 or (features and len(features[0].attributes()) == 0):
            QgsMessageLog.logMessage("Normalizing layer: exporting to temporary GeoJSON with properties.", "WhispAnalysis", Qgis.Info)
            temp_geojson_path = self.export_layer_with_properties(input_layer)
            input_layer = QgsVectorLayer(temp_geojson_path, "TempLayer", "ogr")
            if not input_layer.isValid():
                QgsMessageLog.logMessage("Failed to load temporary layer with properties.", "WhispAnalysis", Qgis.Critical)
                return

        # Check if the input layer is already fully whisped
        existing_whisp_fields = set(field.name() for field in input_layer.fields()).intersection(set(self.whisp_columns_mapping.keys()))
        if len(existing_whisp_fields) == len(self.whisp_columns_mapping):
            QgsMessageLog.logMessage("Input layer already contains all Whisp fields. Creating a clean copy for re‑whisping.", "WhispAnalysis", Qgis.Info)
            input_layer = self.create_clean_layer(input_layer)

        # Process the output file name
        if not os.path.isabs(output_file):
            input_source = input_layer.source()
            input_dir = os.path.dirname(input_source) if input_source else ""
            output_file = os.path.join(input_dir, output_file)
        if not output_file.lower().endswith(".geojson"):
            output_file += ".geojson"

        # Ensure the input layer has a "plotId" field
        if "plotId" not in [field.name() for field in input_layer.fields()]:
            input_layer.startEditing()
            input_layer.dataProvider().addAttributes([QgsField("plotId", QVariant.Int)])
            input_layer.updateFields()
            input_layer.commitChanges()

        # Populate the "plotId" values
        if not input_layer.isEditable():
            input_layer.startEditing()
        for idx, feature in enumerate(input_layer.getFeatures(), start=1):
            feature["plotId"] = idx
            input_layer.updateFeature(feature)
        input_layer.commitChanges()

        # Create the output layer
        output_layer = self.createNewOutputLayer(input_layer, output_file)
        if output_layer is None:
            QgsMessageLog.logMessage("Failed to create new output layer.", "WhispAnalysis", Qgis.Critical)
            return

        QgsProject.instance().addMapLayer(output_layer)
        self.selected_output_layer = output_layer

        self.ensure_required_fields(output_layer, selected_columns)
        geojson = self.get_selected_layer_as_geojson(input_layer)
        if not geojson:
            return


        # Create a modal progress dialog for the API call.
        processing_dialog = QDialog(self.iface.mainWindow())
        processing_dialog.setWindowTitle("Whisp")
        processing_dialog.setWindowIcon(QIcon(":/plugins/whisp_analysis/icon.png"))
        proc_layout = QVBoxLayout(processing_dialog)
        progress_label = QLabel("Sending request to Whisp API...")
        proc_layout.addWidget(progress_label)
        progress_bar = QProgressBar()

        num_features = input_layer.featureCount()
        base_time_ms = 10000  # 10 seconds base time

        geom_type = QgsWkbTypes.geometryType(input_layer.wkbType())
        if geom_type == QgsWkbTypes.PointGeometry:
            # 0.5 seconds per point (i.e., 500 ms per feature)
            additional_time_ms = num_features * 500
        elif geom_type == QgsWkbTypes.PolygonGeometry:
            # Calculate total area in hectares and add 0.1 seconds (100 ms) per hectare.
            d = QgsDistanceArea()
            d.setEllipsoid("WGS84")  # use WGS84 ellipsoid for geodesic area calculation
            total_hectares = 0.0
            for feature in input_layer.getFeatures():
                area_m2 = d.measureArea(feature.geometry())
                total_hectares += area_m2 / 10000.0  # convert square meters to hectares
            additional_time_ms = total_hectares * 100
        else:
            # Fallback: 40ms per second
            additional_time_ms = num_features * 10

        total_time_ms = base_time_ms + additional_time_ms
        ticks = int(math.ceil(total_time_ms / 100.0))  # timer updates every 100ms
        progress_bar.setRange(0, ticks)
        progress_bar.setValue(0)
        proc_layout.addWidget(progress_bar)

        # Add a Cancel button.
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        cancel_button = QPushButton("Cancel")
        btn_layout.addWidget(cancel_button)
        proc_layout.addLayout(btn_layout)

        processing_dialog.setLayout(proc_layout)
        processing_dialog.show()

        # After 3 seconds, change the label text.
        QTimer.singleShot(3000, lambda: progress_label.setText("Whisp API processing..."))

        # Create a timer that updates the progress bar every 100ms.
        timer = QTimer(processing_dialog)
        timer.setInterval(100)
        timer.timeout.connect(lambda: progress_bar.setValue(min(progress_bar.value() + 1, ticks)))
        timer.start()

        # Initialize cancellation flag.
        self.cancelled = False

        def cancel_operation():
            self.cancelled = True
            timer.stop()
            try:
                if self.thread is not None and self.thread.isRunning():
                    self.thread.terminate()  # Forcefully terminate the thread.
            except Exception as e:
                QgsMessageLog.logMessage(f"Error terminating thread: {e}", "WhispAnalysis", Qgis.Warning)
            processing_dialog.reject()  # Close the progress dialog.
            for action in self.actions:
                action.setEnabled(True)
            QgsMessageLog.logMessage("User cancelled the API call.", "WhispAnalysis", Qgis.Warning)

        cancel_button.clicked.connect(cancel_operation)

        for action in self.actions:
            action.setEnabled(False)

        self.worker = WhispWorker(geojson, self.api_key)
        self.thread = QThread()
        self.worker.moveToThread(self.thread)
        self.thread.started.connect(self.worker.run)
        self.worker.finished.connect(lambda result: self.on_api_response_with_progress(result, timer, progress_bar, processing_dialog))
        self.worker.progress.connect(lambda msg: progress_label.setText(msg))
        self.worker.finished.connect(self.thread.quit)
        self.worker.finished.connect(self.worker.deleteLater)
        self.thread.finished.connect(self.thread.deleteLater)
        self.thread.start()



    def on_api_response_with_progress(self, result, timer, progress_bar, processing_dialog):
        timer.stop()
        # Set the progress bar to its current maximum (i.e. API call phase complete)
        current_max = progress_bar.maximum()
        progress_bar.setValue(current_max)
        
        # Define an additional saving stage duration in milliseconds (3 seconds)
        saving_time_ms = 5000  
        saving_ticks = int(math.ceil(saving_time_ms / 100.0))
        
        # Extend the progress bar range to include the saving stage
        new_max = current_max + saving_ticks
        progress_bar.setRange(0, new_max)
        progress_bar.setValue(current_max)
        
        # Immediately change the label text for the saving stage.
        # (This will remain until the saving phase is complete.)
        progress_label = processing_dialog.findChild(QLabel)
        if progress_label:
            progress_label.setText("Saving to output layer...")
        
        # Create a new timer to simulate the saving progress.
        saving_timer = QTimer(processing_dialog)
        saving_timer.setInterval(100)
        
        def update_saving_progress():
            current_value = progress_bar.value()
            if current_value < new_max:
                progress_bar.setValue(current_value + 1)
            else:
                saving_timer.stop()
                processing_dialog.close()
                QApplication.processEvents()  # ensure UI updates
                for action in self.actions:
                    action.setEnabled(True)
                # Now immediately trigger the API response handling (success dialog)
                self.on_api_response(result)
        
        saving_timer.timeout.connect(update_saving_progress)
        saving_timer.start()



    def createNewOutputLayer(self, input_layer, output_file):
        """
        Write out the given layer as GeoJSON by hand (geometry + ALL properties),
        converting any non-serializable values via str(), so json.dump never fails.
        """
        features = []
        for feat in input_layer.getFeatures():
            # 1) geometry as dict
            geom = json.loads(feat.geometry().asJson())

            # 2) build a pure-Python properties dict
            props = {}
            for field in input_layer.fields():
                raw = feat[field.name()]
                val = raw
                # ensure it's JSON-serializable; if not, stringify it
                try:
                    json.dumps(val)
                except (TypeError, ValueError):
                    val = str(raw)
                props[field.name()] = val

            features.append({
                "type": "Feature",
                "geometry": geom,
                "properties": props
            })

        fc = {"type": "FeatureCollection", "features": features}

        # 3) ensure output folder exists
        out_dir = os.path.dirname(output_file)
        if out_dir and not os.path.isdir(out_dir):
            os.makedirs(out_dir, exist_ok=True)

        # 4) write it out
        try:
            with open(output_file, "w", encoding="utf-8") as f:
                json.dump(fc, f)
        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error writing GeoJSON by hand: {e}",
                "WhispAnalysis",
                Qgis.Critical
            )
            return None

        # 5) load into QGIS
        new_layer = self.iface.addVectorLayer(output_file, os.path.basename(output_file), "ogr")
        if not new_layer:
            QgsMessageLog.logMessage(
                f"Failed to load new output layer: {output_file}",
                "WhispAnalysis",
                Qgis.Critical
            )
        return new_layer





    def create_clean_layer(self, input_layer):
        # Filter out fields that are in the Whisp columns mapping
        original_fields = [field for field in input_layer.fields() if field.name() not in self.whisp_columns_mapping]
        
        # Convert the input layer's WKB type to a geometry type string.
        geom_type_str = QgsWkbTypes.displayString(input_layer.wkbType())
        crs = input_layer.crs().authid()
        
        # Create a new memory layer with the proper geometry type string and CRS.
        layer_str = f"{geom_type_str}?crs={crs}"
        clean_layer = QgsVectorLayer(layer_str, "clean_layer", "memory")
        
        if not clean_layer.isValid():
            QgsMessageLog.logMessage("Clean layer failed to initialize", "WhispAnalysis", Qgis.Critical)
            return None
        
        # Add only original fields to the new layer.
        dp = clean_layer.dataProvider()
        dp.addAttributes(original_fields)
        clean_layer.updateFields()
        
        # Copy features from the input layer, keeping only the original attributes.
        features = []
        for feature in input_layer.getFeatures():
            new_feature = QgsFeature()
            new_feature.setGeometry(feature.geometry())
            attr_list = [feature[field.name()] for field in original_fields]
            new_feature.setAttributes(attr_list)
            features.append(new_feature)
        dp.addFeatures(features)
        clean_layer.updateExtents()
        
        return clean_layer



    def on_api_response(self, result):
        self.iface.messageBar().clearWidgets()  # Remove progress bar

        # Re-enable UI actions
        for action in self.actions:
            action.setEnabled(True)

        if "error" in result:
            QgsMessageLog.logMessage(f"Whisp API Error: {result['error']}", "WhispAnalysis", Qgis.Critical)
            QMessageBox.critical(self.iface.mainWindow(), "Whisp Analysis Failed", f"Error: {result['error']}")
            return

        QgsMessageLog.logMessage(f"Whisp API Response: {result}", "WhispAnalysis", Qgis.Info)

        # Update the selected output layer with API results.
        if self.selected_output_layer:
            features = result.get("data", {}).get("features") or result.get("features", [])
            api_data = [feat["properties"] for feat in features]
            self.append_data_to_layer(self.selected_output_layer, api_data)



            msg_box = QMessageBox(self.iface.mainWindow())
            msg_box.setWindowIcon(QIcon(":/plugins/whisp_analysis/icon.png"))
            msg_box.setIcon(QMessageBox.Information)
            msg_box.setWindowTitle("Whisp")
            msg_box.setText("Geometries whisped successfully!\n\nValues appended to the output layer.")
            msg_box.exec_()






    def ensure_required_fields(self, layer, selected_columns):
        new_fields_added = False
        if not layer.isEditable():
            layer.startEditing()
        
        for field_name, type_str in self.whisp_columns_mapping.items():
            # Only add the field if it's one of the user-selected columns.
            if field_name in selected_columns:
                if field_name not in [field.name() for field in layer.fields()]:
                    if type_str == "int":
                        field_type = QVariant.Int
                    elif type_str == "double":
                        field_type = QVariant.Double
                    else:
                        field_type = QVariant.String
                    QgsMessageLog.logMessage(f"Adding new field: {field_name} (Type: {field_type})",
                                            "WhispAnalysis", Qgis.Info)
                    layer.addAttribute(QgsField(field_name, field_type))
                    new_fields_added = True
        
        if new_fields_added:
            layer.commitChanges()
            layer.startEditing()
            QgsMessageLog.logMessage("Committed new fields before API call.",
                                    "WhispAnalysis", Qgis.Info)





    def convert_to_epsg4326(self, layer):
        #Convert the selected layer to EPSG:4326 if it has a different CRS
        if not layer:
            QgsMessageLog.logMessage("No layer selected.", "WhispAnalysis", Qgis.Warning)
            return None

        source_crs = layer.crs()  # Get the current CRS
        target_crs = QgsCoordinateReferenceSystem("EPSG:4326")  # Define target CRS

        if source_crs.authid() != target_crs.authid():
            QgsMessageLog.logMessage(f"Reprojecting layer from {source_crs.authid()} to EPSG:4326...", "WhispAnalysis", Qgis.Info)

            # Determine the geometry type dynamically.
            geom_type_str = QgsWkbTypes.displayString(layer.wkbType())
            layer_name = layer.name() + " (EPSG:4326)"
            reprojected_layer = QgsVectorLayer(f"{geom_type_str}?crs=EPSG:4326", layer_name, "memory")

            # Copy fields from the original layer.
            reprojected_layer_data_provider = reprojected_layer.dataProvider()
            reprojected_layer_data_provider.addAttributes(layer.fields())
            reprojected_layer.updateFields()

            # Set up the coordinate transformation.
            transform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())

            reprojected_layer.startEditing()
            for feature in layer.getFeatures():
                new_feature = QgsFeature()
                new_feature.setAttributes(feature.attributes())
                geom = feature.geometry()
                geom.transform(transform)
                new_feature.setGeometry(geom)
                reprojected_layer_data_provider.addFeature(new_feature)
            reprojected_layer.commitChanges()

            QgsMessageLog.logMessage("Layer reprojected to EPSG:4326 successfully!", "WhispAnalysis", Qgis.Info)
            # Do not add the reprojected layer to the project; it remains in memory.
            return reprojected_layer

        else:
            QgsMessageLog.logMessage("Layer is already in EPSG:4326.", "WhispAnalysis", Qgis.Info)
            return layer








    def get_selected_layer_as_geojson(self, layer):
        # Convert selected layer to EPSG:4326 if necessary and export it as GeoJSON, ensuring a 'plotId' field exists
        
        # Convert CRS to EPSG:4326 if needed
        layer = self.convert_to_epsg4326(layer)

        # Ensure 'plotId' field exists
        if "plotId" not in [field.name() for field in layer.fields()]:
            QgsMessageLog.logMessage("Adding 'plotId' field before export.", "WhispAnalysis", Qgis.Info)
            layer.startEditing()
            layer.addAttribute(QgsField("plotId", QVariant.Int))
            layer.commitChanges()
            layer.startEditing()

        QgsMessageLog.logMessage("Populating 'plotId' values.", "WhispAnalysis", Qgis.Info)
        layer.startEditing()
        for index, feature in enumerate(layer.getFeatures(), start=1):
            feature["plotId"] = index  # Store as integer
            QgsMessageLog.logMessage(f"Assigned plotId {index} to feature ID {feature.id()}", "WhispAnalysis", Qgis.Info)
            layer.updateFeature(feature)
        layer.commitChanges()

        # Convert layer features to GeoJSON
        features = []
        for feature in layer.getFeatures():
            features.append(feature.geometry().asJson())

        geojson = {
            "type": "FeatureCollection",
            "features": [
                {"type": "Feature", "geometry": eval(f), "properties": {"plotId": feature["plotId"]}}
                for f, feature in zip(features, layer.getFeatures())
            ],
        }
        return geojson



    def submit_geojson(self, geojson):
        # Submit GeoJSON to the Whisp API and append results to the layer
        url = "https://whisp.openforis.org/api/submit/geojson"
        headers = {
            "Content-Type": "application/json",
            "x-api-key": self.api_key
        }

        try:
            response = requests.post(url, json=geojson, headers=headers)
            if response.status_code == 200:
                result = response.json()
                QgsMessageLog.logMessage(f"API Response: {result}", "WhispAnalysis", Qgis.Info)

                features = result.get("data", {}).get("features") or result.get("features", [])
                api_data = [feat["properties"] for feat in features]

                layer = self.iface.activeLayer()
                if layer:
                    self.append_data_to_layer(layer, api_data)
                else:
                    QgsMessageLog.logMessage(
                        "No layer selected to append data.", "WhispAnalysis", Qgis.Warning)




            else:
                QgsMessageLog.logMessage(
                    f"Failed to submit GeoJSON. Status code: {response.status_code}, Error: {response.text}",
                    "WhispAnalysis",
                    Qgis.Critical,
                )
        except Exception as e:
            QgsMessageLog.logMessage(f"Error during GeoJSON submission: {str(e)}", "WhispAnalysis", Qgis.Critical)

    

    def append_data_to_layer(self, layer, api_data):
        # Append API response data as attributes to the selected layer, ensuring correct data types
        if not layer.isEditable():
            layer.startEditing()

        # Get field data types
        field_types = {field.name(): field.type() for field in layer.fields()}

        for feature in layer.getFeatures():
            feature_plot_id = str(feature["plotId"])
            matched = False

            for row in api_data:
                if str(row["plotId"]) == feature_plot_id:
                    matched = True
                    QgsMessageLog.logMessage(f"Updating feature {feature.id()} with API data.", "WhispAnalysis", Qgis.Info)

                    for key, value in row.items():
                        if key in field_types:  # Ensure the field exists
                            if field_types[key] == QVariant.Double:  # Numeric field expected
                                try:
                                    if isinstance(value, dict):
                                        QgsMessageLog.logMessage(f"Value for field {key} is a dict; converting to string.", "WhispAnalysis", Qgis.Warning)
                                        feature[key] = str(value)
                                    else:
                                        value_float = float(value)
                                        # Whisp API returns Area for points. If the value is 0.01 or smaller, assign NULL.
                                        if key == "Area" and value_float <= 0.01:
                                            feature[key] = None
                                        else:
                                            if key in ["Centroid_lon", "Centroid_lat"]:
                                                feature[key] = round(value_float, 6)
                                            else:
                                                feature[key] = round(value_float, 3)
                                except (ValueError, TypeError) as e:
                                    QgsMessageLog.logMessage(f"Failed to convert {value} for {key}: {e}", "WhispAnalysis", Qgis.Warning)
                            elif field_types[key] == QVariant.Int:  # Integer field expected
                                try:
                                    feature[key] = int(value)
                                except (ValueError, TypeError) as e:
                                    QgsMessageLog.logMessage(f"Failed to convert {value} to integer for {key}: {e}", "WhispAnalysis", Qgis.Warning)
                            else:
                                feature[key] = str(value)  # For non-numeric fields, just store as string

                    layer.updateFeature(feature)
                    break  # Stop searching once a match is found

            if not matched:
                QgsMessageLog.logMessage(f"No match found for feature {feature.id()}", "WhispAnalysis", Qgis.Warning)

        layer.commitChanges()
        layer.triggerRepaint()
        QgsMessageLog.logMessage("Layer updated with API data.", "WhispAnalysis", Qgis.Info)








