# coding:utf-8
from PyQt5.QtCore import pyqtSignal
from PyQt5.QtCore import QEasingCurve
from PyQt5.QtCore import QPoint
from PyQt5.QtCore import QPropertyAnimation
from PyQt5.QtCore import Qt
from PyQt5.QtCore import QTimer
from PyQt5.QtGui import QColor
from PyQt5.QtGui import QPainter
from PyQt5.QtGui import QPixmap
from PyQt5.QtWidgets import QApplication
from PyQt5.QtWidgets import QFrame
from PyQt5.QtWidgets import QGraphicsDropShadowEffect
from PyQt5.QtWidgets import QGraphicsOpacityEffect
from PyQt5.QtWidgets import QHBoxLayout
from PyQt5.QtWidgets import QLabel
from PyQt5.QtWidgets import QToolButton
from PyQt5.QtWidgets import QWidget

from app.common.style_sheet import setStyleSheet


class Tooltip(QFrame):
    """Tooltip"""

    def __init__(self, text: str, parent=None):
        super().__init__(parent=parent)
        self.setAttribute(Qt.WA_StyledBackground)
        # self.window().setAttribute(Qt.WA_TranslucentBackground)  # hides square box behind widget when borders

        self.text = text
        self.hBox = QHBoxLayout(self)
        self.label = QLabel(text, self)
        # self.ani = QPropertyAnimation(self, b'windowOpacity', self)

        # set layout
        self.hBox.addWidget(self.label)
        self.hBox.setContentsMargins(10, 7, 10, 7)

        # set style
        # self.setDarkTheme(False)
        self.__setQss()

    def showEvent(self, e) -> None:
        shadow_effect = QGraphicsDropShadowEffect(self)
        shadow_effect.setBlurRadius(32)
        shadow_effect.setColor(QColor(0, 0, 0, 60))
        shadow_effect.setOffset(0, 5)
        self.setGraphicsEffect(shadow_effect)
        super().showEvent(e)

    def hideEvent(self, e) -> None:
        super().hideEvent(e)
        self.setGraphicsEffect(None)

    def closeEvent(self, e) -> None:
        super().closeEvent(e)
        self.setGraphicsEffect(None)

    def setText(self, text: str):
        """set text on tooltip"""
        self.text = text
        self.label.setText(text)
        self.label.adjustSize()
        self.adjustSize()

    def __setQss(self):
        """set style sheet"""
        setStyleSheet(self, 'tooltip')
        self.label.adjustSize()
        self.adjustSize()

    def setDarkTheme(self, dark=False):
        """set dark theme"""
        dark = 'true' if dark else 'false'
        self.setProperty('dark', dark)
        self.label.setProperty('dark', dark)
        self.setStyle(QApplication.style())


class StateTooltip(QWidget):
    """State tooltip"""

    closedSignal = pyqtSignal()

    def __init__(self, title, content, parent=None):
        """
        Parameters
        ----------
        title: str
            title of tooltip
        content: str
            content of tooltip
        parant:
            parent window
        """
        super().__init__(parent)
        self.title = title
        self.content = content
        self.delete_later: bool = True

        self.titleLabel = QLabel(self.title, self)
        self.contentLabel = QLabel(self.content, self)
        self.rotateTimer = QTimer(self)
        self.closeTimer = QTimer(self)
        self.opacityEffect = QGraphicsOpacityEffect(self)
        self.slideAni = QPropertyAnimation(self, b'pos')
        self.opacityAni = QPropertyAnimation(self.opacityEffect, b'opacity')
        self.busyImage = QPixmap(':/images/state_tooltip/running.png')
        self.doneImage = QPixmap(':/images/state_tooltip/completed.png')
        self.closeButton = QToolButton(self)

        self.isDone = False
        self.rotateAngle = 0
        self.deltaAngle = 20

        self.__initWidget()

    def __initWidget(self):
        """initialize widgets"""
        self.setAttribute(Qt.WA_StyledBackground)
        self.rotateTimer.setInterval(50)
        self.closeTimer.setInterval(1000)
        self.contentLabel.setMinimumWidth(250)
        self.setGraphicsEffect(self.opacityEffect)
        self.opacityEffect.setOpacity(1)

        # connect signal to slot
        self.closeButton.clicked.connect(self.__onCloseButtonClicked)
        self.rotateTimer.timeout.connect(self.__rotateTimerFlowSlot)
        self.closeTimer.timeout.connect(self.__slowlyClose)

        self.__setQss()
        self.__initLayout()

        self.rotateTimer.start()

    def __initLayout(self):
        """initialize layout"""
        self.setFixedSize(
            max(self.titleLabel.width(), self.contentLabel.width()) + 70, 64
        )
        self.titleLabel.move(40, 11)
        self.contentLabel.move(15, 34)
        self.closeButton.move(self.width() - 30, 23)

    def __setQss(self):
        """set style sheet"""
        self.titleLabel.setObjectName('titleLabel')
        self.contentLabel.setObjectName('contentLabel')
        setStyleSheet(self, 'state_tooltip')
        self.titleLabel.adjustSize()
        self.contentLabel.adjustSize()

    def setTitle(self, title: str):
        """set the title of tooltip"""
        self.title = title
        self.titleLabel.setText(title)
        self.titleLabel.adjustSize()

    def setContent(self, content: str):
        """set the content of tooltip"""
        self.content = content
        self.contentLabel.setText(content)

        # adjustSize() will mask spinner get stuck
        # self.contentLabel.adjustSize()

    def setState(self, isDone=False):
        """set the state of tooltip"""
        self.isDone = isDone
        self.update()
        if self.isDone:
            self.closeTimer.start()

    def __onCloseButtonClicked(self):
        """close button clicked slot"""
        self.closedSignal.emit()
        self.hide()

    def __slowlyClose(self):
        """fade out"""
        self.rotateTimer.stop()
        self.opacityAni.setEasingCurve(QEasingCurve.Linear)
        self.opacityAni.setDuration(300)
        self.opacityAni.setStartValue(1)
        self.opacityAni.setEndValue(0)
        if self.delete_later:
            self.opacityAni.finished.connect(self.deleteLater)
        else:
            self.opacityAni.finished.connect(self.hide)
        self.opacityAni.start()

    def __rotateTimerFlowSlot(self):
        """rotate timer time out slot"""
        try:
            self.rotateAngle = (self.rotateAngle + self.deltaAngle) % 360
            self.update()
        except Exception as e:
            # if the widget get deleted sometimes this function raises an exception
            pass

    def getSuitablePos(self):
        """get suitable position in main window"""
        for i in range(10):
            dy = i * (self.height() + 20)
            pos = QPoint(self.window().width() - self.width() - 30, 30 + dy)
            widget = self.window().childAt(pos + QPoint(2, 2))
            if isinstance(widget, (StateTooltip, ToastTooltip)):
                pos += QPoint(0, self.height() + 20)
            else:
                break
        return pos

    def showEvent(self, e):
        pos = self.getSuitablePos()
        self.slideAni.setDuration(200)
        self.slideAni.setEasingCurve(QEasingCurve.OutQuad)
        self.slideAni.setStartValue(QPoint(self.window().width(), pos.y()))
        self.slideAni.setEndValue(pos)
        self.slideAni.start()
        super().showEvent(e)

    def paintEvent(self, e):
        """paint state tooltip"""
        super().paintEvent(e)
        painter = QPainter(self)
        painter.setRenderHints(QPainter.SmoothPixmapTransform)
        painter.setPen(Qt.NoPen)
        if not self.isDone:
            painter.translate(24, 23)
            painter.rotate(self.rotateAngle)
            painter.drawPixmap(
                -int(self.busyImage.width() / 2),
                -int(self.busyImage.height() / 2),
                self.busyImage,
            )
        else:
            painter.drawPixmap(
                14, 13, self.doneImage.width(), self.doneImage.height(), self.doneImage
            )


class QGisTaskStateToolTip(StateTooltip):
    """qgis task state tooltip"""

    def __init__(self, title: str, content: str, parent=None):
        super().__init__(title=title, content=content, parent=parent)
        self.hide()

    def complete_task(self):
        """complete one task"""
        self.setTitle(self.tr('Fertig'))
        self.setContent(self.tr('Prozess abgeschlossen, bitte überprüfen'))
        self.setState(True)


class ToastTooltip(QWidget):
    """Toast tooltip"""

    def __init__(
        self, title: str, content: str, icon: str, parent=None, p_interval: int = 2500
    ):
        """
        Parameters
        ----------
        title: str
            title of tooltip
        content: str
            content of tooltip
        icon: str
            icon of toast, can be `completed` or `info`
        parent:
            parent window
        """
        super().__init__(parent)
        self.title = title
        self.content = content
        self.icon = f':/images/state_tooltip/{icon}.png'

        self.titleLabel = QLabel(self.title, self)
        self.contentLabel = QLabel(self.content, self)
        self.iconLabel = QLabel(self)
        self.closeButton = QToolButton(self)
        self.closeTimer = QTimer(self)
        self.interval: int = p_interval
        self.opacityEffect = QGraphicsOpacityEffect(self)
        self.opacityAni = QPropertyAnimation(self.opacityEffect, b'opacity')
        self.slideAni = QPropertyAnimation(self, b'pos')

        self.__initWidget()

    def setTitle(self, p_text: str):
        """set the title"""
        self.titleLabel.setText(p_text)

    def setContent(self, p_text: str):
        """set the content"""
        self.contentLabel.setText(p_text)

    def setIcon(self, p_icon_path: str):
        """set the icon path"""
        self.icon = p_icon_path
        self.iconLabel.setPixmap(QPixmap(self.icon))
        self.iconLabel.adjustSize()

    def __initWidget(self):
        """initialize widgets"""
        self.setAttribute(Qt.WA_StyledBackground)
        self.closeTimer.setInterval(self.interval)
        self.contentLabel.setMinimumWidth(250)

        self.iconLabel.setPixmap(QPixmap(self.icon))
        self.iconLabel.adjustSize()
        self.iconLabel.move(15, 13)

        self.setGraphicsEffect(self.opacityEffect)
        self.opacityEffect.setOpacity(1)

        # connect signal to slot
        self.closeButton.clicked.connect(self.hide)
        self.closeTimer.timeout.connect(self.__fadeOut)

        self.__setQss()
        self.__initLayout()

    def __initLayout(self):
        """initialize layout"""
        self.setFixedSize(
            max(self.titleLabel.width(), self.contentLabel.width()) + 70, 64
        )
        self.titleLabel.move(40, 11)
        self.contentLabel.move(15, 34)
        self.closeButton.move(self.width() - 30, 23)

    def __setQss(self):
        """set style sheet"""
        self.titleLabel.setObjectName('titleLabel')
        self.contentLabel.setObjectName('contentLabel')
        setStyleSheet(self, 'state_tooltip')
        self.titleLabel.adjustSize()
        self.contentLabel.adjustSize()

    def __fadeOut(self):
        """fade out"""
        self.opacityAni.setDuration(300)
        self.opacityAni.setStartValue(1)
        self.opacityAni.setEndValue(0)
        self.opacityAni.finished.connect(self.deleteLater)
        self.opacityAni.start()

    def getSuitablePos(self):
        """get suitable position in main window"""
        for i in range(10):
            dy = i * (self.height() + 20)
            pos = QPoint(self.window().width() - self.width() - 30, 30 + dy)
            widget = self.window().childAt(pos + QPoint(2, 2))
            if isinstance(widget, (StateTooltip, ToastTooltip, QGisTaskStateToolTip)):
                pos += QPoint(0, self.height() + 20)
            else:
                break
        return pos

    def showEvent(self, e):
        """
        :param e: show event
        :return: None
        """
        pos = self.getSuitablePos()
        self.slideAni.setDuration(200)
        self.slideAni.setEasingCurve(QEasingCurve.OutQuad)
        self.slideAni.setStartValue(QPoint(self.window().width(), pos.y()))
        self.slideAni.setEndValue(pos)
        self.slideAni.start()
        super().showEvent(e)
        self.closeTimer.start()
