from PyQt5.QtCore import QEvent
from PyQt5.QtCore import QObject
from PyQt5.QtCore import QPoint
from PyQt5.QtGui import QEnterEvent
from PyQt5.QtWidgets import QApplication
from PyQt5.QtWidgets import QPushButton
from PyQt5.QtWidgets import QToolButton

from ...common.config import config
from ...components.widgets.tooltip import Tooltip


class TooltipButton(QToolButton):
    """Tool button with a tooltip"""

    def __init__(self, parent=None):
        super().__init__(parent=parent)
        self.__tooltip = None
        self.__darkTooltip = config.theme == 'dark'
        self.installEventFilter(self)

    def eventFilter(self, obj: QObject, e: QEvent) -> bool:
        if obj is self:
            if e.type() == QEvent.ToolTip:
                return True

        return super().eventFilter(obj, e)

    def enterEvent(self, e):
        super().enterEvent(e)
        if not self.toolTip():
            return

        if self.__tooltip is None:
            self.__tooltip = Tooltip(self.toolTip(), self.window())
            self.__tooltip.setDarkTheme(self.__darkTooltip)

        # update tooltip
        if self.__tooltip.text != self.toolTip():
            self.__tooltip.setText(self.toolTip())

        # the tooltip must be moved outside the button area
        pos = self.mapTo(self.window(), QPoint(0, 0))
        x = pos.x() + self.width() // 2 - self.__tooltip.width() // 2
        y = pos.y() - 2 - self.__tooltip.height()

        # adjust position to prevent tooltips from appearing outside the window
        x = min(max(5, x), self.window().width() - self.__tooltip.width() - 5)

        self.__tooltip.move(x, y)
        self.__tooltip.show()

    def leaveEvent(self, e):
        super().leaveEvent(e)
        if self.__tooltip:
            self.__tooltip.hide()

    def hideEvent(self, e):
        super().hideEvent(e)
        if self.__tooltip:
            self.__tooltip.hide()

    def setDarkToolTip(self, dark=False):
        """set whether to use the dark theme of tooltip"""
        self.__darkTooltip = dark

    def hideToolTip(self):
        """hide tooltip"""
        if self.__tooltip:
            self.__tooltip.hide()


class TooltipPushButton(QPushButton):
    """Push button with a tooltip"""

    def __init__(self, text: str, parent=None):
        super().__init__(text, parent)
        self._tooltip: Tooltip = None
        self.__darkTooltip = False
        self.setMouseTracking(True)
        self.installEventFilter(self)

    def eventFilter(self, obj: QObject, e: QEvent) -> bool:
        if obj is self:
            if e.type() == QEvent.ToolTip:
                return True
        return super().eventFilter(obj, e)

    def enterEvent(self, e: QEnterEvent):
        super().enterEvent(e)
        if not self.toolTip():
            return

        if self._tooltip is None:
            self._tooltip = Tooltip(self.toolTip(), self.window())
            self._tooltip.setDarkTheme(self.__darkTooltip)

        # update tooltip
        if self._tooltip.text != self.toolTip():
            self._tooltip.setText(self.toolTip())

        # the tooltip must be moved outside the button area
        self._move_tooltip(e)

    def _move_tooltip(self, e) -> None:
        """
        move tooltip

        :param p_glob_pos:
        :return:
        """
        pos = self.mapTo(self.window(), QPoint(0, 0))
        x = pos.x() - self._tooltip.width() // 2
        y = pos.y() - 2 - self._tooltip.height()

        # adjust position to prevent tooltips from appearing outside the window
        x = min(max(5, x), self.window().width() - self._tooltip.width() - 5)

        # prevent tooltips from appearing outside the screen
        if (
            e.globalX() + self._tooltip.width() >=
            QApplication.desktop().screenGeometry().width()
        ):
            x = pos.x() - self._tooltip.width()

        self._tooltip.move(x, y)
        self._tooltip.show()

    def leaveEvent(self, e):
        super().leaveEvent(e)
        if self._tooltip:
            self._tooltip.hide()

    def hideEvent(self, e):
        super().hideEvent(e)
        if self._tooltip:
            self._tooltip.hide()

    def setDarkToolTip(self, dark=False):
        """set whether to use the dark theme of tooltip"""
        self.__darkTooltip = dark

    def hideToolTip(self):
        """hide tooltip"""
        if self._tooltip:
            self._tooltip.hide()


class AnalysisInfoButton(TooltipPushButton):
    def _move_tooltip(self, e) -> None:
        """
        move tooltip

        :param p_glob_pos:
        :return:
        """
        pos = self.mapTo(self.window(), QPoint(0, 0))
        x = pos.x() - self._tooltip.width() // 2
        y = pos.y() - self._tooltip.height() + 10

        # adjust position to prevent tooltips from appearing outside the window
        x = min(max(5, x), self.window().width() - self._tooltip.width() - 5)

        # prevent tooltips from appearing outside the screen
        if (
            e.globalX() + self._tooltip.width() >=
            QApplication.desktop().screenGeometry().width()
        ):
            x = pos.x() - self._tooltip.width()

        self._tooltip.move(x, y)
        self._tooltip.show()
