"""
/***************************************************************************
 AnalyzeTask
                                 A QGIS plugin
 wide are mapping
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2020-11-13
        git sha              : $Format:%H$
        copyright            : (C) 2020 by forschung@ciss.de
        email                : forschung@ciss.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import datetime
import time

from qgis.core import QgsTask, QgsMessageLog, Qgis

import weissflaechenkartierung.wfk_sources.xml_config_ctrl as xml_ctrl
import weissflaechenkartierung.wfk_sources.config as cfg
import weissflaechenkartierung.wfk_sources.QGisTool as qgis_tool
import weissflaechenkartierung.wfk_sources.result as r
import weissflaechenkartierung.wfk_sources.legend as leg
import weissflaechenkartierung.wfk_sources.layer as lay


class AnalyzeRunInfo:
    def __init__(self):
        self.layer = None
        self.blacklist: bool = False
        self.configuration = None


class AnalyzeTask(QgsTask):
    """
        class which holds information about a QgsTask
        """

    def __init__(self, desc: str,
                 analyze_run_info_list: list = None,
                 parent_dlg=None
                 ):
        """
        :param desc: description
        :param analyze_run_info_list: object which holds information about the analyze
        :param parent_dlg: access to the main window
        """
        QgsTask.__init__(self, description=desc, flags=QgsTask.CanCancel)
        self._analyze_run_info_list: list = analyze_run_info_list
        self._parent_dlg = parent_dlg
        self.task_info = None

    def run(self):
        """
        run func from analyze task
        """
        # init run result class
        self.task_info = r.RunResult()

        # start the task with the config and save the result layers
        try:
            r_list: list = []
            for task_to_run in self._analyze_run_info_list:
                if isinstance(task_to_run, AnalyzeRunInfo):
                    r_list.append(
                        self._run_task(
                            run_configuration=task_to_run.configuration,
                            layer=task_to_run.layer,
                            blacklist=task_to_run.blacklist
                        )
                    )
            self.task_info.run_result = r_list
            return True
        except AttributeError as e:
            if self.has_to_stop():  # if the task has canceled
                QgsMessageLog.logMessage(message="AnalyzeTask::AttributeError -> see python console", level=Qgis.Info)
                print(str(repr(e)))
                return True
            else:
                QgsMessageLog.logMessage(message="AnalyzeTask::AttributeError -> see python console", level=Qgis.Info)
                print(str(repr(e)))
        except Exception as e:
            if self.has_to_stop():
                return True
            else:
                QgsMessageLog.logMessage(message="AnalyzeTask::Unexpected -> see python console", level=Qgis.Info)
                print(str(repr(e)))
        return False

    def _run_task(self, run_configuration: cfg.Config, layer: lay.Layer, blacklist: bool):
        """
        execute the analyze task
        """
        QgsMessageLog.logMessage(message="AnalyzeTask::run::config: '" + run_configuration.name + "'",
                                 level=Qgis.Info)
        QgsMessageLog.logMessage(message="AnalyzeTask::run::layer name: '" + layer.name() + "'",
                                 level=Qgis.Info)

        if layer.selectedFeatureCount() == 0:
            layer_to_analyze = qgis_tool.QGisTool.clone_layer(input_layer=layer)
            print("AnalyzeTask::run::there where no selections on the layer")
        else:
            layer_to_analyze = qgis_tool.QGisTool.clone_selected_layer(input_layer=layer)
            print("AnalyzeTask::run::there where " + str(layer.selectedFeatureCount()) + " selections on the layer")

        self.setProgress(5)
        step_process = 1

        if len(run_configuration.module_list) == 0:
            QgsMessageLog.logMessage(message="AnalyzeTask:: there are no modules to run",
                                     level=Qgis.Info)
        else:
            for module in run_configuration.module_list:
                QgsMessageLog.logMessage(message="AnalyzeTask::run::module name: '" + module.name + "'",
                                         level=Qgis.Info)

                # run categorization (run result -> analyzed layer)
                layer_to_analyze = module.run_categorization(cat_layer=layer_to_analyze).run_result

                if layer_to_analyze is None:
                    QgsMessageLog.logMessage(message="AnalyzeTask::run::exception while analyzing",
                                             level=Qgis.Critical)
                    return layer_to_analyze

                self.setProgress(20 + (step_process * (20 / len(run_configuration.module_list))))
                step_process += 1

            # create legend object
            legend = leg.Legend(xml_section=xml_ctrl.XMLConfigCtrl.get_first_tag_in_child_nodes(
                xml=run_configuration.config_section,
                tag="legend"
            ),
                module=None)

            # for name convention
            analyzed_layer = layer_to_analyze

            # check if the layer is a black list layer
            # set new layer names
            if blacklist:
                legend.set_blacklisted_layer_name(analyzed_blacklisted_layer=analyzed_layer)
            else:
                legend.set_analyze_layer_name_param(analyzed_layer=analyzed_layer)
            return analyzed_layer
        return layer

    def finished(self, result):
        """
        called when the task has finished
        """
        for layer in self.task_info.run_result:
            qgis_tool.QGisTool.add_layer_to_map(layer=layer)

        self._parent_dlg.ui.btn_start_wfk.setEnabled(True)
        self._parent_dlg.ui.btn_refresh.setEnabled(True)
        self._parent_dlg.ui.btn_analyze.setEnabled(True)
        self._parent_dlg.ui.progressBar_analyze.setVisible(False)
        self._parent_dlg.ui.progressBar_analyze.setValue(0)
        self._parent_dlg.ui.label_movie.hide()
        self._parent_dlg.analyze_task = None
        self._parent_dlg.is_running_analyze = False

        # set end time
        self.task_info.end_time = time.time()
        QgsMessageLog.logMessage(message="analyze task elapsed time: " +
                                         str(datetime.timedelta(
                                             seconds=self.task_info.get_elapsed_time())) + " (H/M/S)",
                                 level=Qgis.Info)
