# -*- coding: utf-8 -*-
"""
/***************************************************************************
 WfkTask
                                 A QGIS plugin
 wide are mapping
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2020-11-13
        git sha              : $Format:%H$
        copyright            : (C) 2020 by forschung@ciss.de
        email                : forschung@ciss.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import datetime
import sys
import time

from weissflaechenkartierung.message_box_dlg import MsgMode, MessageBoxDlg
from qgis.core import QgsTask, QgsMessageLog, Qgis

import weissflaechenkartierung.wfk_sources.xml_config_ctrl as xml_ctrl
import weissflaechenkartierung.wfk_sources.config as cfg
import weissflaechenkartierung.wfk_sources.QGisTool as qgis_tool
import weissflaechenkartierung.wfk_sources.result as r
import weissflaechenkartierung.wfk_sources.legend as leg
import weissflaechenkartierung.wfk_sources.analyze_task as analyze


class WfkTask(QgsTask):
    """
    class which holds information about a QgsTask
    """

    def __init__(self, desc: str,
                 config_name: str,
                 dlg,
                 save_inter_results: bool,
                 layer,
                 config_path: str,
                 qgis_interface
                 ):
        """
        :param desc: description
        :param layer: layer object from the QGIS interface
        :param dlg: parent dlg class (caller)
        :param qgis_interface:
        :param save_inter_results:
        :param config_path: path to the configuration file
        """
        QgsTask.__init__(self, description=desc, flags=QgsTask.CanCancel)
        self._config_name = config_name
        self._config_path = config_path
        self._qgis_interface = qgis_interface
        self._qgis_layer = layer
        self._parent_dlg = dlg
        self._save_intermediate_results = save_inter_results
        self._layer_to_analyze_list: list = []

        self.task_info = None

    @staticmethod
    def handle_error(error):
        """
        handle error printing
        """
        QgsMessageLog.logMessage(message="WfkTask::AttributeError -> see python console", level=Qgis.Info)
        print(str(repr(error)))

    def run(self):
        """
        collect the needed data
        """
        self.task_info = r.RunResult()

        # initialize the xml controller
        config_ctrl = xml_ctrl.XMLConfigCtrl(xml_path=self._config_path)

        # get the first xml configuration obj by name
        first_config = config_ctrl.get_first_config_by_name(name=self._config_name)
        QgsMessageLog.logMessage(message="WfkTask:run: initialize first config: " + self._config_name,
                                 level=Qgis.Info)

        # initialize the config object
        config_obj = cfg.Config(parameters=config_ctrl.get_parameter_from_xml(xml=first_config),
                                name=config_ctrl.get_name_from_xml(element=first_config),
                                qgis_layer=self._qgis_layer,
                                config_section=first_config)

        # config_obj.module_list = modules
        config_obj.add_modules(config_ctrl.get_modules_from_xml(element=first_config,
                                                                config=config_obj))
        try:
            # start the task with the configuration and save the result layers
            self.task_info.run_result = self._start(config_obj=config_obj)
        except Exception as e:
            print(str(repr(e)))
            return False
        return True

    def _start(self, config_obj) -> list:
        """
        starts the process
        """
        result_list = []

        # print ou the start time
        QgsMessageLog.logMessage(message="- WfkTask:run: config: [" + config_obj.name + "] at: " + \
                                         self.task_info.get_time_formatted(elapsed_time=self.task_info.start_time),
                                 level=Qgis.Info)

        if self.has_to_stop():
            return result_list
        self.setProgress(5)

        # clone the wfk layer from qgis (from qgis interface obj)
        wfk_layer = qgis_tool.QGisTool.clone_layer(input_layer=self._qgis_layer)
        step = 0
        module_list_len = len(config_obj.module_list)

        # iterate over the configuration modules
        for module in config_obj.module_list:

            QgsMessageLog.logMessage(message="run module = " + module.name, level=Qgis.Info)

            # run the connector from the module
            if self.has_to_stop():
                return result_list

            module.run_connector()

            self.setProgress(5 + (step * (75 / module_list_len)) + (10 / module_list_len))

            # run the layer from the module
            if self.has_to_stop():
                return result_list

            module.run_layer()

            self.setProgress(5 + ((step * (75 / module_list_len)) + (30 / module_list_len)))

            # run differences
            if self.has_to_stop():
                return result_list

            diff_run_result = module.run_difference(diff_layer=wfk_layer)
            wfk_layer = diff_run_result.run_result  # run result -> diffed layer

            # check if the user would like to see the black listed features separately
            if hasattr(module.difference, "save_blacklisted_layer"):
                if module.difference.save_blacklisted_layer:
                    result_list.append(module.difference.diff_layer)

            # check if the difference class name is the blacklist class
            if module.difference.__class__.__name__ == "BlacklistingDifference":
                if module.difference.diff_layer is not None:
                    print(self.__class__.__name__ + " - prepare for blacklisting analyzing")
                    # create run analyze information object for analyze step
                    analyze_info = analyze.AnalyzeRunInfo()
                    analyze_info.layer = module.difference.diff_layer
                    analyze_info.blacklist = True
                    analyze_info.configuration = config_obj
                    self._layer_to_analyze_list.append(analyze_info)

            self.setProgress(5 + (step * (75 / module_list_len)) + (75 / module_list_len))
            step += 1

        if self.has_to_stop():
            return result_list

        QgsMessageLog.logMessage(message="run multi to single...", level=Qgis.Info)
        wfk_layer = qgis_tool.QGisTool.multi_2_single(base_layer=wfk_layer,
                                                      name_layer="tmp")

        if self.has_to_stop():
            return result_list

        if self._save_intermediate_results:
            for module in config_obj.module_list:
                result_list.extend(module.add_to_map())

        if self.has_to_stop():
            return result_list

        # legend
        legend = leg.Legend(xml_section=xml_ctrl.XMLConfigCtrl.get_first_tag_in_child_nodes(
            xml=config_obj.config_section,
            tag="legend"
        ),
            module=None)

        result_list.extend(legend.add_wfk_layer_to_map_param(wfk_layer=wfk_layer))

        # create run analyze information object
        analyze_info = analyze.AnalyzeRunInfo()

        analyze_info.layer = wfk_layer
        analyze_info.configuration = config_obj

        self._layer_to_analyze_list.append(analyze_info)
        return result_list

    def finished(self, result):
        """
        called when the task has finished, either successfully or wrong
        """
        if isinstance(result, bool):
            if result:
                try:
                    for layer in self.task_info.run_result:
                        qgis_tool.QGisTool.add_layer_to_map(layer=layer)
                        self._parent_dlg.analyze_info_list = self._layer_to_analyze_list
                    self._parent_dlg.change_menu(obj_name="btn_analyze")
                except Exception as e:
                    self.handle_error(error=e)
                    MessageBoxDlg(parent=self._parent_dlg,
                                  mode=MsgMode.error,
                                  txt=str(repr(e)))

                if self._parent_dlg.ui.checkBox_one_run.isChecked() and len(self._layer_to_analyze_list) > 0:
                    QgsMessageLog.logMessage(message="Afterwards the analysis starts", level=Qgis.Info)
                    self._parent_dlg.start_analyzing()
            else:
                self.handle_error(error=result)
                MessageBoxDlg(parent=self._parent_dlg,
                              mode=MsgMode.error,
                              txt=str(repr(sys.exc_info())))
                self._parent_dlg.analyze_info_list = []

        self._parent_dlg.ui.btn_start_wfk.setText("Start")
        self._parent_dlg.is_running_wfk = False
        self._parent_dlg.wfk_qgis_task = None
        self._parent_dlg.ui.progressBar.setVisible(False)
        self._parent_dlg.ui.progressBar.setValue(0)
        self._parent_dlg.ui.label_movie.hide()
        self._parent_dlg.ui.btn_refresh.setEnabled(True)
        self._parent_dlg.ui.btn_analyze.setEnabled(True)

        # set end time
        self.task_info.end_time = time.time()
        QgsMessageLog.logMessage(message="the wfk task elapsed time: " +
                                         str(datetime.timedelta(
                                             seconds=self.task_info.get_elapsed_time())) + " (H/M/S)",
                                 level=Qgis.Info)

    def has_to_stop(self) -> bool:
        """
        checks if the task needs to stopped
        """
        return True if self.isCanceled() else False
