# -*- coding: utf-8 -*-

"""
This script download polygons from the CORINE landcover polygons for a selected bounding box.
"""
from PyQt5.QtCore import QVariant
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingLayerPostProcessorInterface,
    QgsProcessing,
    QgsFeatureSink,
    QgsVectorLayer,
    QgsProcessingException,
    QgsProcessingAlgorithm,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterFeatureSource,
    QgsCoordinateReferenceSystem,
    QgsProcessingUtils,
    QgsSnappingConfig,
    QgsProject,
    QgsTolerance
)

from qgis import processing
import os, inspect


def set_snapping_options():
    proj = QgsProject.instance()
    config = proj.snappingConfig()
    proj.setAvoidIntersectionsMode(QgsProject.AvoidIntersectionsMode(1))
    proj.setTopologicalEditing(True)
    config.setMode(QgsSnappingConfig.ActiveLayer)
    config.setType(QgsSnappingConfig.Vertex)
    config.setTolerance(10)
    config.setUnits(QgsTolerance.Pixels)
    config.setEnabled(True)
    proj.setSnappingConfig(config)
    return proj



class Renamer (QgsProcessingLayerPostProcessorInterface):
    def __init__(self, layer_name):
        self.name = layer_name
        super().__init__()

    def postProcessLayer(self, layer, context, feedback):
        layer.setName(self.name)


class DownloadCorineRoughness(QgsProcessingAlgorithm):
    """
    This scripts extracts a roughness map from polygons
    and saves it as a WAsP .map file
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    dest_id = None  # Save a reference to the output layer id
    INPUT = "INPUT"
    TABLE= "TABLE"
    OUTPUT = "OUTPUT"

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return DownloadCorineRoughness()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "download_corine"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Get CORINE landcover polygons")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("WAsP scripts")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "wasp_scripts"

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr(
            """Extracts landcover polygons from CORINE data

            The input is a 'Bounding box' layer with a single Polygon where we want to retrieve the data. Polygons from CORINE that are inside this area are downloaded. Note that the projection of the clipping layer determines the projection of the output. More information about the CORINE data can be found <a href='https://land.copernicus.eu/pan-european/corine-land-cover'>here</a>.

            The returned vector layer contains 'id', 'z0' and 'd' attributes which represents a polygon with certain landcover 'id' with a roughness length 'z0' and displacement height 'd', respectively. The lookup table that is used for tying a ID to roughness is found <a href='https://doi.org/10.5194/wes-6-1379-2021'>here</a>."""
        )

    def initAlgorithm(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        # self.addParameter(
        #    QgsProcessingParameterExtent(
        #        self.EXTENT,
        #        self.tr('Extent layer')
        #    )
        # )

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Clipping layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )


        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT, self.tr("CORINE layer")
            )
        )


    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)

        # If source was not found, throw an exception to indicate that the algorithm
        # encountered a fatal error. The exception text can be any string, but in this
        # case we use the pre-built invalidSourceError method to return a standard
        # helper text for when a source cannot be evaluated
        if source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT)
            )

        # Send some information to the user
        feedback.pushInfo("CRS is {}".format(source.sourceCrs().authid()))
        clipping_layer = self.parameterAsVectorLayer(parameters, self.INPUT, context)

        if clipping_layer.crs().isGeographic():
            raise QgsProcessingException("WAsP files cannot be saved in geographic coordinates! Please choose a projection with a metric coordinate system!")

        nlines = clipping_layer.featureCount()
        if nlines > 1:
            raise QgsProcessingException(
                f"The clipping layer must consist of a single feature only to avoid downloading too many files! Yours contains {nlines}"
            )

        clipping_layer_features = clipping_layer.getFeatures()
        for f in clipping_layer_features:
            geom = f.geometry()
            size = geom.area()
            if size > 3.6E9:
                raise QgsProcessingException(
                    "The clipping layer must be smaller than 60x60 km!"
                )

        # we slightly extend the mask layer so we have enough sample even in corner pixels
        # when we resample to a more coarse resolution
        buffered_layer = processing.run(
        "native:buffer",
        {
            "DISSOLVE": False,
            "DISTANCE": 100,
            "END_CAP_STYLE": 2,
            "INPUT": clipping_layer,
            "JOIN_STYLE": 1,
            "MITER_LIMIT": 1,
            "OUTPUT": "TEMPORARY_OUTPUT",
            "SEGMENTS": 1,
        },
        context=context,
        feedback=feedback,
        )["OUTPUT"]
        
        feedback.pushInfo('Reprojecting mask layer to latlon projection...')
        clipping_layer_reproj = processing.run("native:reprojectlayer", {
        'INPUT': buffered_layer,
        'OUTPUT': 'TEMPORARY_OUTPUT',
        'TARGET_CRS' : "EPSG:4326"
        }, context=context, feedback=feedback)['OUTPUT']

        clipping_layer_reproj_single = processing.run("native:multiparttosingleparts", {
        'INPUT': clipping_layer_reproj,
        'OUTPUT': 'TEMPORARY_OUTPUT',
        }, context=context, feedback=feedback)['OUTPUT']

        clipping_layer_features = clipping_layer_reproj_single.getFeatures()
        list_pts = []
        for feat in clipping_layer_features:
            poly = feat.geometry().asPolygon()
            for line in poly:
                for i in range(len(line)):
                    x = line[i][0]
                    y = line[i][1]
                    list_pts.append(f"{x},{y}")
        query_string = ";".join(list_pts)

        # load worldcover grid
        base_url = "https://api.globalwindatlas.info/qgis-data/v1/corine_2018?points_str="
        feedback.pushInfo(f"Opening CORINE polygons from the URL '{base_url}{query_string}'")
        corine_polygons = QgsVectorLayer(f"{base_url}{query_string}", "Vector_Layer", "ogr")
        layerCrs = QgsCoordinateReferenceSystem.fromEpsgId(4326)
        corine_polygons.setCrs(layerCrs) # geojson is always epsg:4326

        if not corine_polygons.isValid():
            raise QgsProcessingException(
                f"Querying the URL '{base_url}{query_string}' did not return a valid GeoJSON object!"
            )

        alg_params = {
            'FIELD_LENGTH': 0,
            'FIELD_NAME': 'id',
            'FIELD_PRECISION': 0,
            'FIELD_TYPE': 1,
            'FORMULA': f"523",
            'INPUT': clipping_layer,
            'NEW_FIELD': True,
            'OUTPUT': 'TEMPORARY_OUTPUT',
        }
        clipping_layer_with_id = processing.run('native:fieldcalculator', alg_params, context=context, feedback=feedback)["OUTPUT"]

        if corine_polygons.featureCount() == 0:
            feedback.pushWarning(f"The clipping layer did not contain any usuable landcover areas! Make sure you are in an area that is covered by the CORINE data (https://land.copernicus.eu/pan-european/corine-land-cover/clc2018). A single offshore landcover map with the size of your bounding box will be generated.")
            corine_polygons_reproj_clipped = clipping_layer_with_id
        else:
            # we slightly extend the mask layer so we have enough sample even in corner pixels
            # to do zero buffer to fix ring self-intersections (see here: https://gis.stackexchange.com/questions/265661/fixing-polygon-self-intersection-in-qgis)
            feedback.pushInfo("Buffered features: "+ str(corine_polygons.featureCount()))
            corine_polygons_buf = processing.run(
            "native:buffer",
            {
                "DISSOLVE": False,
                "DISTANCE": 0,
                "END_CAP_STYLE": 0,
                "INPUT": corine_polygons,
                "JOIN_STYLE": 0,
                "MITER_LIMIT": 1,
                "OUTPUT": "TEMPORARY_OUTPUT",
                "SEGMENTS": 1,
            },
            context=context,
            feedback=feedback,
            )["OUTPUT"]
            feedback.pushInfo("Buffered features: "+ str(corine_polygons_buf.featureCount()))
            feedback.pushInfo('Reprojecting mask layer to input projection...')
            corine_polygons_reproj = processing.run("native:reprojectlayer", {
            'INPUT': corine_polygons_buf,
            'OUTPUT': 'TEMPORARY_OUTPUT',
            'TARGET_CRS' : source.sourceCrs()
            }, context=context, feedback=feedback)['OUTPUT']

            corine_polygons_reproj_clipped = processing.run(
                "native:clip",
                {
                    "INPUT": corine_polygons_reproj,
                    "OUTPUT": "TEMPORARY_OUTPUT",
                    "OVERLAY": clipping_layer,
                },
                context=context,
                feedback=feedback,
            )["OUTPUT"]


        feedback.pushInfo('Joining CORINE roughness table')
        script_dir = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        path = os.path.join(script_dir, "landcovertables", "CORINE.gpkg") # this file contains the landcover lookup table to convert landcover classes to roughness
        table = QgsVectorLayer(path, "Vector_Layer", "ogr")

        if not table.isValid():
            raise QgsProcessingException(f"Could not load table {path}")

        #for feature in table.getFeatures():
        #    feedback.pushInfo(str(feature["index"]) + str(feature["d"]) + str(feature["z0"])  + str(feature["desc"]) )

        alg_params = {
            'DISCARD_NONMATCHING': True,
            'FIELD': 'id',
            'FIELDS_TO_COPY': ['d','z0','desc'],
            'FIELD_2': 'id',
            'INPUT': corine_polygons_reproj_clipped,
            'INPUT_2': table,
            'METHOD': 0,
            'PREFIX': '',
            'OUTPUT': 'TEMPORARY_OUTPUT'
        }
        corine_polygons_z0 = processing.run('native:joinattributestable', alg_params, context=context, feedback=feedback)['OUTPUT']

        corine_polygons_z0_single = processing.run(
            "native:multiparttosingleparts",
            {"INPUT": corine_polygons_z0, "OUTPUT": "TEMPORARY_OUTPUT"}
            , context=context, feedback=feedback
        )["OUTPUT"]

        (sink, self.dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            corine_polygons_z0_single.fields(),
            corine_polygons_z0_single.wkbType(),
            corine_polygons_z0_single.sourceCrs(),
        )

        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / (corine_polygons_z0_single.featureCount() if corine_polygons_z0_single.featureCount() else 0)
        features = corine_polygons_z0_single.getFeatures()
        feedback.pushInfo("Number of roughness lines to write: " + str(total))
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
            feedback.setProgress(int(current * total))

        # set project options such that snapping is enabled to avoid errors when people modify the polygons
        proj = set_snapping_options()

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: self.dest_id}

    def postProcessAlgorithm(self, context, feedback):
        global renamer
        renamer = Renamer('CORINE landcover polygons')
        context.layerToLoadOnCompletionDetails(self.dest_id).setPostProcessor(renamer)
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        processed_layer = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)
        processed_layer.loadNamedStyle(os.path.join(cmd_folder, "styles", "z0_polygons.qml"))
        processed_layer.triggerRepaint()

        return {self.OUTPUT: self.dest_id}