# -*- coding: utf-8 -*-

"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""
from PyQt5.QtCore import QVariant
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsProcessingUtils,
    QgsFeatureSink,
    QgsVectorLayer,
    QgsField,
    QgsPoint,
    QgsGeometry,
    QgsFeature,
    QgsProcessingException,
    QgsProcessingAlgorithm,
    QgsProcessingParameterNumber,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterBoolean,
    QgsProcessingLayerPostProcessorInterface
)

from qgis import processing
import os, inspect

def is_clockwise(points):
    """Return the signed area enclosed by a ring using the linear time
    algorithm at http://www.cgafaq.info/wiki/Polygon_Area. A value >= 0
    indicates a counter-clockwise oriented ring."""
    xs, ys = map(list, zip(*points))
    xs.append(xs[1])
    ys.append(ys[1])
    return sum(xs[i] * (ys[i + 1] - ys[i - 1]) for i in range(1, len(points))) / 2.0 < 0


class Renamer (QgsProcessingLayerPostProcessorInterface):
    def __init__(self, layer_name):
        self.name = layer_name
        super().__init__()
        
    def postProcessLayer(self, layer, context, feedback):
        layer.setName(self.name)

class PolygonsToRoughnessLines(QgsProcessingAlgorithm):
    """
    This is algorithm convert polygons into WAsP roughness
    lines, which by definition have a left and right hand side
    roughness length z0. These are called z0_left and z0_right.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    dest_id = None
    INPUT = "INPUT"
    DOCLIP = "DOCLIP"
    OUTPUT = "OUTPUT"
    VALUE = "roughness_length_outside"

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return PolygonsToRoughnessLines()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "polygons_to_z0_lines"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Polygons to roughness lines")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("WAsP scripts")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "wasp_scripts"

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr(
            """Convert a roughness polygon layer into lines with 'z0_right' and 'z0_left'

            The input vector layer should contain an attribute 'z0' which represents a polygon with that roughness length. Each polygon is then converted to lines with 'z0_right' and 'z0_left'. 
            The second input is the 'Roughness length outside specified areas', which determines the roughness length outside the polygons that are defined in first input. 
            In addition, one can clip the outermost roughness lines by 50 m, which can be convenient when the polygons were extracted from CORINE data. In this way, WAsP can treat the roughness length in each sector as 'open-ended', which can better represent the real landscape beyond the last roughness line."""
        )

    def initAlgorithm(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Polygon data layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT, self.tr("Output layer")
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.VALUE,
                description=self.tr("Roughness length outside specified areas"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.0,
                optional=False,
                minValue=0,
                maxValue=5,
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                name=self.DOCLIP,
                description=self.tr("Clip border roughness lines"),
                defaultValue=True,
                optional=False,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)

        # If source was not found, throw an exception to indicate that the algorithm
        # encountered a fatal error. The exception text can be any string, but in this
        # case we use the pre-built invalidSourceError method to return a standard
        # helper text for when a source cannot be evaluated
        if source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT)
            )

        # Send some information to the user
        feedback.pushInfo("CRS is {}".format(source.sourceCrs().authid()))
        layer_in = self.parameterAsVectorLayer(parameters, self.INPUT, context)
        layer = processing.run(
            "native:multiparttosingleparts",
            {"INPUT": layer_in, "OUTPUT": "TEMPORARY_OUTPUT"},
            context=context,
            feedback=feedback,
        )["OUTPUT"]
        doclipping = self.parameterAsBool(parameters, self.DOCLIP, context)

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / layer.featureCount() if layer.featureCount() else 0
        features = layer.getFeatures()
        feedback.pushInfo("Number of features to convert: " + str(layer.featureCount()))

        field_name = "z0"
        segments = {}  # initialize dictionairy with segment
        j = 0
        set_z0_beyond_last = not doclipping
        z0_beyond_last = self.parameterAsDouble(parameters, self.VALUE, context)
        background_z0 = -999  # roughness value for lines where nothing was assigned.
        idx = layer.fields().indexFromName(field_name)
        if idx == -1:
            raise QgsProcessingException(
                self.tr(
                    f"The field {field_name} does not exist in layer {layer_in.name()}!"
                )
            )
        zz = 0

        id_displ = layer.fields().indexFromName("d")
        for current, feat in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # polylines
            poly = feat.geometry().asPolygon()
            if id_displ != -1:
                if feat.attributes()[id_displ] != 0.0:
                    raise QgsProcessingException(
                        self.tr(
                            f"None-zero displacement height (d) found in the input! Use the script 'Polygons to landcover lines' instead and save it as a WAsP .gml file using the 'Save WAsP landcover map'."
                        )
                    )
            bp = 0
            zz = 0
            for line in poly:
                if zz == 0:
                    # if zz is zero it is the first line, which is always the exterior line
                    if is_clockwise(line):
                        line = line[::-1]
                else:
                    # these are interior lines
                    if not is_clockwise(line):
                        line = line[::-1]   
                zz += 1
                # feedback.pushInfo(f"Feature {current}, Number of points in line {bp}: "+str(len(line)))
                for i in range(len(line) - 1):
                    lsegment = (
                        (line[i][0], line[i][1]),
                        (line[i + 1][0], line[i + 1][1]),
                    )
                    rsegment = (
                        (line[i + 1][0], line[i + 1][1]),
                        (line[i][0], line[i][1]),
                    )
                    z0id = feat.attributes()[idx]
                    if lsegment in segments:
                        if segments[lsegment]["z0_left"] == background_z0:
                            segments[lsegment]["z0_left"] = z0id
                        elif segments[lsegment]["z0_right"] == background_z0:
                            segments[lsegment]["z0_right"] = z0id
                    elif rsegment in segments:
                        if segments[rsegment]["z0_left"] == background_z0:
                            segments[rsegment]["z0_left"] = z0id
                        elif segments[rsegment]["z0_right"] == background_z0:
                            segments[rsegment]["z0_right"] = z0id
                    else:
                        segments[lsegment] = {
                            "z0_left": z0id,
                            "z0_right": background_z0,
                            "fid": j,
                        }
                    j += 1
                bp += 1
                # feedback.pushInfo(str(lsegment))
            # Update the progress bar

        newlayer = QgsVectorLayer("LineString", "z0lines", "memory")
        newlayer.setCrs(layer.crs())
        prov = newlayer.dataProvider()
        attributes = [
            QgsField("fid", QVariant.Int),
            QgsField("z0_left", QVariant.Double),
            QgsField("z0_right", QVariant.Double),
        ]
        prov.addAttributes(attributes)

        # create features
        feats = []
        newlayer.startEditing()
        bb = 0

        for segment in segments:
            # if (bb % 100) == 0:
            #    feedback.pushInfo(str(segment))
            if (
                segments[segment]["z0_left"] == -999
                or segments[segment]["z0_right"] == -999
            ):
                # if we want roughness change after lines that are on the edge of
                # our domain we overwrite the -999 values.
                if set_z0_beyond_last:
                    if segments[segment]["z0_left"] == -999:
                        segments[segment]["z0_left"] = z0_beyond_last
                    if segments[segment]["z0_right"] == -999:
                        segments[segment]["z0_right"] = z0_beyond_last
                elif doclipping:
                    if segments[segment]["z0_left"] == -999 or segments[segment]["z0_right"] == -999:
                        continue
                else:
                    continue

            if segments[segment]["z0_left"] == segments[segment]["z0_right"] and layer.featureCount() != 1:
                continue

            feat = QgsFeature()
            vertices = [
                QgsPoint(segment[0][0], segment[0][1]),
                QgsPoint(segment[1][0], segment[1][1]),
            ]
            # note we are reversing the order of the columns here
            # this is not a bug, but the adoption of lines in WAsP
            # and Qgis is aparently reversed.
            attributes = [
                segments[segment]["fid"],
                segments[segment]["z0_left"],
                segments[segment]["z0_right"],
            ]
            feat.setGeometry(QgsGeometry.fromPolyline(vertices))
            feat.setAttributes(attributes)
            feats.append(feat)
            bb += 1

        feedback.pushInfo(f"nr segments {bb}")
        prov.addFeatures(feats)
        newlayer.updateExtents()
        newlayer.commitChanges()

        clone_layer = processing.run(
            "native:dissolve",
            {
                "INPUT": newlayer,
                "FIELD": ["z0_left", "z0_right"],
                "OUTPUT": "TEMPORARY_OUTPUT",
            },
            context=context,
            feedback=feedback,
        )["OUTPUT"]

        # if doclipping:
        #     buffered_layer = processing.run(
        #         "native:buffer",
        #         {
        #             "DISSOLVE": False,
        #             "DISTANCE": -50,
        #             "END_CAP_STYLE": 2,
        #             "INPUT": dissolved_convex,
        #             "JOIN_STYLE": 0,
        #             "MITER_LIMIT": 2,
        #             "OUTPUT": "TEMPORARY_OUTPUT",
        #             "SEGMENTS": 5,
        #         },
        #         context=context,
        #         feedback=feedback,
        #     )["OUTPUT"]

        #     feedback.pushInfo("Clipping to bounding box...")
        #     clone_layer = processing.run(
        #         "native:clip",
        #         {
        #             "INPUT": clone_layer,
        #             "OUTPUT": "TEMPORARY_OUTPUT",
        #             "OVERLAY": buffered_layer,
        #         },
        #         context=context,
        #         feedback=feedback,
        #     )["OUTPUT"]

        (sink, self.dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            clone_layer.fields(),
            clone_layer.wkbType(),
            clone_layer.sourceCrs(),
        )
        
        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))

        if clone_layer.featureCount() == 0:
            raise QgsProcessingException(
                self.tr(
                    f"No features left after clipping border lines! If you are working with a single polygon, try to uncheck the 'clip border lines' and specify a 'Roughness length outside specified areas'."
                )
            )
        
        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / clone_layer.featureCount() if clone_layer.featureCount() else 0
        features = clone_layer.getFeatures()
        feedback.pushInfo("Number of roughness lines to write: " + str(total))
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
            feedback.setProgress(int(current * total))
        feedback.pushInfo(f"nr segments {current}")
        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: self.dest_id}
        
    def postProcessAlgorithm(self, context, feedback):
        # post process to give our desired style to the output object

        processed_layer = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)

        dir = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        processing.run("native:setlayerstyle", {'INPUT':processed_layer,'STYLE':os.path.join(dir,"styles", "z0_lines.qml")})

        global renamer
        renamer = Renamer('roughness_lines')
        context.layerToLoadOnCompletionDetails(self.dest_id).setPostProcessor(renamer)
        
        return {self.OUTPUT: self.dest_id}
