# -*- coding: utf-8 -*-

"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""
from PyQt5.QtCore import QVariant
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessingUtils,
    QgsProcessing,
    QgsFeatureSink,
    QgsVectorLayer,
    QgsField,
    QgsPoint,
    QgsGeometry,
    QgsFeature,
    QgsProcessingException,
    QgsProcessingAlgorithm,
    QgsProcessingParameterNumber,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterBoolean,
    QgsProcessingLayerPostProcessorInterface,
)

from qgis import processing
import os, inspect

def is_clockwise(points):
    """Return the signed area enclosed by a ring using the linear time
    algorithm at http://www.cgafaq.info/wiki/Polygon_Area. A value >= 0
    indicates a counter-clockwise oriented ring."""
    xs, ys = map(list, zip(*points))
    xs.append(xs[1])
    ys.append(ys[1])
    return sum(xs[i] * (ys[i + 1] - ys[i - 1]) for i in range(1, len(points))) / 2.0 < 0
    
class Renamer (QgsProcessingLayerPostProcessorInterface):
    def __init__(self, layer_name):
        self.name = layer_name
        super().__init__()
        
    def postProcessLayer(self, layer, context, feedback):
        layer.setName(self.name)
    
class PolygonsToLandcoverLines(QgsProcessingAlgorithm):
    """
    This is algorithm converts polygons into WAsP landcover
    lines and a lookup table for the id of each lines.
    By definition the lines have a left and right hand side
    ID. These are called id_left and id_right.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.
    dest_id = None
    table_dest_id = None
    INPUT = "INPUT"
    DOCLIP = "DOCLIP"
    OUTPUT = "OUTPUT"
    TABLE= "TABLE"
    VALUE = "roughness_length_outside"

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return PolygonsToLandcoverLines()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "polygons_to_landcover_lines"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Polygons to landcover lines")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr("WAsP scripts")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "wasp_scripts"

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr(
            """Convert a roughness polygon layer into lines with 'id_right' and 'id_left'

            The vector layer should contain a attribute 'z0' which represents the roughness length of a polygon. 
            Optionally, it can contain a displacement height 'd', which defines the displacement height of that land cover.
            Optionally, it will look for a field 'id' and it will use that field as the ID value of a polygon. Each polygon is then converted to lines with 'id_right' and 'id_left' and a corresponding roughness table. In this roughness table each ID is tied to a certain z0 and d.
            The second input is the 'Roughness length outside specified areas', which determines the roughness length outside the polygons that are defined in first input. The landcover class with this roughness length is added to the landcover table. 
            In addition, one can clip the outermost roughness lines by 50 m, which can be convenient when the polygons were extracted from a landcover database. In this way, WAsP can treat the roughness length in each sector as 'open-ended', which can better represent the real landscape beyond the last roughness line. If you have made a hand digitized map yourself, you should NOT check this box, as it will clip some of your digitized polygons."""
        )

    def initAlgorithm(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("Polygon data layer"),
                [QgsProcessing.TypeVectorPolygon],
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT, self.tr("Output layer")
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.TABLE, self.tr("landcover_table")
            )
        )
        
        self.addParameter(
            QgsProcessingParameterNumber(
                name=self.VALUE,
                description=self.tr("Roughness length outside specified areas"),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=0.0,
                optional=False,
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                name=self.DOCLIP,
                description=self.tr("Clip border roughness lines"),
                defaultValue=True,
                optional=False,
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)

        # If source was not found, throw an exception to indicate that the algorithm
        # encountered a fatal error. The exception text can be any string, but in this
        # case we use the pre-built invalidSourceError method to return a standard
        # helper text for when a source cannot be evaluated
        if source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.INPUT)
            )

        # Send some information to the user
        feedback.pushInfo("CRS is {}".format(source.sourceCrs().authid()))
        layer_in = self.parameterAsVectorLayer(parameters, self.INPUT, context)
        layer = processing.run(
            "native:multiparttosingleparts",
            {"INPUT": layer_in, "OUTPUT": "TEMPORARY_OUTPUT"},
           context=context,
           feedback=feedback,
        )["OUTPUT"]
        doclipping = self.parameterAsBool(parameters, self.DOCLIP, context)

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / layer.featureCount() if layer.featureCount() else 0
        features = layer.getFeatures()
        feedback.pushInfo("Number of features to convert: " + str(layer.featureCount()))

        field_name = "z0"
        segments = {}  # initialize dictionairy with segment
        j = 0
        z0_beyond_last = self.parameterAsDouble(parameters, self.VALUE, context)
        background_z0 = -999  # roughness value for lines where nothing was assigned.
        idx = layer.fields().indexFromName("z0")               
        if idx == -1:
            raise QgsProcessingException(
                self.tr(
                    f"The field {field_name} does not exist in layer {layer_in.name()}!"
                )
            )
        
        lct = {}
        
        # find maximum possible land cover id we can use
        newlct = {}
        for current, feat in enumerate(features):
            try:
                displ = feat["d"]
            except KeyError:
                displ = 0.0 
            try:
                desc = feat["desc"]
            except KeyError:
                desc = ''    
            z0 = feat["z0"]      
            try:
                pid = int(feat["id"])
            except KeyError:
                pid = int(feat["index"])
                feedback.pushWarning("The landcover ID's should be called 'id' but was called 'index', using this word for landcover IDs is deprecated.")
            newlct[(z0, displ)] = {"id":pid, "desc":desc}

        
        lct = {}
        newidx = [val["id"] for key, val in newlct.items()]
        if newidx:
            newidx = max(newidx) + 1
        else:
            newidx = 0
        for key, val in newlct.items():          
            if not val["id"] in lct.keys():
                lct[val["id"]] = {"z0":key[0], "d":key[1],"desc":val["desc"]}                
            else:                
                feedback.pushInfo(f"Found key {key} which already existed, which means these are probably customized polygons. New entry with ID={newidx} is created.")
                lct[newidx] = {"z0":key[0], "d":key[1], "desc":""}
                newlct[key]["id"] = newidx
                newlct[key]["desc"] = ""
                newidx = newidx + 1


        features = layer.getFeatures()
        for current, feat in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # polylines            
            poly = feat.geometry().asPolygon()
            
            # set defaults for d and desc
            try:
                displ = feat["d"]
            except KeyError:
                displ = 0.0

            index = newlct[(feat["z0"],displ)]["id"]
            zz = 0
            for line in poly:
                # we revert the line whether we loop thorugh our line clockwise or counterclockwise
                # WAsP has the opposite default storage order as corine
                line = line[::-1]
                if is_clockwise(line) and zz==0:
                    line = line[::-1]
                zz += 1
                # feedback.pushInfo(f"Feature {current}, Number of points in line {bp}: "+str(len(line)))
                for i in range(len(line) - 1):
                    lsegment = (
                        (line[i][0], line[i][1]),
                        (line[i + 1][0], line[i + 1][1]),
                    )
                    rsegment = (
                        (line[i + 1][0], line[i + 1][1]),
                        (line[i][0], line[i][1]),
                    )
                    z0id = index
                    if lsegment in segments:
                        if segments[lsegment]["id_left"] == background_z0:
                            segments[lsegment]["id_left"] = z0id
                        elif segments[lsegment]["id_right"] == background_z0:
                            segments[lsegment]["id_right"] = z0id
                    elif rsegment in segments:
                        if segments[rsegment]["id_left"] == background_z0:
                            segments[rsegment]["id_left"] = z0id
                        elif segments[rsegment]["id_right"] == background_z0:
                            segments[rsegment]["id_right"] = z0id
                    else:
                        segments[lsegment] = {
                            "id_left": z0id,
                            "id_right": background_z0,
                            "fid": j,
                        }
                    j += 1
        
        # make sure that the roughness length beyond the last line is included in the lookup table
        lct[newidx] = {"z0": z0_beyond_last, "d": 0.0, "desc": "Roughness length beyond last line"}
        feedback.pushInfo(f"Adding ID {newidx} to with background z0={z0_beyond_last}.")

        # start filling the land cover table layer
        bb = 0
        table_layer = QgsVectorLayer('None', "landcover_table", "memory")
        table_layer.setCrs(layer.crs())
        prov = table_layer.dataProvider()
        attributes = [
            QgsField("id", QVariant.Int),
            QgsField("z0", QVariant.Double),
            QgsField("d", QVariant.Double),
            QgsField("desc", QVariant.String),
        ]
        prov.addAttributes(attributes)
        feats = []
        table_layer.startEditing()
        for key, value in lct.items():
            if feedback.isCanceled():
                break
            feat = QgsFeature()
            attributes = [
                key,
                value["z0"],
                value["d"],
                value["desc"]
            ]
            feat.setAttributes(attributes)
            feats.append(feat)
            bb += 1
        feedback.pushInfo(f"{bb} table elements added")
        prov.addFeatures(feats)
        table_layer.updateExtents()
        table_layer.commitChanges()
        table_layer.setName('landcover_table')
        
        newlayer = QgsVectorLayer("LineString", "z0lines", "memory")
        newlayer.setCrs(layer.crs())
        prov = newlayer.dataProvider()
        attributes = [
            QgsField("fid", QVariant.Int),
            QgsField("id_left", QVariant.Double),
            QgsField("id_right", QVariant.Double),
        ]
        prov.addAttributes(attributes)

        # create features
        feats = []
        newlayer.startEditing()
        bb = 0
        
        for segment in segments:
            if feedback.isCanceled():
                break
            if (
                segments[segment]["id_left"] == background_z0
                or segments[segment]["id_right"] == background_z0
            ):
                # if we want roughness change after lines that are on the edge of
                # our domain we overwrite the -999 values.
                if doclipping:
                    if segments[segment]["id_left"] == background_z0 or segments[segment]["id_right"] == background_z0:
                        continue
                else:
                    if segments[segment]["id_left"] == background_z0:
                        segments[segment]["id_left"] = newidx
                    if segments[segment]["id_right"] == background_z0:
                        segments[segment]["id_right"] = newidx

            if segments[segment]["id_left"] == segments[segment]["id_right"] and layer.featureCount() != 1:
                continue

            feat = QgsFeature()
            vertices = [
                QgsPoint(segment[0][0], segment[0][1]),
                QgsPoint(segment[1][0], segment[1][1]),
            ]
            # note we are reversing the order of the columns here
            # this is not a bug, but the adoption of lines in WAsP
            # and Qgis is aparently reversed.
            attributes = [
                segments[segment]["fid"],
                segments[segment]["id_left"],
                segments[segment]["id_right"],
            ]
            feat.setGeometry(QgsGeometry.fromPolyline(vertices))
            feat.setAttributes(attributes)
            feats.append(feat)
            bb += 1

        feedback.pushInfo(f"nr segments {bb}")
        prov.addFeatures(feats)
        newlayer.updateExtents()
        newlayer.commitChanges()

        clone_layer = processing.run(
            "native:dissolve",
            {
                "INPUT": newlayer,
                "FIELD": ["id_left", "id_right"],
                "OUTPUT": "TEMPORARY_OUTPUT",
            },
            context=context,
            feedback=feedback,
        )["OUTPUT"]
        
        if clone_layer.featureCount() == 0:
            raise QgsProcessingException(
                self.tr(
                    f"No features left after clipping border lines! If you are working with a single polygon, try to uncheck the 'clip border lines' and specify a 'Roughness length outside specified areas'."
                )
            )
        
        (sink, self.dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            clone_layer.fields(),
            clone_layer.wkbType(),
            clone_layer.sourceCrs(),
        )
        
        (table_sink, self.table_dest_id) = self.parameterAsSink(
            parameters,
            self.TABLE,
            context,
            table_layer.fields(),
            table_layer.wkbType(),
            table_layer.sourceCrs(),
        )

        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))
            
        if table_sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.TABLE))

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = clone_layer.featureCount() if source.featureCount() else 0
        features = clone_layer.getFeatures()
        feedback.pushInfo("Number of roughness lines to write: " + str(total))
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)
            feedback.setProgress(int(current))
        
        total = table_layer.featureCount() if source.featureCount() else 0
        features = table_layer.getFeatures()
        feedback.pushInfo("Number of table items to write: " + str(total))
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break
            # Add a feature in the sink
            table_sink.addFeature(feature, QgsFeatureSink.FastInsert)
            feedback.setProgress(int(current))

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.      
        return {self.OUTPUT: self.dest_id, self.TABLE: self.table_dest_id}
        
    def postProcessAlgorithm(self, context, feedback):
        # post process to give our desired style to the output object

        processed_layer = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)

        dir = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        processing.run("native:setlayerstyle", {'INPUT':processed_layer,'STYLE':os.path.join(dir,"styles","id_lines.qml")})
        
        # make sure that the table has the name 'landcover_table' so that pywasp can import it
        global renamer
        renamer = Renamer('landcover_table')
        context.layerToLoadOnCompletionDetails(self.table_dest_id).setPostProcessor(renamer)
        renamer = Renamer('landcover_lines')
        context.layerToLoadOnCompletionDetails(self.dest_id).setPostProcessor(renamer)

        return {self.OUTPUT: self.dest_id, self.TABLE: self.table_dest_id}
          
