"""
Wtyczka Warszawa GIS

Wtyczka zapewniająca łatwy dostęp do danych przestrzennych
udostępnianych przez m.st. Warszawę.

    begin                : 2025-03-1
    copyright            : (C) 2025 by Maryla Jeż
    email                : maryla4gis@gmail.com
"""

import os

from qgis.PyQt.QtCore import Qt, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction

from qgis.gui import QgisInterface

from .egib.waw_egib import WawEgib
from .orto.waw_orto import WawOrto
from .rest.waw_rest import WawRest


class WarszawaGis:
    """Klasa wtyczki."""

    def __init__(self, iface: QgisInterface):
        """Konstruktor.
        
        :param iface: Interfejs aplikacji umożliwiający jego modyfikację.
        """

        self.iface = iface
        self.menu = None
        self.plugins = [
            WawOrto(iface),
            WawEgib(iface),
            WawRest(iface)
        ]
        self.toolbar = None

    def initGui(self):
        """Tworzy elementy interfejsu wtyczki i dodaje je do okna QGISa."""

        # elementy menu wtyczki
        icon = QIcon(os.path.join(os.path.dirname(__file__), 'img', 'waw.svg'))
        self.menu = self.iface.pluginMenu().addMenu(icon, 'Warszawa GIS')
        self.toolbar = self.iface.addToolBar('Warszawa GIS')

        for plugin in self.plugins:
            plugin.initGui(self.menu, self.toolbar)
            self.menu.addSeparator()
            self.toolbar.addSeparator()

        # SERWIS MAPOWY
        # https://www.flaticon.com/free-icon/web_4416517
        iconMapService = QIcon(os.path.join(os.path.dirname(__file__), 'img', 'web.png'))
        actionMapService = QAction(iconMapService, 'Serwis mapowy', self.iface.mainWindow())
        actionMapService.setStatusTip('Otwórz stronę serwisu mapowego w przeglądarce.')
        actionMapService.triggered.connect(self.goToMapService)

        self.menu.addAction(actionMapService)
        self.menu.addSeparator()
        self.toolbar.addAction(actionMapService)

        # DOKUMENTACJA
        iconInfo = QIcon(os.path.join(os.path.dirname(__file__), 'img', 'info.svg'))
        actionInfo = QAction(iconInfo, 'O wtyczce', self.iface.mainWindow())
        actionInfo.triggered.connect(self.goToDocumentation)

        self.menu.addAction(actionInfo)

    def unload(self):
        """Usuwa elementy interfejsu wtyczki z okna QGISa."""

        for plugin in self.plugins:
            plugin.unload()

        self.menu.clear()
        self.menu.deleteLater()
        self.toolbar.clear()
        self.toolbar.deleteLater()

    def goToMapService(self):
        """Otwiera stronę serwisu mapowego w przeglądarce."""

        url = QUrl("https://mapa.um.warszawa.pl/")
        QDesktopServices.openUrl(QUrl(url))

    def goToDocumentation(self):
        """Otwiera stronę dokumentacji wtyczki w przeglądarce."""

        path = os.path.join(os.path.dirname(__file__), 'docs/build/html', 'index.html')
        url = QUrl.fromLocalFile(path)
        QDesktopServices.openUrl(QUrl(url))
