"""
Moduł do tworzenia zapytania usługi REST.
"""

from collections import namedtuple
from typing import Optional

from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtNetwork import QNetworkRequest

from qgis.core import QgsPointXY


class WawRestRequest(QNetworkRequest):
    """Klasa zapytania usługi REST."""

    DISTRICTS = ('Warszawa', 'Bemowo', 'Białołęka', 'Bielany', 'Mokotów',
                 'Ochota', 'Praga-Południe', 'Praga-Północ', 'Rembertów',
                 'Śródmieście', 'Targówek', 'Ursus', 'Ursynów', 'Wawer',
                 'Wesoła', 'Wilanów', 'Włochy', 'Wola', 'Żoliborz')
    """Dostępne dzielnice (całe miasto jako pierwsze)."""

    DataLayer = namedtuple('DataLayer', ['id', 'displayName', 'allowedForCity',
                                         'allowedForDistrict', 'allowedForMapClick'])

    DATA = (
        DataLayer('BiuraUrzeduMiasta', 'Biura Urzędu m. st. Warszawy', True, True, False),
        DataLayer('GraniceDzielnic', 'Dzielnice', True, False, True),
        # DataLayer('GraniceDzialek', 'Granice działek', False, False, True),
        DataLayer('InnePomocSpoleczna', 'Inne jednostki pomocy społecznej', True, True, False),
        DataLayer('JednostkiOrganizacyjne', 'Jednostki organizacyjne m. st. Warszawy',
                  True, True, False),
        # DataLayer('StanowiskaHandlowe', 'Miejsca handlowe', True, True, False),
        DataLayer('ZasiegiPlanow', 'Miejscowe Plany Zagospodarowania Przestrzennego',
                  True, True, True),
        DataLayer('NieruchomosciNaSprzedaz', 'Nieruchomości na sprzedaż', True, True, False),
        DataLayer('Obreby', 'Obręby', False, True, False),
        DataLayer('ObszaryMSI', 'Obszary Miejskiego Systemu Informacji', True, False, True),
        # DataLayer('ObszaryMSIExt', 'Obszary Miejskiego Systemu Informacji rozszerzone',
                  # False, True, False),
        DataLayer('OsrodkiPomocySpolecznej', 'Ośrodki Pomocy Społecznej', True, True, False),
        DataLayer('ParkingiParkAndRide', 'Parkingi Park and Ride', True, False, False),
        DataLayer('PlaceZabaw', 'Place zabaw', True, True, False),
        DataLayer('PlacowkiOswiatowe', 'Placówki oświatowe (publiczne i niepubliczne)',
                  True, True, False),
        DataLayer('Pomniki', 'Pomniki', True, True, False),
        DataLayer('Promy', 'Promy', True, False, False),
        DataLayer('PunktyAdresowe', 'Punkty adresowe', False, True, False),
        # DataLayer('PunktyAdresoweExt', 'Punkty adresowe rozszerzone', False, True, False),
        DataLayer('PomocPrawna', 'Punkty pomocy prawnej', True, True, False),
        DataLayer('RoweryTowarowe', 'Rowery towarowe', True, True, False),
        DataLayer('SamoobslugoweStacjeNapraw', 'Samoobsługowe stacje napraw', True, True, False),
        # DataLayer('StrefaPlatnegoParkowania', 'Strefa Płatnego Parkowania', True, False, True),
        # DataLayer('StrefaTAXI', 'Strefy TAXI', True, False, True),
        DataLayer('Targowiska', 'Targowiska', True, True, False),
        DataLayer('TargowiskaJednodniowe', 'Targowiska jednodniowe', True, True, False),
        DataLayer('TrasyRowerowe', 'Trasy rowerowe', True, True, False),
        DataLayer('Ulice', 'Ulice i place', True, True, False),
        # DataLayer('UliceExt', 'Ulice i place rozszerzone', False, True, False),
        DataLayer('UrzedyDzielnic', 'Urzędy dzielnic m. st. Warszawy', True, True, False),
        DataLayer('UrzedyStanuCywilnego', 'Urzędy Stanu Cywilnego', True, True, False),
        DataLayer('ZlobkiKlubyDzieciece', 'Żłobki i kluby dziecięce', True, True, False)
    )
    """Dostępne warstwy danych

    namedtuple (id, wyświetlana nazwa, czy dostępne dla całego miasta,
    czy dostępne dla dzielnicy, czy dostępne poprzez kliknięcie na mapie)
    """

    def __init__(self, dataLayer: str, area: Optional[str] = None,
                 point: Optional[QgsPointXY] = None, isWgs84: bool = False):
        """Konstruktor.

        :param dataLayer: Warstwa danych, która ma zostać pobrana z usługi REST.
        :param area: Zasięg, dla którego ma zostać pobrana warstwa danych.
        :param point: Punkt określający dla jakiego obszaru mają zostać pobrane dane.
        :param isWgs84: Czy warstwa danych ma zostać pobrana w układzie współrzędnych WGS84.
        """

        self.dataLayer = dataLayer
        self.area = area
        self.point = point
        crsUrl = 'wgs84' if isWgs84 else 'pl2000'

        if self.area == 'Warszawa':
            self.url = f'http://mapa.um.warszawa.pl/WebServices/{self.dataLayer}/{crsUrl}/findAll'
        elif self.point:
            x = round(self.point.x(), 6)
            y = round(self.point.y(), 6)
            self.url = f'http://mapa.um.warszawa.pl/WebServices/' \
                       f'{self.dataLayer}/{crsUrl}/findByCoordinates/{x}/{y}'
        else:
            self.url = f'http://mapa.um.warszawa.pl/WebServices/' \
                       f'{self.dataLayer}/{crsUrl}/findByDistrict/{area}'

        super().__init__(QUrl(self.url))

    def layerName(self) -> str:
        """Nazwa docelowej warstwy."""

        if self.area == 'Warszawa' or self.point is not None:
            return f'{self.dataLayer}'
        return f'{self.dataLayer}_{self.area}'
