"""
Moduł Ortofotomapy Warszawy.
Tworzy przyciski zawierające akcje dodawania ortofotomap
i innych WMS-ów do projektu w QGISie.
"""

import os

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMenu, QToolBar, QToolButton

from qgis.core import QgsProject, QgsRasterLayer
from qgis.gui import QgisInterface


class WawOrto:
    """Klasa wtyczki."""

    ORTOFOTOMAPY = (
        'RASTER_FOTOPLAN_2021',
        'RASTER_FOTOPLAN_2020',
        'RASTER_FOTOPLAN_2019',
        'RASTER_FOTOPLAN_2018',
        'RASTER_FOTOPLAN_2017',
        'RASTER_FOTOPLAN_2016',
        'RASTER_FOTOPLAN_2015',
        'RASTER_FOTOPLAN_2014',
        'RASTER_ORTOFOTO_2013',
        'RASTER_FOTOPLAN_2012',
        'RASTER_FOTOPLAN_2011',
        'RASTER_ORTOFOTO_2010',
        'RASTER_ORTOFOTO_2008',
        'RASTER_ORTOFOTO_2005',
    )

    UKOSNE = (
        'RASTER_FOTOPLAN_2018_N',
        'RASTER_FOTOPLAN_2018_E',
        'RASTER_FOTOPLAN_2018_S',
        'RASTER_FOTOPLAN_2018_W',
    )

    HISTORYCZNE = (
        'RASTER_ORTOFOTO_1945',
        'RASTER_PLAN_1939',
        'HISTORIA_ULICE_1939',
        'RASTER_FOTOPLAN_1935',
    )

    HISTORYCZNE_GRANICE = (
        'HISTORIA_GRANICA_1992',
        'HISTORIA_GRANICA_1977',
        'HISTORIA_GRANICA_1957',
        'HISTORIA_GRANICA_1951',
        'HISTORIA_GRANICA_1939_1945',
        'HISTORIA_GRANICA_1938',
        'HISTORIA_GRANICA_1930',
        'HISTORIA_GRANICA_1916',
    )

    def __init__(self, iface: QgisInterface):
        """Konstruktor."""

        self.iface = iface
        self.pluginDir = os.path.dirname(__file__)
        self.toolButtonOrto = None
        self.toolButtonOrtoMenu = None
        self.toolButtonHistorical = None
        self.toolButtonHistoricalMenu = None

    def initGui(self, menu: QMenu, toolbar: QToolBar):
        """Tworzy akcje wczytywania ortofotomap
        i dodaje je do menu i paska narzędzi."""

        iconOrto = QIcon(
            os.path.join(self.pluginDir, 'img', 'wawa_orto.png')
        )
        actionOrto = QAction(
            iconOrto, 'Ortofotmapy Warszawy', self.iface.mainWindow()
        )

        self.toolButtonOrto = QToolButton()
        self.toolButtonOrto.setDefaultAction(actionOrto)
        self.toolButtonOrto.setIcon(iconOrto)
        self.toolButtonOrto.setMenu(QMenu())
        self.toolButtonOrto.setPopupMode(QToolButton.InstantPopup)

        self.toolButtonOrtoMenu = self.toolButtonOrto.menu()
        self.toolButtonOrtoMenu.setIcon(iconOrto)
        self.toolButtonOrtoMenu.setTitle('Ortofotmapy Warszawy')

        menu.addMenu(self.toolButtonOrtoMenu)
        toolbar.addWidget(self.toolButtonOrto)

        iconHistorical = QIcon(
            os.path.join(self.pluginDir, 'img', 'wawa_historyczna.png')
        )
        actionHistorical = QAction(
            iconHistorical, 'Warszawa historyczna', self.iface.mainWindow()
        )

        self.toolButtonHistorical = QToolButton()
        self.toolButtonHistorical.setDefaultAction(actionHistorical)
        self.toolButtonHistorical.setIcon(iconHistorical)
        self.toolButtonHistorical.setMenu(QMenu())
        self.toolButtonHistorical.setPopupMode(QToolButton.InstantPopup)

        self.toolButtonHistoricalMenu = self.toolButtonHistorical.menu()
        self.toolButtonHistoricalMenu.setIcon(iconHistorical)
        self.toolButtonHistoricalMenu.setTitle('Warszawa historyczna')

        menu.addMenu(self.toolButtonHistoricalMenu)
        toolbar.addWidget(self.toolButtonHistorical)

        for wms in self.ORTOFOTOMAPY:
            action = QAction(wms, self.iface.mainWindow())
            self.toolButtonOrtoMenu.addAction(action)
            action.triggered.connect(lambda _, wms=wms: self.addWmsLayer(wms))

        self.toolButtonOrtoMenu.addSeparator()

        for wms in self.UKOSNE:
            action = QAction(wms, self.iface.mainWindow())
            self.toolButtonOrtoMenu.addAction(action)
            action.triggered.connect(lambda _, wms=wms: self.addWmsLayer(wms))

        for wms in self.HISTORYCZNE:
            action = QAction(wms, self.iface.mainWindow())
            self.toolButtonHistoricalMenu.addAction(action)
            action.triggered.connect(lambda _, wms=wms: self.addWmsLayer(wms))

        self.toolButtonHistoricalMenu.addSeparator()

        for wms in self.HISTORYCZNE_GRANICE:
            action = QAction(wms, self.iface.mainWindow())
            self.toolButtonHistoricalMenu.addAction(action)
            action.triggered.connect(lambda _, wms=wms: self.addWmsLayer(wms))

    def unload(self):
        """Usuwa akcje."""

        self.toolButtonOrtoMenu.clear()
        self.toolButtonHistoricalMenu.clear()

    def addWmsLayer(self, wms: str):
        """Dodaje warstwę WMS do projektu.

        :param wms: Nazwa warstwy WMS, która ma zostać dodana do projektu.
        """

        url = (
            f'crs=EPSG:2178&dpiMode=7&format=image/png&layers={wms}&styles'
            '&tilePixelRatio=0&url=https://wms.um.warszawa.pl/serwis'
        )

        rlayer = QgsRasterLayer(url, wms, 'wms')
        if not rlayer.isValid():
            self.iface.messageBar().pushCritical(
                'Błąd', 'Nie udało się wczytać warstwy.'
            )
        QgsProject.instance().addMapLayer(rlayer)
