"""
Moduł EGiB - okno wtyczki.
Implementuje dokowalne okno wtyczki oraz funkcjonalność
pobierania danych dotyczących ewidencji gruntów i budynków.
"""

from json import loads
import os
import webbrowser

from qgis.PyQt import uic
from qgis.PyQt.QtCore import pyqtSlot, QUrl
from qgis.PyQt.QtGui import QHideEvent
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest
from qgis.PyQt.QtWidgets import QDockWidget, QCheckBox

from qgis.core import (
    Qgis,
    QgsApplication,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsFeature,
    QgsGeometry,
    QgsNetworkAccessManager,
    QgsPointXY,
    QgsProject,
    QgsVectorLayer,
    QgsTask,
)
from qgis.gui import QgsMapToolEmitPoint
from qgis.utils import iface

from .task import DownloadEgibTask

FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), 'waw_egib_dockwidget_base.ui')
)


class WawEgibDockWidget(QDockWidget, FORM_CLASS):
    """Okno wtyczki do pobierania danych EGiB m.st. Warszawy."""

    def __init__(self, parent=None):
        """Konstruktor."""

        super().__init__(parent)

        self.setupUi(self)

        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.crs2178 = QgsCoordinateReferenceSystem('EPSG:2178')
        self.parcelsClickLayer = None
        self.networkManager = QgsNetworkAccessManager()
        self.pointTool = QgsMapToolEmitPoint(self.canvas)
        self.project = QgsProject.instance()
        self.taskId = None
        self.taskManager = QgsApplication.taskManager()

        self.pointTool.canvasClicked.connect(self.canvasClicked)

    def hideEvent(self, event: QHideEvent):
        """Po ukryciu okna wtyczki wyłącza przycisk narzędzia mapy
        do pobierania działek za pomocą kliknięcia."""

        self.btnMapClick.setChecked(False)
        super().hideEvent(event)

    def on_btnCheckAllDistricts_clicked(self):
        """Zaznacza wszystkie pola wyboru dla dzielnic."""

        self.setCheckboxes(True)

    def on_btnClearAllDistricts_clicked(self):
        """Odznacza wszystkie pola wyboru dla dzielnic."""

        self.setCheckboxes(False)

    def setCheckboxes(self, checkState: bool):
        """Zaznacza lub odznacza wszystkie pola wyboru dla dzielnic.

        :param checkState: Definiuje, czy pola wyboru mają zostać zaznaczone, czy odznaczone.
        """

        checkboxes = self.findChildren(QCheckBox)
        for checkbox in checkboxes:
            if checkbox.objectName() not in ('Dzialki', 'Budynki'):
                checkbox.setChecked(checkState)

    @pyqtSlot()
    def on_btnDwnl_clicked(self):
        """Tworzy zadanie pobrania paczek danych z danymi EGiB
        i przekazuje je do wykonania."""

        parcels = self.Dzialki.isChecked()
        buildings = self.Budynki.isChecked()
        oneLayer = self.rbtnOneLayer.isChecked()

        districts = []
        for checkbox in self.findChildren(QCheckBox):
            if checkbox.objectName() not in ('Dzialki', 'Budynki') and checkbox.isChecked():
                districts.append(checkbox.objectName())

        if districts and (parcels or buildings):
            self.btnDwnl.setEnabled(False)

            task = DownloadEgibTask(
                description='Pobieranie danych EGiB',
                parcels=parcels,
                buildings=buildings,
                oneLayer=oneLayer,
                districts=districts,
            )

            self.taskManager.addTask(task)
            self.taskId = self.taskManager.taskId(task)
            self.taskManager.statusChanged.connect(self.enableBtnDwnl)
        elif not parcels and not buildings:
            self.iface.messageBar().pushWarning(
                'Błąd', 'Zaznacz wybrane dane EGiB do pobrania - działki lub budynki.'
            )
        elif not districts:
            self.iface.messageBar().pushWarning('Błąd', 'Zaznacz przynajmniej jedną dzielnicę.')

    def enableBtnDwnl(self, taskId: int, status: int):
        """Odblokowuje przycisk pobierania danych.

        :param taskId: Id zadania pobrania danych EGiB.
        :param status: Status zadania pobrania danych EGiB.
        """

        if taskId == self.taskId and status in (
            QgsTask.Complete,
            QgsTask.Terminated,
            QgsTask.CancelWithoutPrompt,
        ):
            self.btnDwnl.setEnabled(True)

    @pyqtSlot()
    def on_btnWWW_clicked(self):
        """Otwiera w przeglądarce stronę m.st. Warszawy, na której znajdują się
        udostępniane dane EGiB."""

        url = 'https://architektura.um.warszawa.pl/udostepniane-dane-egib'

        webbrowser.open(url)

    def on_btnMapClick_toggled(self, status: bool):
        """Włącza lub wyłącza narzędzie mapy do pobierania działek
        za pomocą kliknięcia.

        :param status: Stan przycisku - włączony lub wyłączony.
        """

        if status is True:
            self.canvas.setMapTool(self.pointTool)
            self.btnMapClick.setText(' przestań pobierać działki')
        elif status is False:
            self.canvas.unsetMapTool(self.pointTool)
            self.btnMapClick.setText(' wskaż punkt na mapie')

    def canvasClicked(self, point: QgsPointXY):
        """Pobiera punkt kliknięcia na mapie i tworzy url do pobrania danych.

        :param point: Punkt kliknięcia na mapie.
        """

        if self.project.crs() != self.crs2178:
            pointGeom = QgsGeometry.fromPointXY(point)
            tr = self.transformFromProjectCrs()
            pointGeom.transform(tr)
            point = pointGeom.asPoint()

        x = round(point.x(), 6)
        y = round(point.y(), 6)

        url = (
            'http://mapa.um.warszawa.pl/WebServices/GraniceDzialek/'
            f'pl2000/findByCoordinates/{x}/{y}'
        )

        parcelData = self.fetchParcelData(url)
        if parcelData:
            self.addParcel(parcelData)

    def transformFromProjectCrs(self) -> QgsCoordinateTransform:
        """Tworzy transformer z układu współrzędnych projektu
        do układu 2178.

        :returns: Transformer projekt -> 2178.
        """

        srcCrs = self.project.crs()
        dstCrs = self.crs2178

        tr = QgsCoordinateTransform(srcCrs, dstCrs, self.project)

        return tr

    def transformToProjectCrs(self) -> QgsCoordinateTransform:
        """Tworzy transformer z układu 2178
        do układu współrzędnych projektu.

        :returns: Transformer 2178 -> projekt.
        """

        srcCrs = self.crs2178
        dstCrs = self.project.crs()

        tr = QgsCoordinateTransform(srcCrs, dstCrs, self.project)

        return tr

    def on_nrObreb_returnPressed(self):
        """Ustawia pole wpisywania numeru działki jako aktywne."""

        self.nrDzialka.setFocus(True)

    def on_nrDzialka_returnPressed(self):
        """Przechodzi do pobierania działki na podstawie numeru obrębu i działki."""

        self.on_btnDwnlParcelByObrAndDz_clicked()

    @pyqtSlot()
    def on_btnDwnlParcelByObrAndDz_clicked(self):
        """Tworzy url do pobrania działki na podstawie numeru obrębu i działki."""

        nrObr = self.nrObreb.text()
        nrDz = self.nrDzialka.text().replace('/', '_')

        if not nrObr and not nrDz:
            self.iface.messageBar().pushWarning(
                'Uwaga', 'Najpierw wpisz numer obrębu i numer działki.'
            )
        elif not nrObr.isnumeric() or not nrDz.replace('_', '').isnumeric():
            self.iface.messageBar().pushWarning(
                'Uwaga', 'Numer obrębu lub działki powinien zawierać cyfry.'
            )
        else:
            url = (
                'http://mapa.um.warszawa.pl/WebServices/GraniceDzialek/'
                f'pl2000/findByNrObrAndNrDz/{nrObr}/{nrDz}'
            )

            parcelData = self.fetchParcelData(url)
            if parcelData:
                self.addParcel(parcelData)

    def fetchParcelData(self, url: str) -> dict:
        """Pobiera dane o działce z usługi REST i przekształca na słownik danych.

        :param url: Adres usługi pobierania danych o działce.
        :returns: Dane o działce ewidencyjnej.
        """

        request = QNetworkRequest(QUrl(url))
        reply = self.networkManager.blockingGet(request)
        content = reply.content()
        content = content.data().decode('utf-8')
        content = loads(content)

        if reply.error() == QNetworkReply.NetworkError.NoError:
            # Qt 0, HTTP 200
            return content

        if reply.error() == QNetworkReply.NetworkError.ProtocolInvalidOperationError:
            # Qt 302, HTTP 400
            self.iface.messageBar().pushMessage(
                'Błąd',
                content['message'],
                Qgis.MessageLevel.Critical,
                duration=5
            )
        else:
            self.iface.messageBar().pushCritical('Błąd', 'Nie udało się pobrać danych.')

    def addParcel(self, parcelData: dict):
        """Przetwarza dane o działce ewidencyjnej - tworzy warstwę pobranych działek
        i zapisuje na niej obiekty.

        :param parcelData: Dane o działce ewidencyjnej.
        """

        try:
            self.parcelsClickLayer = self.project.mapLayersByName('wybrane_dzialki_warszawa')[0]
        except IndexError:
            self.parcelsClickLayer = QgsVectorLayer(
                'Polygon?crs=EPSG:2178&'
                'field=objectid:integer(10,0)&field=nrObr:string(10,0)&field=nrDz:string(10,0)',
                'wybrane_dzialki_warszawa',
                'memory',
            )
            self.project.addMapLayer(self.parcelsClickLayer)

        # zapytanie usługi REST luźno dopasowuje znalezione numery działek;
        # w przypadku zwrócenia kilku wyników
        # wybierz ten z numerem wpisanym w oknie wtyczki
        type_ = parcelData['type']
        if type_ == 'FeatureCollection':
            for feature in parcelData['features']:
                if feature['properties']['nrDz'] == self.nrDzialka.text():
                    parcelData = feature

        properties = parcelData['properties']
        objectid = properties['objectid']
        nrDz = properties['nrDz']
        nrObr = properties['nrObr']
        coords = parcelData['geometry']['coordinates'][0]

        parcelVertices = [QgsPointXY(point[0], point[1]) for point in coords]
        parcelGeom = QgsGeometry.fromPolygonXY([parcelVertices])

        parcelFeature = QgsFeature()
        parcelFeature.setAttributes([objectid, nrObr, nrDz])
        parcelFeature.setGeometry(parcelGeom)

        self.parcelsClickLayer.startEditing()
        self.parcelsClickLayer.addFeature(parcelFeature)
        saveLayer = self.parcelsClickLayer.commitChanges()
        if not saveLayer:
            self.iface.messageBar().pushCritical(
                'Błąd', 'Dane nie zostały poprawnie zapisane na warstwie.'
            )
        else:
            if self.project.crs() != self.crs2178:
                tr = self.transformToProjectCrs()
                parcelGeom.transform(tr)

            self.canvas.setExtent(parcelGeom.boundingBox())
