"""
Moduł EGiB - zadanie.
Implementuje zadanie pobrania danych EGiB w tle.
"""

import os
from shutil import rmtree
from tempfile import mkdtemp, NamedTemporaryFile
from zipfile import ZipFile

from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest

from qgis import processing
from qgis.core import (
    Qgis,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsGeometry,
    QgsMessageLog,
    QgsNetworkAccessManager,
    QgsProject,
    QgsRectangle,
    QgsTask,
)
from qgis.utils import iface

MESSAGE_CATEGORY = 'Pobieranie EGiB Warszawa'


class DownloadEgibTask(QgsTask):
    """Zadanie pobrania danych wektorowych o działkach i budynkach m.st. Warszawy."""

    def __init__(self, description, parcels, buildings, oneLayer, districts):
        """Konstruktor."""

        super().__init__(description, QgsTask.CanCancel)

        self.parcels = parcels
        self.buildings = buildings
        self.oneLayer = oneLayer
        self.districts = districts

        self.canvas = iface.mapCanvas()
        self.exception = None
        self.iface = iface
        self.networkManager = QgsNetworkAccessManager()
        self.project = QgsProject.instance()
        self.tempFolder = None

    def downloadAndExtractEgib(self, url: str, egibData: str, district: str):
        """Pobiera i rozpakowuje dane EGiB.

        :param url: Adres pobierania danych EGiB.
        :param egibData: Rodzaj pobieranych danych EGiB.
        :param district: Nazwa dzielnicy.
        """

        request = QNetworkRequest(QUrl(url.format(egibData, district)))
        reply = self.networkManager.blockingGet(request)

        if reply.error() == QNetworkReply.NetworkError.NoError:
            # Qt 0, HTTP 200
            content = reply.content()

            with NamedTemporaryFile(suffix='.zip', prefix=f'{egibData}_{district}_') as tempFile:
                tempFile.write(content)

                with ZipFile(tempFile, 'r') as zipObject:
                    zipObject.extractall(self.tempFolder)

        elif reply.error() == QNetworkReply.NetworkError.ServiceUnavailableError:
            # Qt 403, HTTP 503
            QgsMessageLog.logMessage(
                "Błąd. Serwer jest tymczasowo niedostępny.",
                MESSAGE_CATEGORY,
                Qgis.Warning
            )
        elif reply.error() == QNetworkReply.NetworkError.ProtocolInvalidOperationError:
            # Qt 302, HTTP 400
            QgsMessageLog.logMessage(
                "Błąd. Nie odnaleziono strony.",
                MESSAGE_CATEGORY,
                Qgis.Warning
            )
        elif reply.error() == QNetworkReply.NetworkError.ContentNotFoundError:
            # Qt 203, HTTP 404
            QgsMessageLog.logMessage(
                "Błąd. Nie znaleziono danych.",
                MESSAGE_CATEGORY,
                Qgis.Warning
            )
        else:
            QgsMessageLog.logMessage(
                "Wystąpił nieoczekiwany błąd.",
                MESSAGE_CATEGORY,
                Qgis.Warning
            )

    def addMergedShpLayers(self, egibData: str):
        """Łączy dane EGiB w jedną warstwę nadając jej układ współrzędnych
        i dodaje ją do projektu.

        :param egibData: Rodzaj danych EGiB.
        """

        LAYERS = []

        for district in self.districts:
            parcelsShpPath = os.path.join(self.tempFolder, f'{egibData}_{district}.shp')
            LAYERS.append(parcelsShpPath)

        mergedLayer = processing.run(
            "native:mergevectorlayers",
            {'LAYERS': LAYERS,
            'CRS': QgsCoordinateReferenceSystem('EPSG:2178'),
            'OUTPUT': 'TEMPORARY_OUTPUT'}
        )['OUTPUT']

        cleanLayer = processing.run(
            "native:deletecolumn",
            {'INPUT': mergedLayer,
            'COLUMN': ['layer', 'path'],
            'OUTPUT': 'TEMPORARY_OUTPUT'}
        )['OUTPUT']

        cleanLayer.setName(f'{egibData}_Warszawa')
        self.project.addMapLayer(cleanLayer)

        cleanLayerExtent = cleanLayer.extent()

        if self.project.crs().authid() != 'EPSG:2178':
            cleanLayerExtent = self.transformExtent(cleanLayerExtent)

        self.canvas.setExtent(cleanLayerExtent)

    def addSeparateShpLayers(self):
        """Dodaje pojedyncze warstwy danych EGiB do projektu,
        nadając im układ współrzędnych."""

        allShpLayersExtent = QgsRectangle()

        filesList = os.listdir(self.tempFolder)
        filesList = [file for file in filesList if file.endswith('.shp')]
        filesList.sort(reverse=True)

        for file in filesList:
            layerWithCrs = processing.run(
                "native:assignprojection",
                {'INPUT': os.path.join(self.tempFolder, file),
                'CRS': QgsCoordinateReferenceSystem('EPSG:2178'),
                'OUTPUT': 'TEMPORARY_OUTPUT'}
            )['OUTPUT']

            layerWithCrs.setName(file[:-4])
            self.project.addMapLayer(layerWithCrs)
            allShpLayersExtent.combineExtentWith(layerWithCrs.extent())

        if self.project.crs().authid() != 'EPSG:2178':
            allShpLayersExtent = self.transformExtent(allShpLayersExtent)

        self.canvas.setExtent(allShpLayersExtent)

    def transformExtent(self, extent: QgsRectangle) -> QgsRectangle:
        """Przekształca geometrię zasięgu warstw dodawanych do projektu
        z układu EPSG:2178 do układu projektu w celu poprawnego
        ustawienia zasięgu mapy.

        :param extent: Zasięg pobranych warstw EGiB w układzie EPSG:2178.
        :returns: Zasięg pobranych warstw EGiB w układzie projektu.
        """

        extent = QgsGeometry.fromRect(extent)

        tr = QgsCoordinateTransform(
            QgsCoordinateReferenceSystem('EPSG:2178'),
            self.project.crs(),
            self.project
        )

        extent.transform(tr)

        return extent.boundingBox()

    def run(self) -> bool:
        """Pobiera dane EGiB, tworzy warstwę/warstwy wektorowe i wczytuje je do projektu.

        :returns: Zwraca wartość logiczną prawda, gdy zadanie zakończyło się pomyślnie
            lub fałsz, gdy zadanie nie zostało ukończone.
        """

        QgsMessageLog.logMessage(
            'Rozpoczęto zadanie "{}"'.format(self.description()),
            MESSAGE_CATEGORY,
            Qgis.Info
        )

        self.tempFolder = mkdtemp(prefix='egib_warszawa_')

        url = 'https://mapa.um.warszawa.pl/BGIK/files/egib/{}_{}.zip'

        # pobieranie plików źródłowych
        for district in self.districts:
            if self.parcels:
                self.downloadAndExtractEgib(url, 'Dzialki', district)
            if self.buildings:
                self.downloadAndExtractEgib(url, 'Budynki', district)

        # dodanie danych egib w postaci połączonego shp
        if self.oneLayer:
            if self.parcels:
                self.addMergedShpLayers('Dzialki')
            if self.buildings:
                self.addMergedShpLayers('Budynki')

        # dodanie danych egib w postaci pojedynczych warstw shp
        if not self.oneLayer:
            self.addSeparateShpLayers()

        if self.isCanceled():
            return False

        return True

    def finished(self, result):
        """Informuje o statusie ukończenia zadania."""

        if result:
            QgsMessageLog.logMessage(
                'Ukończono zadanie "{name}".'.format(name=self.description()),
                MESSAGE_CATEGORY,
                Qgis.Success
            )
            rmtree(self.tempFolder)
        else:
            if self.exception is None:
                QgsMessageLog.logMessage(
                    '"{name}" nie zostało zakończone. Brak komunikatu błędu. '
                    'Prawdopodobnie zostało anulowane przez użytkownika.'.format(
                        name=self.description()
                    ),
                    MESSAGE_CATEGORY,
                    Qgis.Warning
                )
            else:
                QgsMessageLog.logMessage(
                    '"{name}" Wystąpił błąd: {exception}.'.format(
                        name=self.description(),
                        exception=self.exception
                    ),
                    MESSAGE_CATEGORY,
                    Qgis.Critical
                )
                raise self.exception

    def cancel(self):
        """Anulowanie zadania."""

        QgsMessageLog.logMessage(
            'Zadanie "{name}" zostało anulowane.'.format(
                name=self.description()
            ),
            MESSAGE_CATEGORY,
            Qgis.Info
        )
        super().cancel()
