"""
Moduł do pobierania danych wektorowych m.st. Warszawy.
Implementuje dokowalne okno wtyczki oraz funkcjonalność
pobierania danych wektorowych udostępnianych poprzez usługę REST.
"""

import os.path
from json import loads
from typing import Optional

from qgis.PyQt import uic
from qgis.PyQt.QtCore import pyqtSlot
from qgis.PyQt.QtNetwork import QNetworkReply
from qgis.PyQt.QtWidgets import QDockWidget, QWidget

from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsGeometry,
    QgsNetworkAccessManager,
    QgsPointXY,
    QgsProject,
    QgsVectorLayer,
)
from qgis.gui import QgsMapToolEmitPoint
from qgis.utils import iface

from .waw_rest_request import WawRestRequest

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'waw_rest_dockwidget_base.ui'))


class WawRestDockWidget(QDockWidget, FORM_CLASS):
    """Okno wtyczki."""

    def __init__(self, parent: Optional[QWidget] = None):
        """Konstruktor."""

        super().__init__(parent)
        self.setupUi(self)
        self.cboxArea.addItems(WawRestRequest.DISTRICTS)
        for layer in WawRestRequest.DATA:
            if layer.allowedForCity or layer.allowedForDistrict:
                self.cboxDataByArea.addItem(layer.displayName, layer.id)
            if layer.allowedForMapClick:
                self.cboxDataByCoords.addItem(layer.displayName, layer.id)

        self.canvas = iface.mapCanvas()
        self.iface = iface
        self.project = QgsProject.instance()
        self.pointTool = QgsMapToolEmitPoint(self.canvas)
        self.pointTool.canvasClicked.connect(self.canvasClicked)

    def isWgs84(self) -> bool:
        """Sprawdza czy został wybrany układ współrzędnych WGS84."""

        return not self.rbtnPl2000.isChecked()

    @pyqtSlot()
    def on_btnDwnl_clicked(self):
        """Pobiera dane w zasięgu miasta lub dzielnicy."""

        dataLayer = self.cboxDataByArea.currentData()
        area = self.cboxArea.currentText()
        request = WawRestRequest(dataLayer, area, None, self.isWgs84())
        self.addLayerFromRest(request)

    @pyqtSlot(str)
    def on_cboxDataByArea_currentTextChanged(self, newText: str):
        """Ustala ograniczenia pobierania danych bazując na dostępnym zasięgu danych."""

        # pobierz nazwę wybranej warstwy danych
        for layer in WawRestRequest.DATA:
            if layer.displayName == newText:
                dataLayer = layer

        # wyłącz opcje wyboru dzielnic z listy
        if dataLayer.allowedForCity and not dataLayer.allowedForDistrict:
            self.iface.messageBar().pushMessage(
                'Uwaga',
                'Wybrany zestaw danych można pobrać tylko dla całego miasta.',
                duration=5,
            )
            self.toggleQcboxItems(city=True, districts=False)
            self.cboxArea.setCurrentText('Warszawa')
        # wyłącz opcję wyboru całego miasta z listy
        elif not dataLayer.allowedForCity and dataLayer.allowedForDistrict:
            self.iface.messageBar().pushMessage(
                'Uwaga',
                'Wybrany zestaw danych można pobrać tylko w podziale na dzielnice.',
                duration=5,
            )
            self.toggleQcboxItems(city=False, districts=True)
        # włącz wszystkie opcje wyboru na liście
        else:
            for i in range(self.cboxArea.count()):
                self.cboxArea.model().item(i).setEnabled(True)

    def toggleQcboxItems(self, city: bool, districts: bool):
        """Włącza lub wyłącza opcje wyboru zasięgów przestrzennych
        pobieranych danych bazując na ograniczeniach usługi REST.

        :param city: Określa czy włączyć, czy wyłączyć opcję wyboru dla miasta.
        :param districts: Określa czy włączyć, czy wyłączyć opcje wyboru dla dzielnic.
        """

        idx = self.cboxArea.findText('Warszawa')
        self.cboxArea.model().item(idx).setEnabled(city)

        for district in WawRestRequest.DISTRICTS[1:]:
            idx = self.cboxArea.findText(district)
            self.cboxArea.model().item(idx).setEnabled(districts)

    def on_btnCoords_clicked(self):
        """Tworzy narzędzie mapy do kliknięcia na mapie."""

        self.btnCoords.setChecked(True)
        self.canvas.setMapTool(self.pointTool)

    def canvasClicked(self, point: QgsPointXY):
        """Przetwarza parametry zapytania bazując na miejscu kliknięcia
        na mapie oraz układzie współrzędnych.

        :param point: Punkt kliknięcia na mapie.
        """

        self.btnCoords.setChecked(False)
        self.canvas.unsetMapTool(self.pointTool)

        avaliableCrs = ('EPSG:2178', 'EPSG:4326')  # crs dostępny w usłudze REST
        projectCrsId = self.project.crs().authid()
        crsEpsg = 'EPSG:2178' if self.rbtnPl2000.isChecked() else 'EPSG:4326'
        # przekształć współrzędne punktu, aby poprawnie pobrać dane
        if projectCrsId not in avaliableCrs or projectCrsId != crsEpsg:
            point = self.transformPointCoords(point, crsEpsg)

        dataLayer = self.cboxDataByCoords.currentData()

        request = WawRestRequest(dataLayer, None, point, self.isWgs84())
        self.addLayerFromRest(request)

    def transformPointCoords(self, point: QgsPointXY,
                             crsEpsg: str) -> QgsPointXY:
        """Przekształca współrzędne punktu klikniętego na mapie, gdy układ
        współrzędnych projektu jest inny niż układ dostępny w usłudze REST
        lub gdy układ współrzędnych jest inny niż wybrany w oknie wtyczki.

        :param point: Punkt kliknięcia na mapie.
        :param crsEpsg: Kod EPSG układu współrzędnych wybranego w oknie wtyczki.
        :returns: Punkt przekształcony do układu współrzędnych wybranego w oknie wtyczki.
        """

        pointGeom = QgsGeometry.fromPointXY(point)
        tr = QgsCoordinateTransform(self.project.crs(),
                                    QgsCoordinateReferenceSystem(crsEpsg),
                                    self.project)
        pointGeom.transform(tr)

        return pointGeom.asPoint()

    def addLayerFromRest(self, request: WawRestRequest):
        """Pobiera dane z usługi REST, a następnie tworzy warstwę i dodaje
        ją do projektu.

        :param request: Obiekt WawRestRequest (REST url).
        """

        self.txtUrl.setText(request.url)

        networkManager = QgsNetworkAccessManager().instance()
        reply = networkManager.blockingGet(request)
        content = reply.content()
        content = content.data().decode('utf-8')

        if reply.error() == QNetworkReply.NetworkError.NoError:  # Qt 0, HTTP 200
            layer = QgsVectorLayer(content, request.layerName(), 'ogr')
            crsEpsg = 'EPSG:4326' if self.isWgs84() else 'EPSG:2178'
            layer.setCrs(QgsCoordinateReferenceSystem(crsEpsg))
            self.project.addMapLayer(layer)

            if layer.crs() != self.project.crs():
                self.zoomToLayer(layer)
            else:
                self.canvas.setExtent(layer.extent())
        elif reply.error() == QNetworkReply.NetworkError.ProtocolInvalidOperationError:
            # Qt 302, HTTP 400
            error = loads(content)
            self.iface.messageBar().pushCritical('Błąd', f'{error["message"]}')
        elif reply.error() == QNetworkReply.NetworkError.ContentNotFoundError:  # Qt 203, HTTP 404
            self.iface.messageBar().pushCritical('Błąd', 'Nie znaleziono danych.')
        else:
            self.iface.messageBar().pushCritical('Błąd', 'Wystąpił nieoczekiwany błąd.')

    def zoomToLayer(self, layer: QgsVectorLayer):
        """Przybliża do właściwego zasięgu na mapie, przekształcając zasięg
        w razie konieczności.

        :param layer: Warstwa wektorowa utworzona na podstawie odpowiedzi usługi REST.
        """

        layerExtentGeom = QgsGeometry.fromRect(layer.extent())
        tr = QgsCoordinateTransform(layer.crs(), self.project.crs(),
                                    self.project)
        layerExtentGeom.transform(tr)
        bbox = layerExtentGeom.boundingBox()

        self.canvas.setExtent(bbox)
