# -*- coding: utf-8 -*-
"""
/***************************************************************************
 WalkPotentialAlgorithm
                                 A QGIS plugin
 Processing algorithm for walk potential analysis
                              -------------------
        begin                : 2025-10-13
        copyright            : (C) 2025 by Mark Stosberg
        email                : mark@stosberg.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterString,
    QgsProcessingParameterNumber,
    QgsProcessingParameterFile,
    QgsProcessingParameterDefinition,
    QgsProcessingException,
    QgsProcessingUtils,
    QgsFeatureSink,
    QgsProject,
    QgsFeatureRequest,
    QgsGraduatedSymbolRenderer,
    QgsRendererRange,
    QgsSymbol,
    QgsWkbTypes
)
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QColor

from .walk_potential_calculator import WalkPotentialCalculator


class WalkPotentialAlgorithm(QgsProcessingAlgorithm):
    """
    Processing algorithm for calculating walk potential.
    """

    def __init__(self):
        super().__init__()
        self._last_output_id = None

    # Parameter names
    INPUT_BOUNDARY = 'INPUT_BOUNDARY'
    ISOCHRONE_URL = 'ISOCHRONE_URL'
    CONCURRENCY = 'CONCURRENCY'
    AMENITIES_FILE = 'AMENITIES_FILE'
    OUTPUT = 'OUTPUT'

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        """
        Returns a new instance of the algorithm.
        """
        return WalkPotentialAlgorithm()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm.
        """
        return 'walkpotential'

    def displayName(self):
        """
        Returns the translated algorithm name.
        """
        return self.tr('Walk Potential Analysis')

    def group(self):
        """
        Returns the name of the group this algorithm belongs to.
        """
        return ''

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to.
        """
        return ''

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm.
        """
        return self.tr(
            "Analyzes pedestrian access to amenities (shops, parks, restaurants, etc.) "
            "within a boundary area.\n\n"
            "The analysis generates a hexagonal grid over the input boundary and calculates "
            "a walk potential score (0-100) for each cell based on how many amenity types "
            "are within a 10-minute walk.\n\n"
            "Parameters:\n"
            "- Boundary Layer: Polygon layer defining the analysis area\n"
            "- Isochrone URL: Service URL for calculating walk times (use {{lat}} and {{lon}} as placeholders)\n"
            "- Concurrency: Number of simultaneous requests (8-16 for local servers, 2-4 for public APIs)\n"
            "- Amenities File: Optional custom JSON file defining amenity queries (leave blank for defaults)\n\n"
            "Note: This analysis queries OpenStreetMap data and may take several minutes "
            "depending on the size of the boundary area."
        )

    def initAlgorithm(self, config=None):
        """
        Define the inputs and outputs of the algorithm.
        """
        # Input boundary layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_BOUNDARY,
                self.tr('Boundary Layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        # Isochrone service URL
        default_url = 'http://valhalla1.openstreetmap.de/isochrone?json={"polygons":true,"locations":[{"lat":{{lat}}, "lon":{{lon}}}],"costing":"pedestrian","contours":[{"time":10.0}]}'
        self.addParameter(
            QgsProcessingParameterString(
                self.ISOCHRONE_URL,
                self.tr('Isochrone Service URL (use {{lat}} and {{lon}} as placeholders)'),
                defaultValue=default_url
            )
        )

        # Concurrency setting
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CONCURRENCY,
                self.tr('Concurrency (simultaneous requests)'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=4,
                minValue=1,
                maxValue=64
            )
        )

        # Optional custom amenities file
        self.addParameter(
            QgsProcessingParameterFile(
                self.AMENITIES_FILE,
                self.tr('Custom Amenities File'),
                behavior=QgsProcessingParameterFile.File,
                fileFilter='JSON files (*.json)',
                optional=True
            )
        )

        # Output layer
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Walk Potential')
            )
        )

    def flags(self):
        """
        Returns the flags for the algorithm.
        """
        return super().flags() | QgsProcessingAlgorithm.FlagNoThreading
    
    def processAlgorithm(self, parameters, context, feedback):
        """
        Execute the algorithm.
        """
        # Get parameters
        boundary_source = self.parameterAsSource(parameters, self.INPUT_BOUNDARY, context)
        if boundary_source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT_BOUNDARY))

        isochrone_url = self.parameterAsString(parameters, self.ISOCHRONE_URL, context)
        concurrency = self.parameterAsInt(parameters, self.CONCURRENCY, context)
        amenities_file = self.parameterAsFile(parameters, self.AMENITIES_FILE, context)

        # Convert source to layer
        # We need to create a temporary layer from the source
        from qgis.core import QgsVectorLayer
        boundary_layer = boundary_source.materialize(
            QgsFeatureRequest(),
            feedback
        )
        
        if boundary_layer is None:
            raise QgsProcessingException('Could not materialize boundary layer')

        # Progress callback
        def progress_callback(message):
            if feedback.isCanceled():
                raise QgsProcessingException('Calculation cancelled by user')
            feedback.pushInfo(message)

        # Create calculator
        try:
            # Handle empty amenities file - use None to trigger default
            if not amenities_file or not amenities_file.strip():
                amenities_file = None
                
            calculator = WalkPotentialCalculator(
                isochrone_url=isochrone_url,
                concurrency=concurrency,
                amenities_file=amenities_file,
                progress_callback=progress_callback
            )
            
            # Run calculation
            feedback.pushInfo('Starting walk potential calculation...')
            result_layer = calculator.calculate(boundary_layer)
            
            if feedback.isCanceled():
                raise QgsProcessingException('Calculation cancelled by user')
            
            # Add result to output sink
            (sink, dest_id) = self.parameterAsSink(
                parameters,
                self.OUTPUT,
                context,
                result_layer.fields(),
                result_layer.wkbType(),
                result_layer.crs()
            )
            
            # Keep track of the output layer id for styling later
            self._last_output_id = dest_id
            
            if sink is None:
                raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))
            
            # Copy features from result layer to sink
            for feature in result_layer.getFeatures():
                if feedback.isCanceled():
                    break
                sink.addFeature(feature, QgsFeatureSink.FastInsert)
            
            # Apply styling before completing
            # Need to add the layer to the project first to style it
            output_layer = context.getMapLayer(dest_id)
            if not output_layer:
                # Try getting it from project
                output_layer = QgsProject.instance().mapLayer(dest_id)
            
            if output_layer:
                feedback.pushInfo('Applying styling to output layer...')
                self._apply_graduated_style(output_layer)
                output_layer.triggerRepaint()
                feedback.pushInfo('Styling applied')
            else:
                feedback.pushInfo(f'Debug: dest_id={dest_id}, could not retrieve layer for styling')
            
            feedback.pushInfo('Walk potential calculation complete!')
            
            # Store the dest_id for post-processing
            results = {self.OUTPUT: dest_id}
            return results
            
        except Exception as e:
            raise QgsProcessingException(str(e))
    
    def postProcessAlgorithm(self, context, feedback):
        """
        Post-process the algorithm to apply styling to the output layer.
        """
        layer = None
        # Try to fetch the layer by the recorded output id
        if self._last_output_id:
            layer = context.getMapLayer(self._last_output_id) or QgsProject.instance().mapLayer(self._last_output_id)
        
        if layer and layer.isValid() and layer.type() == layer.VectorLayer:
            self._apply_graduated_style(layer)
            layer.triggerRepaint()
            feedback.pushInfo('Applied graduated styling to output layer')
        else:
            feedback.pushInfo('Warning: Could not locate output layer to apply styling')
        
        return {}
    
    def _apply_graduated_style(self, layer):
        """
        Apply graduated color styling to the walk potential layer.
        
        :param layer: The layer to style
        """
        # Define 5 ranges with colors from red (low) to green (high)
        ranges = [
            (0, 20, '#d73027', 'Very Low'),      # Red
            (20, 40, '#fc8d59', 'Low'),          # Orange
            (40, 60, '#fee08b', 'Medium'),       # Yellow
            (60, 80, '#91cf60', 'High'),         # Light green
            (80, 100, '#1a9850', 'Very High')    # Dark green
        ]
        
        # Create renderer ranges
        renderer_ranges = []
        for lower, upper, color, label in ranges:
            symbol = QgsSymbol.defaultSymbol(layer.geometryType())
            symbol.setColor(QColor(color))
            symbol.setOpacity(0.7)
            
            # Remove outline for cleaner look
            symbol.symbolLayer(0).setStrokeStyle(Qt.NoPen)
            
            renderer_range = QgsRendererRange(
                lower, upper, symbol, label
            )
            renderer_ranges.append(renderer_range)
        
        # Create and apply graduated renderer
        renderer = QgsGraduatedSymbolRenderer('walkPotential', renderer_ranges)
        renderer.setMode(QgsGraduatedSymbolRenderer.Custom)
        layer.setRenderer(renderer)
