# -*- coding: utf-8 -*-
"""
Geometry utilities for walk potential calculations using QGIS geometry functions
"""

from qgis.core import (
    QgsGeometry, QgsFeature, QgsVectorLayer, QgsField, QgsFields,
    QgsPointXY, QgsRectangle, QgsWkbTypes, QgsCoordinateReferenceSystem,
    QgsCoordinateTransform, QgsProject, QgsJsonUtils
)
from qgis.PyQt.QtCore import QVariant
from typing import List, Dict, Tuple, Optional
import math
import json


def create_hex_grid(boundary: QgsGeometry, hex_side_km: float = 0.05) -> List[QgsFeature]:
    """
    Create hexagonal grid using QGIS processing algorithm in UTM projection
    
    :param boundary: QgsGeometry defining the boundary (assumed WGS84)
    :param hex_side_km: Side length of hexagons in kilometers
    :return: List of QgsFeature objects representing hex cells
    """
    import processing
    from qgis.core import QgsProcessingFeedback
    
    # Define source and target CRS
    source_crs = QgsCoordinateReferenceSystem('EPSG:4326')  # WGS84
    
    # Calculate appropriate UTM zone from boundary centroid
    centroid = boundary.centroid().asPoint()
    lon = centroid.x()
    lat = centroid.y()
    
    # Determine UTM zone
    utm_zone = int((lon + 180) / 6) + 1
    hemisphere = 'north' if lat >= 0 else 'south'
    
    # EPSG code for UTM zones: 326XX for north, 327XX for south (XX = zone number)
    epsg_code = 32600 + utm_zone if hemisphere == 'north' else 32700 + utm_zone
    utm_crs = QgsCoordinateReferenceSystem(f'EPSG:{epsg_code}')
    
    # Transform boundary to UTM
    transform_to_utm = QgsCoordinateTransform(source_crs, utm_crs, QgsProject.instance())
    boundary_utm = QgsGeometry(boundary)
    boundary_utm.transform(transform_to_utm)
    
    # Get bounding box in UTM coordinates
    bbox = boundary_utm.boundingBox()
    extent_str = f"{bbox.xMinimum()},{bbox.xMaximum()},{bbox.yMinimum()},{bbox.yMaximum()}"
    
    # Convert km to meters for UTM
    hex_side_m = hex_side_km * 1000.0
    spacing = hex_side_m * 2
    
    # Calculate extent dimensions to validate spacing
    extent_width = bbox.xMaximum() - bbox.xMinimum()
    extent_height = bbox.yMaximum() - bbox.yMinimum()
    
    # QGIS 3.44+ requires spacing < extent dimensions
    # If the extent is too small, return empty list rather than crash
    min_extent = min(extent_width, extent_height)
    if spacing >= min_extent:
        import sys
        print(f"WARNING: Boundary too small for hex grid. Extent: {min_extent:.1f}m, Required: >{spacing:.1f}m. Skipping grid creation.", file=sys.stderr)
        return []
    
    # Use QGIS processing to create hex grid in UTM
    try:
        result = processing.run("native:creategrid", {
            'TYPE': 4,  # Hexagon
            'EXTENT': extent_str,
            'HSPACING': spacing,  # Horizontal spacing in meters
            'VSPACING': spacing,  # Vertical spacing in meters
            'CRS': utm_crs.authid(),
            'OUTPUT': 'memory:'
        }, feedback=QgsProcessingFeedback())
    except Exception as e:
        raise ValueError(f"Failed to create hex grid: {str(e)}. Extent: {extent_width:.1f}m x {extent_height:.1f}m, Spacing: {spacing:.1f}m")
    
    # Transform back to WGS84 and filter to boundary
    transform_back = QgsCoordinateTransform(utm_crs, source_crs, QgsProject.instance())
    features = []
    
    for feature in result['OUTPUT'].getFeatures():
        geom = feature.geometry()
        geom.transform(transform_back)  # Transform back to WGS84
        center = geom.centroid()
        
        if boundary.contains(center):
            # Create new feature with walkPotential attribute
            new_feature = QgsFeature()
            new_feature.setGeometry(geom)
            new_feature.setAttributes([0])  # walkPotential = 0
            features.append(new_feature)
    
    return features


def geojson_to_qgs_geometry(geojson_geom: Dict) -> Optional[QgsGeometry]:
    """
    Convert GeoJSON geometry to QgsGeometry using QgsJsonUtils
    
    :param geojson_geom: GeoJSON geometry dictionary
    :return: QgsGeometry object or None if conversion fails
    """
    try:
        # Convert dict to JSON string for QgsJsonUtils.geometryFromGeoJson()
        geom_json = json.dumps(geojson_geom)
        geom = QgsJsonUtils.geometryFromGeoJson(geom_json)
        
        # geometryFromGeoJson returns a null geometry if parsing fails
        if geom.isNull():
            import sys
            print(f"DEBUG geojson_to_qgs_geometry: Geometry is null. Input was: {geom_json[:200]}", file=sys.stderr)
            return None
        
        return geom
            
    except Exception as e:
        import sys
        print(f"DEBUG geojson_to_qgs_geometry: Exception: {e}, geom: {geojson_geom}", file=sys.stderr)
        return None


def qgs_geometry_to_geojson(geom: QgsGeometry) -> Dict:
    """
    Convert QgsGeometry to GeoJSON geometry
    
    :param geom: QgsGeometry object
    :return: GeoJSON geometry dictionary
    """
    return json.loads(geom.asJson())




def create_hexagon(center_x: float, center_y: float, side_length: float) -> QgsGeometry:
    """
    Create a hexagon geometry centered at the given point
    
    :param center_x: X coordinate of center
    :param center_y: Y coordinate of center
    :param side_length: Length of each hexagon side
    :return: QgsGeometry representing the hexagon
    """
    points = []
    for i in range(6):
        angle_deg = 60 * i
        angle_rad = math.pi / 180 * angle_deg
        x = center_x + side_length * math.cos(angle_rad)
        y = center_y + side_length * math.sin(angle_rad)
        points.append(QgsPointXY(x, y))
    
    # Close the polygon
    points.append(points[0])
    
    return QgsGeometry.fromPolygonXY([points])


def buffer_geometry(geom: QgsGeometry, distance_km: float) -> QgsGeometry:
    """
    Buffer a geometry by a distance in kilometers
    
    :param geom: Input geometry
    :param distance_km: Buffer distance in kilometers
    :return: Buffered geometry
    """
    # Convert km to degrees (approximate)
    distance_deg = distance_km / 111.0
    return geom.buffer(distance_deg, 5)


def union_geometries(geometries: List[QgsGeometry]) -> Optional[QgsGeometry]:
    """
    Union multiple geometries into one
    
    :param geometries: List of QgsGeometry objects
    :return: Unioned geometry or None if empty
    """
    if not geometries:
        return None
    
    result = geometries[0]
    for geom in geometries[1:]:
        result = result.combine(geom)
    
    return result


def get_bbox(geom: QgsGeometry) -> Tuple[float, float, float, float]:
    """
    Get bounding box of a geometry
    
    :param geom: QgsGeometry object
    :return: Tuple of (minlon, minlat, maxlon, maxlat)
    """
    bbox = geom.boundingBox()
    return (bbox.xMinimum(), bbox.yMinimum(), bbox.xMaximum(), bbox.yMaximum())


def create_memory_layer(name: str, geom_type: str, crs: QgsCoordinateReferenceSystem, 
                        fields: QgsFields = None) -> QgsVectorLayer:
    """
    Create a memory vector layer
    
    :param name: Layer name
    :param geom_type: Geometry type ('Point', 'LineString', 'Polygon', etc.)
    :param crs: Coordinate reference system
    :param fields: Optional fields definition
    :return: QgsVectorLayer object
    """
    uri = f"{geom_type}?crs={crs.authid()}"
    layer = QgsVectorLayer(uri, name, "memory")
    
    if fields:
        provider = layer.dataProvider()
        provider.addAttributes(fields.toList())
        layer.updateFields()
    
    return layer


def calculate_area_sqm(geom: QgsGeometry) -> float:
    """
    Calculate area of a geometry in square meters
    
    :param geom: QgsGeometry object
    :return: Area in square meters
    """
    # This is approximate - for accurate calculation, should reproject to appropriate CRS
    # Using ellipsoidal area calculation
    area_deg_sq = geom.area()
    
    # Very rough approximation: 1 degree^2 ≈ 12,321 km^2 at equator
    # This varies significantly with latitude
    # For better accuracy, should use QgsDistanceArea with proper ellipsoid
    return area_deg_sq * 12321000000  # Convert to m^2
