# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Visualist
                                 A QGIS plugin
 Plugin for Crime Analysts
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-04-15
        copyright            : (C) 2019 by Quentin Rossy
        email                : quentin.rossy@unil.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Quentin Rossy'
__date__ = '2020-01-09'
__copyright__ = '(C) 2020 by Quentin Rossy'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os, math

import plotly as plt
import plotly.graph_objs as go
from plotly import tools

from qgis.PyQt.QtCore import QUrl, QVariant, QDateTime
from qgis.PyQt.QtWebKitWidgets import QWebView
from qgis.PyQt.QtWidgets import (QDialog,
                        QWidget,
                        QVBoxLayout,
                        QLabel)

from qgis.utils import iface
from qgis.core import (QgsApplication,
                       QgsFeatureRequest,
                       QgsFeatureSink,
                       QgsFeature,
                       QgsFields,
                       QgsField,
                       QgsGeometry,
                       QgsWkbTypes,
                       QgsDistanceArea,
                       QgsProject,
                       QgsProcessing,
                       QgsProcessingException,
                       QgsProcessingParameterFileDestination,
                       QgsProcessingParameterNumber,
                       QgsProcessingOutputNumber,
                       QgsProcessingParameterString,
                       QgsProcessingParameterCrs,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingFeatureSource,
                       QgsProcessingParameterField,
                       QgsProcessingParameterExpression,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterFeatureSink,
                       QgsSpatialIndex,
                       QgsCoordinateTransform,
                       QgsRectangle,
                       QgsDateTimeStatisticalSummary,
                       QgsExpression,
                       QgsProcessingUtils)

from processing.gui.BatchPanel import BatchPanel

from .visualist_alg import VisualistAlgorithm
from .TimeFormater import temporalAnalyser as TA
from datetime import datetime
from datetime import timedelta

class WebDialog(QDialog):

    def __init__(self, parent=None, title='WebDialog'):
        super().__init__(parent=parent)
        self.setWindowTitle(title)
        self.html_file = None
        layout = QVBoxLayout()
        layout.setMargin(0)
        layout.setSpacing(6)
        self.WebV = QWebView()
        layout.addWidget(self.WebV)
        self.setLayout(layout)

    def setHTML(self, file_path):
        self.html_file = file_path
        self.WebV.load(QUrl.fromLocalFile(self.html_file))


class MeetingPointsAnalysis(VisualistAlgorithm):

    INPUT1 = 'INPUT1'
    INPUT2 = 'INPUT2'
    CRS = 'CRS'
    STARTDT1_FIELD = 'STARTDATETIME1'
    ENDDT1_FIELD = 'ENDDATETIME1'
    STARTDT2_FIELD = 'STARTDATETIME2'
    ENDDT2_FIELD = 'ENDDATETIME2'
    FILTER1 = 'FILTER1'
    FILTER2 = 'FILTER2'

    DELTA_GEO = 'DELTAGEO'
    DELTA_TIME = 'DELTATIME'

    OUTPUT_LINE = 'OUTPUT_LINE'

    def __init__(self):
        super().__init__()

    def name(self):
        return 'meetingpointanalysis'

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterFeatureSource(self.INPUT1,
                                                              self.tr('First Input point layer'), [QgsProcessing.TypeVectorPoint]))
        sdt1 = QgsProcessingParameterField(self.STARTDT1_FIELD,
                                                      self.tr('Start datetime [first input]'),
                                                      parentLayerParameterName=self.INPUT1,
                                                      type = QgsProcessingParameterField.DateTime)
        sdt1.setFlags(sdt1.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(sdt1)
        edt1 = QgsProcessingParameterField(self.ENDDT1_FIELD,
                                                      self.tr('End datetime [first input]'),
                                                      parentLayerParameterName=self.INPUT1,
                                                      optional=True,
                                                      type = QgsProcessingParameterField.DateTime)
        edt1.setFlags(edt1.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(edt1)
        filter1 = QgsProcessingParameterExpression(self.FILTER1,
                                                           self.tr('Filter expression [first input]'),
                                                           optional=True,
                                                           parentLayerParameterName=self.INPUT1)
        filter1.setFlags(filter1.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(filter1)

        self.addParameter(QgsProcessingParameterFeatureSource(self.INPUT2,
                                                              self.tr('Second Input point layer'), [QgsProcessing.TypeVectorPoint]))
        sdt2 = QgsProcessingParameterField(self.STARTDT2_FIELD,
                                                      self.tr('Start datetime [second input]'),
                                                      parentLayerParameterName=self.INPUT2,
                                                      type = QgsProcessingParameterField.DateTime)
        sdt2.setFlags(sdt2.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(sdt2)
        edt2 = QgsProcessingParameterField(self.ENDDT2_FIELD,
                                                      self.tr('End datetime [second input]'),
                                                      parentLayerParameterName=self.INPUT2,
                                                      optional=True,
                                                      type = QgsProcessingParameterField.DateTime)
        edt2.setFlags(edt2.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(edt2)
        filter2 = QgsProcessingParameterExpression(self.FILTER2,
                                                           self.tr('Filter expression [second input]'),
                                                           optional=True,
                                                           parentLayerParameterName=self.INPUT2)
        filter2.setFlags(filter2.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(filter2)

        self.addParameter(QgsProcessingParameterCrs(self.CRS, 'PROJECTED CRS', defaultValue="EPSG:2056"))
        self.addParameter(QgsProcessingParameterNumber(self.DELTA_GEO,
                                                    self.tr('Maximum spatial distance [meter]'),
                                                    defaultValue=200))
        self.addParameter(QgsProcessingParameterNumber(self.DELTA_TIME,
                                                    self.tr('Maximum temporal distance [min]'),
                                                    defaultValue=60))
        # self.addParameter(QgsProcessingParameterField(self.FIELDS,
        #                                               self.tr('Fields to include'),
        #                                               parentLayerParameterName=self.INPUT,
        #                                               allowMultiple=True, optional=True))
        self.addParameter(QgsProcessingParameterFeatureSink(self.OUTPUT_LINE,
                                                        self.tr('Match'),
                                                        QgsProcessing.TypeVectorLine))
        # self.addParameter(QgsProcessingParameterFeatureSink(self.OUTPUT_POINT,
        #                                                 self.tr('End points'),
        #                                                 QgsProcessing.TypeVectorPoint))

    def postProcessAlgorithm(self, context, feedback):
        """
        PostProcessing Tasks to load html
        """
        # dial = WebDialog(iface.mainWindow(), self.displayName())
        # dial.setHTML(self.path)
        # dial.show()

        output = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)
        path = os.path.join(os.path.abspath(os.path.dirname(__file__)), "utils/styles/asLine.qml")
        feedback.pushInfo('Load symbology from file: {})'.format(path))
        output.loadNamedStyle(path)
        output.triggerRepaint()
        return self.output

    def processAlgorithm(self, parameters, context, feedback):
        self.target_crs = self.parameterAsCrs(parameters, self.CRS, context)

        self.source1 = self.parameterAsSource(parameters, self.INPUT1, context)
        if self.source1 is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT1))
        self.s1_transform = None
        if self.source1.sourceCrs().srsid() != self.target_crs.srsid():
            self.s1_transform = QgsCoordinateTransform(self.source1.sourceCrs(), self.target_crs, context.project())

        self.source2 = self.parameterAsSource(parameters, self.INPUT2, context)
        if self.source2 is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT2))
        self.s2_transform = None
        if self.source2.sourceCrs().srsid() != self.target_crs.srsid():
            self.s2_transform = QgsCoordinateTransform(self.source2.sourceCrs(), self.target_crs, context.project())

        self.sdt1_field = self.parameterAsString(parameters, self.STARTDT1_FIELD, context)
        self.sdt2_field = self.parameterAsString(parameters, self.STARTDT2_FIELD, context)
        self.edt1_field = self.parameterAsString(parameters, self.ENDDT1_FIELD, context)
        if self.edt1_field == "": self.edt1_field = self.sdt1_field
        self.edt2_field = self.parameterAsString(parameters, self.ENDDT2_FIELD, context)
        if self.edt2_field == "": self.edt2_field = self.sdt2_field
        self.sdt1_index = self.source1.fields().lookupField(self.sdt1_field)
        self.sdt2_index = self.source1.fields().lookupField(self.sdt2_field)
        self.edt1_index = self.source1.fields().lookupField(self.edt1_field)
        self.edt2_index = self.source1.fields().lookupField(self.edt2_field)

        self.delta_geo = self.parameterAsDouble(parameters, self.DELTA_GEO, context)
        self.delta_time = self.parameterAsDouble(parameters, self.DELTA_TIME, context)

        #Create output for lines
        fields = QgsFields()
        fields.append(QgsField('SourceId', QVariant.Int, "int", 9, 0))
        fields.append(QgsField('TargetId', QVariant.Int, "int", 9, 0))
        fields.append(QgsField('SpatialDistance', QVariant.LongLong))
        fields.append(QgsField('TemporalDistance', QVariant.String, '', 50))
        fields.append(QgsField('SourceStartDateTime', QVariant.DateTime))
        fields.append(QgsField('SourceEndDateTime', QVariant.DateTime))
        fields.append(QgsField('TargetStartDateTime', QVariant.DateTime))
        fields.append(QgsField('TargetEndDateTime', QVariant.DateTime))
        (self.sink, self.dest_id) = self.parameterAsSink(parameters, self.OUTPUT_LINE, context,
                                                fields, QgsWkbTypes.LineString, self.target_crs)
        if self.sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT_LINE))

        distances = self.compute(feedback, context)
        self.createLayer(distances, feedback)
        self.output = {self.OUTPUT_LINE: self.dest_id}
        return self.output

    def compute(self, feedback, context):
        self.distances = []
        if self.calcSpatialArea(feedback):
            if self.calcTemporalBounds(feedback):
                self.computeDistances(feedback, context)
            else:
                feedback.pushInfo(self.tr("No temporal overlapping: no match possible!"))
        else:
            feedback.pushInfo(self.tr("No spatial overlapping: no match possible!"))
        return self.distances

    def createLayer(self, distances, feedback):
        for data in distances:
            f = QgsFeature()
            geom = QgsGeometry().fromPolylineXY(data[1])
            # feedback.pushDebugInfo(self.tr("Geometry: {}").format(geom.constGet().clone()))
            f.setAttributes(data[0])
            f.setGeometry(geom)
            if feedback.isCanceled():
                return {}
            self.sink.addFeature(f, QgsFeatureSink.FastInsert)

    def getBBox(self, source, transform):
        bb = source.sourceExtent()
        if transform: bb = transform.transformBoundingBox(source.sourceExtent(), QgsCoordinateTransform.ForwardTransform)
        bb = QgsRectangle(bb.xMinimum()-self.delta_geo,
                            bb.yMinimum()-self.delta_geo,
                            bb.xMaximum() + self.delta_geo,
                            bb.yMaximum() + self.delta_geo)
        return bb

    def calcSpatialArea(self, feedback):
        feedback.setProgressText(self.tr('Check global spatial overlap'))
        self.layerZones = {} #kay layer id, value: qgsRectangle
        bb1 = self.getBBox(self.source1, self.s1_transform)
        bb2 = self.getBBox(self.source2, self.s2_transform)
        if bb1.intersects(bb2):
            self.spatialArea = bb1.intersect(bb2)
            feedback.pushDebugInfo(self.tr("Spatial overlap: {}").format(self.spatialArea.toString()))
            return True
        return False

    def dateStr(self, dt):
        #2016-01-06T09:03:52
        return dt.toString("yyyy-MM-ddThh:mm:ss")

    def getTimePeriod(self, source, start, end, feedback):
        fields = [start]
        if end: fields.append(end)
        request = QgsFeatureRequest().setFlags(QgsFeatureRequest.NoGeometry).setSubsetOfAttributes(fields, source.fields())
        features = source.getFeatures(request, QgsProcessingFeatureSource.FlagSkipGeometryValidityChecks)
        stat = QgsDateTimeStatisticalSummary()
        for current, ft in enumerate(features):
            if feedback.isCanceled(): break
            stat.addValue(ft[start])
            if end: stat.addValue(ft[end])
        stat.finalize()
        min = stat.statistic(QgsDateTimeStatisticalSummary.Min)
        max = stat.statistic(QgsDateTimeStatisticalSummary.Max)
        self.periods[source] = [min, max]
        feedback.pushDebugInfo(self.tr("Period of {}: {} - {}").format(source.sourceName(),self.dateStr(min),self.dateStr(max)))

    def calcTemporalBounds(self, feedback):
        feedback.setProgressText(self.tr('Check global temporal overlap'))
        self.periods = {} #key source, value: [minDate, maxDate]
        minDate, maxDate = None, None
        self.getTimePeriod(self.source1, self.sdt1_field, self.edt1_field, feedback)
        self.getTimePeriod(self.source2, self.sdt2_field, self.edt2_field, feedback)

        if self.periods[self.source1][1] < self.periods[self.source2][0] or self.periods[self.source1][0] > self.periods[self.source2][1]:
            return False

        self.period = self.periods[self.source1]
        if self.periods[self.source1][0] < self.periods[self.source2][0]: self.period[0] = self.periods[self.source2][0]
        if self.periods[self.source1][1] > self.periods[self.source2][1]: self.period[1] = self.periods[self.source2][1]
        self.period[0] = self.period[0].addSecs(-self.delta_time*60)
        self.period[1] = self.period[1].addSecs(self.delta_time*60)
        feedback.pushDebugInfo(self.tr("Temporal overlap: {} - {}").format(self.dateStr(self.period[0]),self.dateStr(self.period[1])))
        return True

    def filter(self, source, fields, feedback):
        filter = '{} >= {} AND {} <= {}'.format(
                                QgsExpression.quotedColumnRef(fields[0]),
                                QgsExpression.quotedValue(self.dateStr(self.period[0])),
                                QgsExpression.quotedColumnRef(fields[1]),
                                QgsExpression.quotedValue(self.dateStr(self.period[1]))
                                )
        feedback.pushInfo(self.tr("Filter: {}").format(filter))
        req = QgsFeatureRequest().setFilterExpression(filter).setSubsetOfAttributes(
                                [source.fields().lookupField(fields[0]),
                                source.fields().lookupField(fields[1])])
        features = source.getFeatures(req)
        i = 0
        for current, inFeat in enumerate(features):
            i += 1
        feedback.pushDebugInfo(self.tr("{} lines selected in {}").format(i, source.sourceName()))
        return req

    def convert_timedelta(self, duration):
        days, seconds = duration.days, duration.seconds
        hours = days * 24 + seconds // 3600
        minutes = (seconds % 3600) // 60
        seconds = (seconds % 60)
        return '{} hours, {} minutes, {} seconds'.format(hours, minutes, seconds)

    def computeDistances(self, feedback, context):
        time_delta = timedelta(minutes=self.delta_time) #days=0, seconds=0, microseconds=0, milliseconds=0, minutes=0, hours=0, weeks=0
        req1 = self.filter(self.source1, [self.sdt1_field, self.edt1_field], feedback)
        req2 = self.filter(self.source2, [self.sdt2_field, self.edt2_field], feedback)
        distArea = QgsDistanceArea()
        distArea.setSourceCrs(self.target_crs, context.transformContext())
        distArea.setEllipsoid(context.project().ellipsoid())
        total = 100.0 / self.source1.featureCount() if self.source1.featureCount() else 0
        distList = []
        feedback.setProgressText(self.tr("Run spatiotemporal comparison"))
        # spatialIndex = QgsSpatialIndex(self.source2.getFeatures(req2), feedback)
        spatialIndex = QgsSpatialIndex(self.source2.getFeatures(req2.setSubsetOfAttributes([]).setDestinationCrs(self.target_crs, context.transformContext())), feedback)
        for current, inFeat in enumerate(self.source1.getFeatures(req1)):
            if feedback.isCanceled():
                break
            inGeom = inFeat.geometry().asPoint()
            if self.source1.sourceCrs().srsid() != self.target_crs.srsid():
                inGeom = self.s1_transform.transform(inGeom)
            sdt1 = inFeat.attribute(self.sdt1_field).toPyDateTime()
            edt1 = inFeat.attribute(self.edt1_field).toPyDateTime()
            rect = QgsRectangle(
                    inGeom.x()-self.delta_geo*2,
                    inGeom.y()-self.delta_geo*2,
                    inGeom.x()+self.delta_geo*2,
                    inGeom.y()+self.delta_geo*2)
            neighbours = spatialIndex.intersects(rect)
            # feedback.pushDebugInfo(self.tr("{} neighbours for {}").format(len(neighbours), rect))
            # request = QgsFeatureRequest().setFilterRect(areaOfInterest)
            for outFeat in self.source2.getFeatures(QgsFeatureRequest(neighbours)):
                if feedback.isCanceled():
                    break
                sdt2 = outFeat.attribute(self.sdt2_field).toPyDateTime()
                edt2 = outFeat.attribute(self.edt2_field).toPyDateTime()
                if sdt2 >= edt1 + time_delta or sdt1 >= edt2 + time_delta:
                    continue
                temporalDistance = timedelta(0, 0, 0)
                if sdt2 >= edt1:
                    temporalDistance = sdt2 - edt1
                elif sdt1 >= edt2:
                    temporalDistance = sdt1 - edt2
                outGeom = outFeat.geometry().asPoint()
                if self.source2.sourceCrs().srsid() != self.target_crs.srsid():
                    outGeom = self.s2_transform.transform(outGeom)
                spatialDistance = distArea.measureLine(inGeom, outGeom)
                if spatialDistance <= self.delta_geo:
                    # feedback.pushDebugInfo(self.tr("{} is near {}: geoDist : {} times: {} {} / {} {}").format(inFeat.id(), outFeat.id(), spatialDistance, sdt1, edt1, sdt2, edt2))
                    self.distances.append([
                        [inFeat.id(),
                            outFeat.id(),
                            spatialDistance,
                            self.convert_timedelta(temporalDistance),
                            inFeat.attribute(self.sdt1_field),
                            inFeat.attribute(self.edt1_field),
                            outFeat.attribute(self.sdt2_field),
                            outFeat.attribute(self.edt2_field)
                            ],
                        [inGeom, outGeom]
                    ])
            feedback.setProgress(int(current * total))
