# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Visualist
                                 A QGIS plugin
 Plugin for Crime Analysts
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-04-15
        copyright            : (C) 2019 by Quentin Rossy
        email                : quentin.rossy@unil.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Quentin Rossy'
__date__ = '2019-04-18'
__copyright__ = '(C) 2019 by Quentin Rossy'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os, tempfile, sys
from . import pysal
import numpy

from qgis.PyQt.QtCore import QVariant
from qgis.core import (
                QgsProcessingParameterFeatureSink,
                QgsProcessingParameterVectorLayer,
                QgsProcessingParameterField,
                QgsProcessingException,
                QgsProcessing,
                QgsField,
                QgsProject,
                QgsFeatureSink,
                QgsVectorFileWriter,
                QgsFeature,
                QgsFeatureRequest,
                QgsProcessingUtils,
                QgsProcessingParameterEnum,
                QgsProcessingParameterNumber,
                QgsMessageLog,
                QgsWkbTypes
                )


from .visualist_alg import VisualistAlgorithm
from .utils import renderers

#Convenient function to debug
NAME = "Visualist"
log = lambda m: QgsMessageLog.logMessage(m, NAME)

class LocalIndicatorSpatialA(VisualistAlgorithm):
    dest_id = None  # Save a reference to the output layer id

    POLYGONS = 'POLYGONS'
    FIELD = 'FIELD'
    LISA = 'LISA'
    W = 'W'
    DISTANCE = 'DISTANCE'
    OUTPUT = 'OUTPUT'

    LISA_TYPE = {0:'Moran\'s I',1:'Getis-Ord Gi*'}
    W_TYPE = {0:'Queen', 1:'Rook', 2:'Bishop', 3:'Nearest neighbours'} #3:'Distance',

    def __init__(self):
        super().__init__()

    def name(self):
        return 'lisamap'

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterVectorLayer(self.POLYGONS,
                                    self.tr('Polygons'), [QgsProcessing.TypeVectorPolygon]))

        self.addParameter(QgsProcessingParameterField(self.FIELD,
                                    self.tr('Values'),
                                    type=QgsProcessingParameterField.Numeric,
                                    parentLayerParameterName=self.POLYGONS,
                                    allowMultiple=False, defaultValue=None))

        self.addParameter(QgsProcessingParameterEnum(self.LISA,
                                    self.tr('LISA Indicator'),
                                    self.LISA_TYPE.values(),
                                    allowMultiple=False, defaultValue=0))

        self.addParameter(QgsProcessingParameterEnum(self.W,
                                    self.tr('Spatial weights matrix'),
                                    self.W_TYPE.values(),
                                    allowMultiple=False, defaultValue=0))

        self.addParameter(QgsProcessingParameterNumber(self.DISTANCE,
                                    self.tr('Number of neighbours'),
                                    type=QgsProcessingParameterNumber.Integer,
                                    optional=True))

        self.addParameter(QgsProcessingParameterFeatureSink(self.OUTPUT,
                                    self.tr('Spatial Autocorrelation Map'),
                                    QgsProcessing.TypeVectorPolygon))

    def checkParameterValues(self, parameters, context):
            distance = self.parameterAsInt(parameters, self.DISTANCE, context)
            proximity = self.parameterAsEnum(parameters, self.W, context)
            if distance <= 0:
                if proximity == 3:
                    return False, self.tr('Invalid distance')
                elif proximity == 4:
                    return False, self.tr('Invalid number of near neighbours')
            return super(LocalIndicatorSpatialA, self).checkParameterValues(parameters, context)

    def postProcessAlgorithm(self, context, feedback):
        """
        PostProcessing Tasks
        """
        output = QgsProcessingUtils.mapLayerFromString(self.dest_id, context)
        r = renderers.MapRender(output)
        r.zscore('MORANS_Z' if self.lisa == 0 else 'GETISORD_Z')

        return {self.OUTPUT: self.dest_id}

    def processAlgorithm(self, parameters, context, feedback):
        poly_source = self.parameterAsVectorLayer(parameters, self.POLYGONS, context)
        if poly_source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.POLYGONS))
        # feedback.pushInfo('Input layer: {} '.format(poly_source))
        field_name = self.parameterAsString(parameters, self.FIELD, context)
        if field_name:
            field_name_index = poly_source.fields().lookupField(field_name)
        else:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.FIELD))

        self.lisa = self.parameterAsEnum(parameters, self.LISA, context)
        proximity = self.parameterAsEnum(parameters, self.W, context)
        distance = self.parameterAsInt(parameters, self.DISTANCE, context)

        #add fields for local Moran Results
        fields = poly_source.fields()
        fields.append(QgsField('MORANS_P' if self.lisa == 0 else 'GETISORD_P', QVariant.Double))
        fields.append(QgsField('MORANS_Z' if self.lisa == 0 else 'GETISORD_Z', QVariant.Double))
        fields.append(QgsField('MORANS_I' if self.lisa == 0 else 'GETISORD_Gistar', QVariant.Double))
        if self.lisa == 0:
            fields.append(QgsField('MORANS_C', QVariant.String))

        (sink, self.dest_id) = self.parameterAsSink(parameters, self.OUTPUT, context,
                                               fields, poly_source.wkbType(), poly_source.sourceCrs(), QgsFeatureSink.RegeneratePrimaryKey)
        if sink is None:
            raise QgsProcessingException(self.invalidSinkError(parameters, self.OUTPUT))



        #Save in a temporary shp and laod with pysal
        temp_file = os.path.join(tempfile.gettempdir(), 'lisa.shp')
        opt = QgsVectorFileWriter.SaveVectorOptions()
        opt.driverName = 'ESRI Shapefile'
        opt.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
        opt.overrideGeometryType = QgsWkbTypes.Polygon
        writerError = QgsVectorFileWriter.writeAsVectorFormatV3(layer=poly_source,
                            fileName=temp_file,
                            transformContext=QgsProject.instance().transformContext(),
                            options=opt)
        if writerError[0] != 0:
            feedback.pushInfo('Temp file writting error: {}'.format(writerError[1]))
        # Get data from field
        values = []
        features = poly_source.getFeatures()
        for current, polygon_feature in enumerate(features):
            n = polygon_feature[field_name_index]
            n_str = str(n)
            val = float(str(n)) if n_str != 'NULL' else 0
            values.append(val)
        feedback.pushInfo('Number of values of Y: {}'.format(len(values)))
        y = numpy.array(values)

        if proximity == 0: # queen
            w = pysal.lib.weights.contiguity.Queen.from_shapefile(temp_file) #, idVariable='fid'
        elif proximity == 1: # 1 for rook
            w = pysal.lib.weights.contiguity.Rook.from_shapefile(temp_file)
        elif proximity == 2: # 2 for bishop
            wq = pysal.lib.weights.contiguity.Queen.from_shapefile(temp_file)
            wr = pysal.lib.weights.contiguity.Rook.from_shapefile(temp_file)
            #feedback.pushInfo('W-Queen matrix size: {}'.format(wq.sparse.shape))
            #feedback.pushInfo('W-Rook matrix size: {}'.format(wr.sparse.shape))
            w = pysal.lib.weights.set_operations.w_difference(wq, wr, constrained=False, silence_warnings=True)
        elif proximity == 3: # 2 for NN
            w = pysal.lib.weights.distance.KNN.from_shapefile(temp_file, k=distance)
        #elif proximity == 3: # 2 for distance bnd
        #    w = pysal.lib.weights.distance.DistanceBand.from_shapefile(temp_file, threshold=distance)

        feedback.pushInfo('W matrix size: {}'.format(w.sparse.shape))
        if len(values) != w.sparse.shape[0]:
            feedback.pushInfo('Dimension mismatch: unkown error!')
            return False
        if self.lisa == 0:
            results = pysal.explore.esda.moran.Moran_Local(y, w)
            cotype = {1:'HH',2:'LH',3:'LL',4:'HL'}
        elif self.lisa == 1:
            results = pysal.explore.esda.getisord.G_Local(y, w, star=True)

        # sig_q = results.q * (results.p_sim <= 0.01) # could make significance level an option
        features = poly_source.getFeatures()
        total = 100.0 / poly_source.featureCount() if poly_source.featureCount() else 0
        for current, polygon_feature in enumerate(features):
            if feedback.isCanceled():
                break
            output_feature = QgsFeature()
            geom = polygon_feature.geometry()
            output_feature.setGeometry(geom)
            attrs = polygon_feature.attributes()
            attrs.append(float(results.p_sim[current]))
            attrs.append(float(results.z_sim[current]))
            if self.lisa == 0:
                attrs.append(float(results.Is[current]))
                c = cotype[results.q[current]]
                attrs.append(c)
            elif self.lisa == 1:
                attrs.append(float(results.Gs[current]))
            output_feature.setAttributes(attrs)

            sink.addFeature(output_feature, QgsFeatureSink.FastInsert)
            feedback.setProgress(int(current * total))
        return {self.OUTPUT: self.dest_id}
