#%%
# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ViewDrone
                                 A QGIS plugin
 A QGIS plugin for viewshed analysis in drone mission planning
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-10-09
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Helena Walther Lange
        email                : HEWA@niras.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Import standard and third-party libraries
from fileinput import filename
import os
from re import sub
import shutil
import subprocess
import numpy as np
from osgeo import gdal, ogr, gdalconst
import subprocess


# Import QGIS libraries
from qgis.core import (
    Qgis,
    QgsProject,
    QgsFeature,
    QgsField,
    QgsGeometry,
    QgsVectorLayer,
    QgsRasterLayer,
    QgsPointXY,
    QgsColorRampShader,
    QgsRasterShader,
    QgsSingleBandPseudoColorRenderer,
)

from qgis.PyQt.QtCore import QVariant, QSettings, QTranslator, QCoreApplication, QTemporaryFile
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtCore import Qt
from qgis.gui import QgsMapLayerComboBox, QgsMessageBar
from PyQt5.QtWidgets import QDialog
from PyQt5.QtGui import QColor

# Initialize Qt resources from file resources.py
from .resources import *
from .view_drone_dialog import Ui_ViewDroneDialogBase
import os.path

class ViewDroneDialog(QDialog):
    def __init__(self):
        super(ViewDroneDialog, self).__init__()
        self.ui = Ui_ViewDroneDialogBase()
        self.ui.setupUi(self)

class ViewDrone:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ViewDrone_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&ViewDrone')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ViewDrone', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/view_drone/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'ViewDrone'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&ViewDrone'),
                action)
            self.iface.removeToolBarIcon(action)


    def show_message(self, message, level = Qgis.Info, duration=5):
        """Show message to the user"""
        # Clear previous message
        self.iface.messageBar().clearWidgets()
        # Display message
        self.iface.messageBar().pushMessage("ViewDrone", message, level=level, duration=duration)


    def run(self):
        """Run method that performs all the real work"""

        # Initialize and execute the dialog
        self.dlg = ViewDroneDialog()

        result = self.dlg.exec_()

        if result:
            self.show_message("Processing", duration=9)
            # Retrieve user inputs

            # Digital surface model
            input_DSM = self.dlg.ui.DSM.currentLayer()

            # Area polygon
            input_Area = self.dlg.ui.Area.currentLayer()

            # Target height
            input_target_height = self.dlg.ui.TargetHeight.value()

            # Observer height
            input_observer_height = self.dlg.ui.ObserverHeight.value()

            # Range
            input_range = self.dlg.ui.Range.value()

            # Grid size
            input_grid_size = self.dlg.ui.GridSize.value()

            # Output directory
            outdir = self.dlg.ui.OutputDir.filePath()

            #%%

            # Function that creates a grid of points based on the polygon layer
            def create_grid_from_polygon(layer, grid_size):
                # Create point vector layer in the same CRS as the input and add ID attribute to the grid layer
                grid_layer = QgsVectorLayer("Point?crs={}".format(layer.crs().toWkt()), "grid", "memory")
                provider = grid_layer.dataProvider()
                provider.addAttributes([QgsField("id", QVariant.Int)])
                grid_layer.updateFields()
                
                id = 0
                # Loop trough each feature in the input layer
                for feature in layer.getFeatures():
                    # Use the geometry of the feature to get a bounding box
                    geom = feature.geometry()
                    bounds = geom.boundingBox()
                    # Extract the minimum and maximum x and y values
                    x_min = bounds.xMinimum()
                    x_max = bounds.xMaximum()
                    y_min = bounds.yMinimum()
                    y_max = bounds.yMaximum()
                    
                    # Generate point within the bounding box with intervals from the grid size
                    x = x_min
                    while x < x_max:
                        y = y_min
                        while y < y_max:
                            point = QgsGeometry.fromPointXY(QgsPointXY(x, y))
                            if geom.contains(point.asPoint()):
                                f = QgsFeature()
                                f.setGeometry(point)
                                f.setAttributes([id])
                                provider.addFeature(f)
                                id += 1
                            y += grid_size
                        x += grid_size
                
                # Update layer extent and add as a temporary scratch layer in the project
                grid_layer.updateExtents()
                QgsProject.instance().addMapLayer(grid_layer)
                return grid_layer

            # Function that writes the data to a raster file
            def write_data_as_raster(image_data, fn_raster, fn_output, fn_poly = None):
                """From MAGI utils repos
                Written by Peter Gelsbo (PEGE@niras.dk)"""
                data = gdal.Open(fn_raster, gdalconst.GA_ReadOnly)
                geo_transform = data.GetGeoTransform()
                
                x_min = geo_transform[0]
                y_min = geo_transform[3]

                x_res = data.RasterXSize
                y_res = data.RasterYSize

                srs = data.GetProjection()

                pixel_width = geo_transform[1]

                target_ds = gdal.GetDriverByName('GTiff').Create(fn_output, x_res, y_res, 1, gdal.GDT_Float32, ['COMPRESS=DEFLATE'])
                target_ds.SetGeoTransform((x_min, pixel_width, 0, y_min, 0, -pixel_width))
                target_ds.SetProjection(srs)

                
                band = target_ds.GetRasterBand(1)
                NoData_value = -1
                band.SetNoDataValue(NoData_value)
                band.FlushCache()
                
                target_ds.GetRasterBand(1).WriteArray(image_data)

                if fn_poly != None:
                    
                    driver = ogr.GetDriverByName("ESRI Shapefile")
                    outDatasource = driver.CreateDataSource(fn_poly)
                    outLayer = outDatasource.CreateLayer("polygonized", srs=None)

                    fd = ogr.FieldDefn( 'DN', ogr.OFTInteger )
                    outLayer.CreateField( fd )

                    gdal.Polygonize( target_ds.GetRasterBand(1), None, outLayer, 0)



                target_ds = None

            # Create the grid layer
            grid_layer = create_grid_from_polygon(input_Area,input_grid_size)

            # Open the DSM raster and define the band
            src_ds = gdal.Open(input_DSM.source())
            band = src_ds.GetRasterBand(1)

            # Function to run the GDAL viewshed command 
            def run_viewshed(ox,oy,tmp_raster,band):
                try:
                    #Temporary output raster
                    cmd = f'gdal_viewshed -a_nodata -1 -f "GTiff" -oz {input_target_height} -tz {input_observer_height} -md {input_range} -ox {ox} -oy {oy} -vv 1 -iv 0 -ov 0 -cc 0.85714 "{input_DSM.source()}" "{tmp_raster}"'
                    subprocess.run(cmd, capture_output=True, text=True, shell=True)

                    return tmp_raster
                
                except Exception as e:
                    raise e

            # Create a folder for the temporary files
            sub_dir = 'temp_files'
            sub_dir_path = os.path.join(outdir, sub_dir)

            try:
                # Check if temporary directory exists
                if not os.path.exists(sub_dir_path):
                    os.makedirs(sub_dir_path)

                    # List to store the paths of the temporary viewshed rasters
                    results = []

                    # Run viewshed analysis for each point in the grid
                    for i, feature in enumerate(grid_layer.getFeatures()):
                        geom = feature.geometry()

                        ox, oy = geom.asPoint().x(), geom.asPoint().y()
                        result = run_viewshed(ox, oy, f'{sub_dir_path}/temp_tif_{i}', band)
                        if result:
                            results.append(result)


                    # Create virtual raster layer (VRT) from the temporary viewshed rasters and read as numpy array
                    vrt = gdal.BuildVRT(os.path.join(sub_dir_path, 'stack.vrt'), results, separate=True)
                    arr = vrt.ReadAsArray()
                    vrt = None

                    # Calculate the number of points
                    num_points = grid_layer.featureCount()
                    # Calculate the sum of all viewsheds
                    data_sum = np.sum(arr, axis=0)
                    # Scale the sum to the number of points
                    scaled_sum = data_sum / num_points
                    # Define a path for the output file
                    viewshed_sum_raster_path = os.path.join(outdir, 'viewshed.tif')
                    write_data_as_raster(scaled_sum, os.path.join(sub_dir_path, 'stack.vrt'), viewshed_sum_raster_path)
                   
                   # Add the output layer to the QGIS project
                    viewshed_sum_layer = QgsRasterLayer(viewshed_sum_raster_path, "Viewshed", "gdal")
                    if viewshed_sum_layer.isValid():
                        
                        # Set colour map for output layer
                        color_ramp = QgsColorRampShader()
                        color_ramp.setColorRampType(QgsColorRampShader.Exact)

                        colors = [
                        QgsColorRampShader.ColorRampItem(0, QColor(0, 0, 0, 0 )),
                        QgsColorRampShader.ColorRampItem(1, QColor('green'))
                        ]

                        color_ramp.setColorRampItemList(colors)

                        shader = QgsRasterShader()
                        shader.setRasterShaderFunction(color_ramp)

                        renderer = QgsSingleBandPseudoColorRenderer(viewshed_sum_layer.dataProvider(), 1, shader)
                        viewshed_sum_layer.setRenderer(renderer)
                        viewshed_sum_layer.triggerRepaint()

                        # Set opacity to 80%
                        viewshed_sum_layer.setOpacity(0.8)

                        # Add layer to QGIS
                        QgsProject.instance().addMapLayer(viewshed_sum_layer)

                    else:
                        print("Failed to load viewshed.tif")

            finally:
                # Delete the temporary folder with the viewshed rasters
                if os.path.exists(sub_dir_path):
                    shutil.rmtree(sub_dir_path)

                self.show_message("Done! :-)", level=Qgis.Success, duration=5)
                        
        
           