# -*- coding: utf-8 -*-
"""
/***************************************************************************
 VectorToMap - QGIS Plugin
 Automates the generation of print layouts for vector features.
 Author: Matheus Durso
 ***************************************************************************/
"""

import re
import unicodedata
import os.path

# Standard library and QGIS/PyQt imports
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QTimer
from qgis.PyQt.QtGui import QIcon, QPixmap, QImage, QFont
from qgis.PyQt.QtWidgets import QAction, QCheckBox, QTableWidgetItem, QFrame
from qgis.core import (
    QgsProject, QgsPrintLayout, QgsLayoutSize, QgsUnitTypes, QgsLayoutItemMap, 
    QgsLayoutPoint, QgsMapLayerProxyModel, QgsFeatureRequest, QgsSettings, 
    QgsLayoutExporter, QgsRectangle, QgsCoordinateTransform, QgsLayoutItemPage, 
    QgsLayoutItemLabel, QgsLayoutItem
)
from qgis.utils import iface

# Resources generated by Plugin Builder
from .resources import *
from .vector_to_map_dialog import VectorToMapDialog

class VectorToMap:
    def __init__(self, iface):
        """Initialize the plugin and define global state variables."""
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr(u'&Vector to Map')
        self.first_start = True
        self.is_rendering = False # Flag to prevent infinite rendering loops

    def tr(self, message):
        """Translate strings using the QGIS internationalization system."""
        return QCoreApplication.translate('VectorToMap', message)

    def add_action(self, icon_path, text, callback, enabled_flag=True, add_to_menu=True, add_to_toolbar=True, status_tip=None, whats_this=None, parent=None):
        """Helper function to add buttons and menus to the QGIS interface."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        if status_tip: action.setStatusTip(status_tip)
        if whats_this: action.setWhatsThis(whats_this)
        if add_to_toolbar: self.iface.addToolBarIcon(action)
        if add_to_menu: self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        """Set up icons and menus when the plugin starts."""
        icon_path = ':/plugins/vector_to_map/icon.png'
        self.action = self.add_action(icon_path, text='VectorToMap (Atlas & Layout)', callback=self.run, parent=self.iface.mainWindow())
        self.iface.addPluginToVectorMenu('Vector to Map', self.action)

    def unload(self):
        """Remove plugin actions when it is disabled."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&Vector to Map'), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Initialize the UI and set up the 'Render Preview' button logic (v0.2.0)"""
        from qgis.PyQt.QtWidgets import QPushButton, QDialogButtonBox

        if self.first_start:
            self.first_start = False
            self.dlg = VectorToMapDialog()
            
            # Setup rendering engine timer
            self.is_rendering = False
            self.timer_preview = QTimer()
            self.timer_preview.setSingleShot(True)
            self.timer_preview.timeout.connect(self.atualizar_preview)

            # --- SETUP 'Render Preview' BUTTON ---
            # Use addButton with ActionRole to keep it close to the OK/Cancel group
            self.btn_render = QPushButton("Render Preview")
            self.dlg.button_box.addButton(self.btn_render, QDialogButtonBox.ActionRole)
            self.btn_render.clicked.connect(self.atualizar_preview)

            # INITIAL STATE: Auto-preview disabled by default
            if hasattr(self.dlg, 'chk_preview_auto'):
                self.dlg.chk_preview_auto.setChecked(False)
                # Render immediately if the user turns on auto-preview
                self.dlg.chk_preview_auto.stateChanged.connect(
                    lambda state: self.timer_preview.start(100) if state == Qt.Checked else None
                )

            # Restore window geometry and Splitter positions (50/50 split)
            settings = QgsSettings()
            geometria = settings.value("/VectorToMap/geometry")
            if geometria: self.dlg.restoreGeometry(geometria)
            else: self.dlg.resize(1100, 750) 
            if hasattr(self.dlg, 'splitter'): self.dlg.splitter.setSizes([550, 550]) 
            
            self.dlg.lbl_preview.setFrameShape(QFrame.StyledPanel)
            self.dlg.mMapLayerComboBox.setFilters(QgsMapLayerProxyModel.VectorLayer)

            # Layer and selection signals
            self.dlg.mMapLayerComboBox.layerChanged.connect(self.atualizar_lista_atributos)
            self.dlg.mMapLayerComboBox.layerChanged.connect(self.disparar_preview_se_autorizado)
            self.dlg.chk_selecionar_todos.stateChanged.connect(self.marcar_desmarcar_todos)

            # Triggers for fixed UI widgets
            widgets = [self.dlg.combo_tamanho_pagina, self.dlg.combo_presets, 
                       self.dlg.rb_retrato, self.dlg.rb_paisagem, 
                       self.dlg.chk_modo_formulario, self.dlg.chk_modo_individual]
            
            for w in widgets:
                if hasattr(w, 'currentIndexChanged'): w.currentIndexChanged.connect(self.disparar_preview_se_autorizado)
                elif hasattr(w, 'stateChanged'): w.stateChanged.connect(self.disparar_preview_se_autorizado)
                elif hasattr(w, 'toggled'): w.toggled.connect(self.disparar_preview_se_autorizado)

            self.dlg.resizeEvent = lambda event: self.disparar_preview_se_autorizado()
            self.dlg.button_box.accepted.connect(self.processar_clique_ok)
            self.dlg.button_box.rejected.connect(self.dlg.close)
            self.atualizar_lista_atributos()

        self.dlg.show()

    def disparar_preview_se_autorizado(self):
        """Start the preview timer only if auto-update is checked."""
        if hasattr(self.dlg, 'chk_preview_auto') and self.dlg.chk_preview_auto.isChecked():
            self.timer_preview.start(600)

    def marcar_desmarcar_todos(self, state):
        """Bulk check/uncheck attributes and consult auto-preview status."""
        check = (state == Qt.Checked)
        container = self.dlg.scrollAreaWidgetContents
        for cb in container.findChildren(QCheckBox):
            cb.setChecked(check)
        
        self.disparar_preview_se_autorizado()

    def montar_design_da_pagina(self, layout, camada, feicoes_da_pagina, preset, orientacao, tamanho_pg, pagina_index=0):
        """Core Factory: Handles Iron-Lock geometry and custom scaling logic."""
        from qgis.core import (QgsLayoutItemMap, QgsLayoutItemLabel, QgsLayoutSize, 
                               QgsUnitTypes, QgsLayoutPoint, QgsCoordinateTransform, 
                               QgsRectangle, QgsProject, QgsLayoutItem)
        
        # 1. PAGE SETUP
        tamanhos = {'A4': (210, 297), 'A3': (297, 420), 'A2': (420, 594), 'A1': (594, 841), 'A0': (841, 1189)}
        dim = tamanhos.get(tamanho_pg, (210, 297))
        w_pg, h_pg = (dim[1], dim[0]) if orientacao == "Paisagem" else (dim[0], dim[1])
        
        pagina = layout.pageCollection().pages()[pagina_index]
        pagina.setPageSize(QgsLayoutSize(w_pg, h_pg, QgsUnitTypes.LayoutMillimeters))
        
        y_zero_folha = pagina_index * (h_pg + 10) 
        margin_padrao = 10.0 # 1cm margins

        # 2. IRON-LOCK GEOMETRY
        if preset == "75% da Altura (Base fixada)":
            header_h = 35.0
            h_util = h_pg - (2 * margin_padrao) - header_h
            w_map_f, h_map_f = w_pg - (2 * margin_padrao), h_util * 0.75
            x_map_f, y_map_f = margin_padrao, y_zero_folha + margin_padrao + header_h
        else: # SQUARE MODEL
            if orientacao == "Retrato":
                # Full width square with 1cm side margins
                w_map_f = w_pg - (2 * margin_padrao)
                h_map_f = w_map_f
                x_map_f, y_map_f = margin_padrao, y_zero_folha + 40.0
            else: # Landscape Square (Positioned to the right)
                h_map_f = h_pg - (2 * margin_padrao)
                w_map_f = h_map_f
                x_map_f = w_pg - margin_padrao - w_map_f
                y_map_f = y_zero_folha + margin_padrao

        map_item = QgsLayoutItemMap(layout)
        map_item.setFrameEnabled(True)
        map_item.attemptResize(QgsLayoutSize(w_map_f, h_map_f, QgsUnitTypes.LayoutMillimeters))
        map_item.attemptMove(QgsLayoutPoint(x_map_f, y_map_f, QgsUnitTypes.LayoutMillimeters))
        
        # 3. MANUAL SCALING LOGIC
        if feicoes_da_pagina:
            ext = QgsRectangle()
            ext.setMinimal()
            for f in feicoes_da_pagina: ext.combineExtentWith(f.geometry().boundingBox())
            trans = QgsCoordinateTransform(camada.crs(), QgsProject.instance().crs(), QgsProject.instance())
            ext_proj = trans.transformBoundingBox(ext)
            map_item.setExtent(ext_proj)
            
            # Scale calculation based on the longest feature side
            # $$Scale = \max\left(\frac{W_{geo}}{W_{frame}}, \frac{H_{geo}}{H_{frame}}\right) \times 1.2$$
            mult = 1000.0 if QgsProject.instance().crs().mapUnits() != QgsUnitTypes.DistanceDegrees else 1.0
            escala_w, escala_h = (ext_proj.width() * mult) / w_map_f, (ext_proj.height() * mult) / h_map_f
            map_item.setScale(max(escala_w, escala_h) * 1.2)

            # RE-LOCK GEOMETRY: Ensure the frame doesn't move after scaling
            map_item.attemptResize(QgsLayoutSize(w_map_f, h_map_f, QgsUnitTypes.LayoutMillimeters))
            map_item.attemptMove(QgsLayoutPoint(x_map_f, y_map_f, QgsUnitTypes.LayoutMillimeters))
        
        layout.addLayoutItem(map_item)

        # 4. ATTRIBUTE DATA PLACEMENT
        colunas = [cb.text() for cb in self.dlg.scrollAreaWidgetContents.findChildren(QCheckBox) if cb.isChecked()]
        limite_fundo = y_zero_folha + h_pg - margin_padrao

        if colunas and feicoes_da_pagina:
            # Special Logic: 75% Landscape -> Left (Form) and Right (Labels) columns
            if preset == "75% da Altura (Base fixada)" and orientacao == "Paisagem":
                mid_x = w_pg / 2
                x_form, y_form = margin_padrao, y_map_f + h_map_f + 3.0
                w_form = mid_x - margin_padrao - 2.0
                h_form = limite_fundo - y_form
                
                x_ind, y_ind = mid_x + 2.0, y_map_f + h_map_f + 3.0
                w_ind = w_pg - margin_padrao - x_ind
                h_ind = limite_fundo - y_ind

                if self.dlg.chk_modo_formulario.isChecked():
                    txt_html = ""
                    for idx, f in enumerate(feicoes_da_pagina):
                        if len(feicoes_da_pagina) > 1: txt_html += f"<b>ITEM {idx+1}</b><br>"
                        for col in colunas:
                            try: txt_html += f"<b>{col}:</b> {str(f.attribute(col)).strip()}<br>"
                            except: continue
                    lbl_f = QgsLayoutItemLabel(layout)
                    lbl_f.setMode(QgsLayoutItemLabel.ModeHtml)
                    lbl_f.setText(txt_html)
                    lbl_f.attemptMove(QgsLayoutPoint(x_form, y_form, QgsUnitTypes.LayoutMillimeters))
                    lbl_f.attemptResize(QgsLayoutSize(w_form, h_form, QgsUnitTypes.LayoutMillimeters))
                    layout.addLayoutItem(lbl_f)

                if self.dlg.chk_modo_individual.isChecked():
                    xi, yi = x_ind, y_ind
                    for f in feicoes_da_pagina:
                        for col in colunas:
                            try:
                                lbl_i = QgsLayoutItemLabel(layout)
                                lbl_i.setText(f"{col}: {str(f.attribute(col)).strip()}")
                                lbl_i.setFrameEnabled(True); lbl_i.adjustSizeToText()
                                # Row-break logic for the right column
                                if xi + lbl_i.rect().width() > (x_ind + w_ind): xi, yi = x_ind, yi + 8
                                if yi + 8 > limite_fundo: break
                                lbl_i.attemptMove(QgsLayoutPoint(xi, yi, QgsUnitTypes.LayoutMillimeters))
                                layout.addLayoutItem(lbl_i)
                                xi += lbl_i.rect().width() + 3.0
                            except: continue
            
            else:
                # Standard Logic: Vertically stacked display
                if orientacao == "Retrato":
                    x_dados, y_dados = margin_padrao, y_map_f + h_map_f + 5.0
                    w_dados = w_pg - (2 * margin_padrao)
                else: # Landscape Square (Map on the right, data on the left)
                    x_dados, y_dados = margin_padrao, y_map_f + (h_map_f / 2)
                    w_dados = x_map_f - (2 * margin_padrao)
                
                h_dados_max = limite_fundo - y_dados

                if self.dlg.chk_modo_formulario.isChecked():
                    txt_html = ""
                    for idx, f in enumerate(feicoes_da_pagina):
                        if len(feicoes_da_pagina) > 1: txt_html += f"<b>ITEM {idx+1}</b><br>"
                        for col in colunas:
                            try: txt_html += f"<b>{col}:</b> {str(f.attribute(col)).strip()}<br>"
                            except: continue
                    lbl_f = QgsLayoutItemLabel(layout)
                    lbl_f.setMode(QgsLayoutItemLabel.ModeHtml)
                    lbl_f.setText(txt_html)
                    h_f = h_dados_max * 0.6 if self.dlg.chk_modo_individual.isChecked() else h_dados_max
                    lbl_f.attemptMove(QgsLayoutPoint(x_dados, y_dados, QgsUnitTypes.LayoutMillimeters))
                    lbl_f.attemptResize(QgsLayoutSize(w_dados, h_f, QgsUnitTypes.LayoutMillimeters))
                    layout.addLayoutItem(lbl_f)
                    if self.dlg.chk_modo_individual.isChecked(): y_dados += h_f + 3.0

                if self.dlg.chk_modo_individual.isChecked():
                    xi, yi = x_dados, y_dados
                    for f in feicoes_da_pagina:
                        for col in colunas:
                            try:
                                lbl_i = QgsLayoutItemLabel(layout)
                                lbl_i.setText(f"{col}: {str(f.attribute(col)).strip()}")
                                lbl_i.setFrameEnabled(True); lbl_i.adjustSizeToText()
                                if xi + lbl_i.rect().width() > (x_dados + w_dados): xi, yi = x_dados, yi + 8
                                if yi + 8 > limite_fundo: break
                                lbl_i.attemptMove(QgsLayoutPoint(xi, yi, QgsUnitTypes.LayoutMillimeters))
                                layout.addLayoutItem(lbl_i); xi += lbl_i.rect().width() + 3.0
                            except: continue

    def atualizar_preview(self):
        """Render the layout preview on the UI using a sample feature."""
        if self.is_rendering or not self.dlg.isVisible(): return
        self.is_rendering = True
        QCoreApplication.processEvents()

        camada = self.dlg.mMapLayerComboBox.currentLayer()
        if not camada: 
            self.is_rendering = False; return

        layout = QgsPrintLayout(QgsProject.instance())
        layout.initializeDefaults()
        
        feicao = next(camada.getFeatures(QgsFeatureRequest().setLimit(1)), None)
        feicoes = [feicao] if feicao else []

        self.montar_design_da_pagina(layout, camada, feicoes, 
            self.dlg.combo_presets.currentText(),
            "Retrato" if self.dlg.rb_retrato.isChecked() else "Paisagem",
            self.dlg.combo_tamanho_pagina.currentText())

        exporter = QgsLayoutExporter(layout)
        image = exporter.renderPageToImage(0)
        if not image.isNull():
            self.dlg.lbl_preview.setPixmap(QPixmap.fromImage(image).scaled(
                self.dlg.lbl_preview.size(), Qt.KeepAspectRatio, Qt.SmoothTransformation))
        
        self.is_rendering = False
        QCoreApplication.processEvents()

    def criar_layout_multi_paginas(self, preset, camada, orientacao, tamanho_pg, paginas_dados, colunas, modos):
        """Generate the definitive multi-page layout in the QGIS Layout Manager."""
        project = QgsProject.instance()
        manager = project.layoutManager()
        
        nome_camada = re.sub(r'[^a-zA-Z0-9_]', '_', unicodedata.normalize('NFD', camada.name()).encode('ascii', 'ignore').decode('utf-8'))
        layout_name = f"VectorToMap_{nome_camada}"
        
        contador = 1
        while manager.layoutByName(layout_name):
            layout_name = f"VectorToMap_{nome_camada}_{contador}"; contador += 1

        layout = QgsPrintLayout(project)
        layout.initializeDefaults()
        layout.setName(layout_name)

        for i, dados in enumerate(paginas_dados):
            if i > 0: layout.pageCollection().addPage(QgsLayoutItemPage(layout))
            self.montar_design_da_pagina(layout, camada, dados['feicoes'], preset, orientacao, tamanho_pg, pagina_index=i)

        layout.refresh()
        manager.addLayout(layout)
        self.iface.openLayoutDesigner(layout) # Open the designer window for the user

    def processar_clique_ok(self):
        """Process configurations and start the layout generation engine."""
        camada = self.dlg.mMapLayerComboBox.currentLayer()
        if not camada: return
        
        container = self.dlg.scrollAreaWidgetContents
        colunas_selecionadas = [cb.text() for cb in container.findChildren(QCheckBox) if cb.isChecked()]

        paginas_dados = []
        campo_atlas = self.dlg.combo_atlas.currentData()
        
        if campo_atlas is None:
            for f in camada.getFeatures(): paginas_dados.append({'feicoes': [f]})
        else:
            idx = camada.fields().indexOf(campo_atlas)
            for v in camada.uniqueValues(idx):
                exp = f'"{campo_atlas}" = \'{v}\'' if isinstance(v, str) else f'"{campo_atlas}" = {v}'
                feicoes = list(camada.getFeatures(QgsFeatureRequest().setFilterExpression(exp)))
                if feicoes: paginas_dados.append({'feicoes': feicoes})
        
        if paginas_dados: 
            self.criar_layout_multi_paginas(self.dlg.combo_presets.currentText(), camada, 
                "Retrato" if self.dlg.rb_retrato.isChecked() else "Paisagem", 
                self.dlg.combo_tamanho_pagina.currentText(), paginas_dados, colunas_selecionadas, None)

    def atualizar_lista_atributos(self):
        """Clear the interface and rebuild the attribute list with preview triggers."""
        camada = self.dlg.mMapLayerComboBox.currentLayer()
        self.dlg.tableWidget.setRowCount(0)
        self.dlg.tableWidget.setColumnCount(0)

        # Clear existing dynamic checkboxes
        layout_dest = self.dlg.verticalLayout 
        while layout_dest.count():
            item = layout_dest.takeAt(0)
            if item.widget(): item.widget().deleteLater()
        
        self.dlg.chk_selecionar_todos.blockSignals(True)
        self.dlg.chk_selecionar_todos.setChecked(False)
        self.dlg.chk_selecionar_todos.blockSignals(False)
        self.dlg.combo_atlas.clear()
        self.dlg.combo_atlas.addItem("--- No Grouping (Each Feature) ---", None)

        if not camada: 
            self.atualizar_estado_modos_exibicao()
            return

        # Rebuild list and connect each checkbox to the preview engine
        for campo in camada.fields():
            nome = campo.name()
            cb = QCheckBox(nome)
            cb.stateChanged.connect(self.atualizar_tabela_por_selecao)
            cb.stateChanged.connect(self.atualizar_estado_modos_exibicao)
            cb.stateChanged.connect(self.verificar_estado_selecionar_todos)
            
            # Individual trigger for auto-preview logic
            cb.stateChanged.connect(self.disparar_preview_se_autorizado)
            
            layout_dest.addWidget(cb)
            self.dlg.combo_atlas.addItem(nome, nome)

        layout_dest.addStretch(1)
        self.atualizar_estado_modos_exibicao()

    def atualizar_tabela_por_selecao(self):
        """Synchronize the sample data table with the user's attribute selection."""
        camada = self.dlg.mMapLayerComboBox.currentLayer()
        tabela = self.dlg.tableWidget
        if not camada: return
        
        colunas = [cb.text() for cb in self.dlg.scrollAreaWidgetContents.findChildren(QCheckBox) if cb.isChecked()]
        tabela.setColumnCount(len(colunas))
        tabela.setHorizontalHeaderLabels(colunas)
        
        if not colunas:
            tabela.setRowCount(0); return
            
        num_linhas = min(10, camada.featureCount())
        tabela.setRowCount(num_linhas)
        for i, feicao in enumerate(camada.getFeatures(QgsFeatureRequest().setLimit(num_linhas))):
            for j, col in enumerate(colunas):
                tabela.setItem(i, j, QTableWidgetItem(str(feicao[col])))
        tabela.resizeColumnsToContents()

    def atualizar_estado_modos_exibicao(self):
        """Manage accessibility for display modes and set 'Form' as default."""
        container = self.dlg.scrollAreaWidgetContents
        tem_selecao = any(cb.isChecked() for cb in container.findChildren(QCheckBox))
        
        # Enable modes only when at least one column is selected
        self.dlg.chk_modo_formulario.setEnabled(tem_selecao)
        self.dlg.chk_modo_individual.setEnabled(tem_selecao)
        
        if tem_selecao:
            # Activate 'Form' mode by default if no mode is currently active
            if not self.dlg.chk_modo_formulario.isChecked() and not self.dlg.chk_modo_individual.isChecked():
                self.dlg.chk_modo_formulario.setChecked(True)
        else:
            # Clear modes if no attributes are selected
            self.dlg.chk_modo_formulario.setChecked(False)
            self.dlg.chk_modo_individual.setChecked(False)

    def verificar_estado_selecionar_todos(self):
        """Synchronize the master 'Select All' checkbox state."""
        container = self.dlg.scrollAreaWidgetContents
        cbs = container.findChildren(QCheckBox)
        total = len(cbs)
        marcados = sum(1 for cb in cbs if cb.isChecked())
        self.dlg.chk_selecionar_todos.blockSignals(True)
        self.dlg.chk_selecionar_todos.setChecked(total > 0 and total == marcados)
        self.dlg.chk_selecionar_todos.blockSignals(False)