# -*- coding: utf-8 -*-
"""
/***************************************************************************
 VecPluginDialog
                                 A QGIS plugin
 Integration with VEC inference service for building detection
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-12-09
        git sha              : $Format:%H$
        copyright            : (C) 2025 FieldWatch AI
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets, QtCore, QtGui
from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtGui import QDesktopServices
from qgis.core import QgsProject, QgsGeometry, QgsWkbTypes, QgsPointXY
from qgis.gui import QgsMapToolCapture, QgsRubberBand

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'vec_plugin_dialog_base.ui'))


class PolygonCaptureTool(QgsMapToolCapture):
    """Custom map tool for capturing polygon geometry."""
    
    finished = QtCore.pyqtSignal(QgsGeometry)
    
    def __init__(self, canvas, cad_dock_widget=None):
        super(PolygonCaptureTool, self).__init__(canvas, cad_dock_widget, QgsMapToolCapture.CapturePolygon)
        self.canvas = canvas
        self.points = []
        self.rubber_band = None
    
    def canvasReleaseEvent(self, event):
        """Handle mouse release events."""
        if event.button() == QtCore.Qt.RightButton:
            # Right click finishes the polygon
            if len(self.points) >= 3:
                # Create polygon from points
                polygon_points = self.points + [self.points[0]]  # Close polygon
                geometry = QgsGeometry.fromPolygonXY([polygon_points])
                self.finished.emit(geometry)
            else:
                # Not enough points
                self.finished.emit(QgsGeometry())
            self._cleanup_rubber_band()
            self.points = []
        elif event.button() == QtCore.Qt.LeftButton:
            # Left click adds a point
            point = self.toMapCoordinates(event.pos())
            self.points.append(QgsPointXY(point))
            # Draw temporary rubber band
            if self.rubber_band is None:
                self.rubber_band = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
                self.rubber_band.setColor(QtCore.Qt.red)
                self.rubber_band.setWidth(2)
            if len(self.points) > 1:
                self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
                for p in self.points:
                    self.rubber_band.addPoint(p, False)
                self.rubber_band.addPoint(self.points[0], True)  # Close polygon
                self.rubber_band.show()
    
    def _cleanup_rubber_band(self):
        """Clean up rubber band."""
        if self.rubber_band:
            self.rubber_band.reset()
            self.rubber_band = None
    
    def deactivate(self):
        """Clean up when tool is deactivated."""
        self._cleanup_rubber_band()
        self.points = []
        super(PolygonCaptureTool, self).deactivate()


class VecPluginDialog(QtWidgets.QDialog, FORM_CLASS):
    # Signal emitted when polygon is drawn
    polygon_drawn = QtCore.pyqtSignal(QgsGeometry)
    # Signal emitted when OK is clicked (but dialog stays open)
    processing_started = QtCore.pyqtSignal()
    
    def __init__(self, parent=None, iface=None):
        """Constructor."""
        super(VecPluginDialog, self).__init__(parent)
        self.iface = iface
        self.crop_geometry = None
        self.map_tool = None
        self.previous_map_tool = None
        
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        
        # Initialize license-related variables
        self.license_key = None
        self.jwt_token = None
        self.token_expiry = None
        
        # Populate the input layer combo box with raster layers
        self.populate_layers()
        
        # Set default output layer name
        self.outputLayerLineEdit.setText("Building_Detections")
        
        # Hide progress bar initially
        self.progressBar.setVisible(False)
        
        # Connect polygon drawing buttons
        self.drawPolygonButton.clicked.connect(self.start_polygon_drawing)
        self.clearPolygonButton.clicked.connect(self.clear_polygon)
        
        # Connect license validation button
        self.validateLicenseButton.clicked.connect(self.validate_license)
        
        # Connect Get License Key button
        self.getLicenseKeyButton.clicked.connect(self.open_fieldwatch_website)
        
        # Connect signal
        self.polygon_drawn.connect(self.on_polygon_drawn)
        
        # Disable OK button initially - polygon must be drawn first AND license must be validated
        ok_button = self.button_box.button(QtWidgets.QDialogButtonBox.Ok)
        if ok_button:
            ok_button.setEnabled(False)
    
    def populate_layers(self):
        """Populate the input layer combo box with available raster layers."""
        self.inputLayerCombo.clear()
        
        # Get all layers from the project
        layers = QgsProject.instance().mapLayers().values()
        
        # Filter for raster layers
        raster_layers = [layer for layer in layers if layer.type() == 1]  # 1 = raster layer
        
        if raster_layers:
            for layer in raster_layers:
                self.inputLayerCombo.addItem(layer.name(), layer)
            # Explicitly enable the combo box when layers are found
            self.inputLayerCombo.setEnabled(True)
        else:
            self.inputLayerCombo.addItem("No raster layers available", None)
            self.inputLayerCombo.setEnabled(False)
    
    def get_input_layer(self):
        """Get the selected input raster layer."""
        if self.inputLayerCombo.currentData():
            return self.inputLayerCombo.currentData()
        return None
    
    def get_output_layer_name(self):
        """Get the output layer name."""
        name = self.outputLayerLineEdit.text().strip()
        if not name:
            return "Building_Detections"
        return name
    
    def get_crop_geometry(self):
        """Get the crop polygon geometry."""
        return self.crop_geometry
    
    def get_license_key(self):
        """Get the license key."""
        return self.license_key
    
    def get_jwt_token(self):
        """Get the JWT token."""
        return self.jwt_token
    
    def validate_license(self):
        """Validate license key and get JWT token."""
        license_key = self.licenseKeyLineEdit.text().strip()
        
        if not license_key:
            QtWidgets.QMessageBox.warning(
                self,
                "License Required",
                "Please enter a license key."
            )
            return
        
        # Disable button during validation
        self.validateLicenseButton.setEnabled(False)
        self.validateLicenseButton.setText("Validating...")
        self.licenseStatusLabel.setText("Validating...")
        self.licenseStatusLabel.setStyleSheet("color: blue;")
        
        # Call validation endpoint
        try:
            from .vec_inference_client import VecInferenceClient
            # Use hardcoded service URL for auth endpoint
            INFERENCE_SERVICE_URL = "https://inference-service-proxy-xt7fd24wta-uc.a.run.app"
            temp_client = VecInferenceClient(INFERENCE_SERVICE_URL)
            token, expiry = temp_client.validate_license_key(license_key)
            
            if token:
                self.jwt_token = token
                self.token_expiry = expiry
                self.license_key = license_key
                self.licenseStatusLabel.setText("✓ Valid")
                self.licenseStatusLabel.setStyleSheet("color: green;")
                
                # Enable OK button if polygon is drawn
                ok_button = self.button_box.button(QtWidgets.QDialogButtonBox.Ok)
                if ok_button and self.crop_geometry:
                    ok_button.setEnabled(True)
            else:
                self.jwt_token = None
                self.token_expiry = None
                self.license_key = None
                self.licenseStatusLabel.setText("✗ Invalid")
                self.licenseStatusLabel.setStyleSheet("color: red;")
                QtWidgets.QMessageBox.warning(
                    self,
                    "Invalid License",
                    "The license key is invalid or expired."
                )
                # Disable OK button
                ok_button = self.button_box.button(QtWidgets.QDialogButtonBox.Ok)
                if ok_button:
                    ok_button.setEnabled(False)
        except Exception as e:
            self.jwt_token = None
            self.token_expiry = None
            self.license_key = None
            self.licenseStatusLabel.setText("✗ Error")
            self.licenseStatusLabel.setStyleSheet("color: red;")
            QtWidgets.QMessageBox.critical(
                self,
                "Validation Error",
                f"Failed to validate license: {str(e)}"
            )
            # Disable OK button
            ok_button = self.button_box.button(QtWidgets.QDialogButtonBox.Ok)
            if ok_button:
                ok_button.setEnabled(False)
        finally:
            self.validateLicenseButton.setEnabled(True)
            self.validateLicenseButton.setText("Validate")
    
    def open_fieldwatch_website(self):
        """Open FieldWatch website in the default browser."""
        QDesktopServices.openUrl(QUrl("https://usefieldwatch.com/"))
    
    def start_polygon_drawing(self):
        """Start polygon drawing on the map canvas."""
        if not self.iface:
            QtWidgets.QMessageBox.warning(
                self,
                "Error",
                "Cannot draw polygon: QGIS interface not available."
            )
            return
        
        canvas = self.iface.mapCanvas()
        if not canvas:
            return
        
        # Store previous map tool
        self.previous_map_tool = canvas.mapTool()
        
        # Create custom polygon capture tool
        cad_dock = self.iface.cadDockWidget() if hasattr(self.iface, 'cadDockWidget') else None
        self.map_tool = PolygonCaptureTool(canvas, cad_dock)
        self.map_tool.finished.connect(self.polygon_finished)
        
        # Set as active tool
        canvas.setMapTool(self.map_tool)
        
        # Update UI
        self.drawPolygonButton.setEnabled(False)
        self.cropStatusValueLabel.setText("Drawing polygon... Click on map to add points, right-click to finish")
        self.cropStatusValueLabel.setStyleSheet("color: blue;")
        
        # Hide dialog temporarily so user can see the map
        self.hide()
    
    def polygon_finished(self, geometry):
        """Called when polygon drawing is finished."""
        if geometry and not geometry.isEmpty():
            self.crop_geometry = geometry
            self.polygon_drawn.emit(geometry)
        else:
            self.clear_polygon()
        
        # Restore previous map tool
        if self.iface and self.previous_map_tool:
            self.iface.mapCanvas().setMapTool(self.previous_map_tool)
        
        # Show dialog again
        self.show()
        self.raise_()
        self.activateWindow()
    
    def on_polygon_drawn(self, geometry):
        """Handle polygon drawn signal."""
        if geometry:
            # Get area info
            area = geometry.area()
            # Convert to readable format (assuming CRS units)
            if area > 1000000:
                area_str = f"{area/1000000:.2f} km²"
            elif area > 10000:
                area_str = f"{area/10000:.2f} ha"
            else:
                area_str = f"{area:.2f} m²"
            
            self.cropStatusValueLabel.setText(f"Area selected: {area_str}")
            self.cropStatusValueLabel.setStyleSheet("color: green;")
            self.clearPolygonButton.setEnabled(True)
            self.drawPolygonButton.setEnabled(True)
            
            # Enable OK button now that polygon is drawn (only if license is validated)
            ok_button = self.button_box.button(QtWidgets.QDialogButtonBox.Ok)
            if ok_button and self.jwt_token:
                ok_button.setEnabled(True)
    
    def clear_polygon(self):
        """Clear the selected crop polygon."""
        self.crop_geometry = None
        self.cropStatusValueLabel.setText("No area selected - Please draw a polygon to continue")
        self.cropStatusValueLabel.setStyleSheet("color: red;")
        self.clearPolygonButton.setEnabled(False)
        self.drawPolygonButton.setEnabled(True)
        
        # Disable OK button when polygon is cleared
        ok_button = self.button_box.button(QtWidgets.QDialogButtonBox.Ok)
        if ok_button:
            ok_button.setEnabled(False)
    
    def accept(self):
        """Override accept to validate polygon is drawn and license is validated, then emit signal instead of closing."""
        if not self.jwt_token:
            QtWidgets.QMessageBox.warning(
                self,
                "License Required",
                "Please validate your license key before proceeding.\n\nEnter your license key and click 'Validate'."
            )
            return
        
        if not self.crop_geometry or self.crop_geometry.isEmpty():
            QtWidgets.QMessageBox.warning(
                self,
                "Polygon Required",
                "Please draw a polygon on the map before proceeding.\n\nClick 'Draw Polygon' and draw an area on the map canvas."
            )
            return
        
        # Emit signal to start processing (dialog stays open)
        self.processing_started.emit()
        
        # Restore previous map tool if drawing was active
        if self.map_tool and self.iface:
            canvas = self.iface.mapCanvas()
            if canvas.mapTool() == self.map_tool:
                if self.previous_map_tool:
                    canvas.setMapTool(self.previous_map_tool)
                else:
                    canvas.unsetMapTool(self.map_tool)
    
    def closeEvent(self, event):
        """Handle dialog close event."""
        # Clear polygon drawing if active
        if self.map_tool and self.iface:
            canvas = self.iface.mapCanvas()
            if canvas.mapTool() == self.map_tool:
                if self.previous_map_tool:
                    canvas.setMapTool(self.previous_map_tool)
                else:
                    canvas.unsetMapTool(self.map_tool)
        super(VecPluginDialog, self).closeEvent(event)
