# -*- coding: utf-8 -*-
"""
/***************************************************************************
 VecPlugin
                                 A QGIS plugin
 Integration with VEC inference service for building detection
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-12-09
        git sha              : $Format:%H$
        copyright            : (C) 2025 FieldWatch AI
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QThread, pyqtSignal, QTimer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QDialogButtonBox, QDialog
from qgis.core import (
    QgsVectorLayer, QgsProject, QgsMessageLog, 
    Qgis
)

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .vec_plugin_dialog import VecPluginDialog
from .vec_inference_client import VecInferenceClient
import os.path
import tempfile


class InferenceWorker(QThread):
    """Worker thread for running inference asynchronously."""
    
    finished = pyqtSignal(str)  # Shapefile path
    error = pyqtSignal(str)
    progress = pyqtSignal(int, str)
    
    def __init__(self, client, raster_layer, crop_geometry=None, crop_geometry_crs=None):
        QThread.__init__(self)
        self.client = client
        self.raster_layer = raster_layer
        self.crop_geometry = crop_geometry
        self.crop_geometry_crs = crop_geometry_crs
    
    def run(self):
        try:
            def progress_callback(value, message):
                self.progress.emit(value, message)
            
            shapefile_path = self.client.process_raster_layer(
                self.raster_layer,
                progress_callback,
                self.crop_geometry,
                crop_geometry_crs=self.crop_geometry_crs
            )
            
            self.finished.emit(shapefile_path)
        except Exception as e:
            self.error.emit(str(e))


class VecPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'VecPlugin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&FieldWatch Vectorizer')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('VecPlugin', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/vec_plugin/binocular.png'
        self.add_action(
            icon_path,
            text=self.tr(u'FieldWatch'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&FieldWatch Vectorizer'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = VecPluginDialog(iface=self.iface)

        # Populate layers in case new layers were added
        self.dlg.populate_layers()
        
        # Reset processing flag
        self._is_processing = False
        
        # Connect to processing_started signal (disconnect first to avoid duplicates)
        try:
            self.dlg.processing_started.disconnect()
        except:
            pass
        self.dlg.processing_started.connect(self._start_processing)

        # show the dialog (non-modal so it stays open during processing)
        self.dlg.show()
        # Run the dialog event loop
        self.dlg.exec_()
    
    def _start_processing(self):
        """Start the inference processing."""
        # Prevent multiple simultaneous processing
        if hasattr(self, '_is_processing') and self._is_processing:
            return
        
        try:
            self._is_processing = True
            
            # Disconnect signal to prevent restart
            try:
                self.dlg.processing_started.disconnect()
            except:
                pass
            
            # Get values from dialog
            input_layer = self.dlg.get_input_layer()
            output_name = self.dlg.get_output_layer_name()
            crop_geometry = self.dlg.get_crop_geometry()
            jwt_token = self.dlg.get_jwt_token()
            
            # Validate license is validated
            if not jwt_token:
                self.iface.messageBar().pushMessage(
                    "Error",
                    "Please validate your license key before processing.",
                    level=1,
                    duration=5
                )
                self._is_processing = False
                return
            
            if not input_layer:
                self.iface.messageBar().pushMessage(
                    "Error", 
                    "Please select an input raster layer.",
                    level=1,  # Warning level
                    duration=3
                )
                self._is_processing = False
                return
            
            # Validate polygon is drawn (mandatory)
            if not crop_geometry or crop_geometry.isEmpty():
                self.iface.messageBar().pushMessage(
                    "Error",
                    "Please draw a polygon on the map before processing.",
                    level=1,
                    duration=5
                )
                self._is_processing = False
                return
            
            # Hardcoded service URLs
            INFERENCE_SERVICE_URL = "https://inference.usefieldwatch.com/"
            UPLOAD_SERVICE_URL = "https://upload.usefieldwatch.com/"
            
            # Initialize inference client with hardcoded URLs and JWT token
            try:
                client = VecInferenceClient(INFERENCE_SERVICE_URL, upload_url=UPLOAD_SERVICE_URL, jwt_token=jwt_token)
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    "Error",
                    "Failed to initialize inference client",
                    level=2,
                    duration=5
                )
                self._is_processing = False
                return
            
            # Disable OK/Cancel buttons during processing
            ok_button = self.dlg.button_box.button(QDialogButtonBox.Ok)
            cancel_button = self.dlg.button_box.button(QDialogButtonBox.Cancel)
            if ok_button:
                ok_button.setEnabled(False)
            if cancel_button:
                cancel_button.setEnabled(False)
            
            # Show progress bar
            self.dlg.progressBar.setVisible(True)
            self.dlg.progressBar.setValue(0)
            self.dlg.statusLabel.setText("Processing...")
            
            # Store output_name and input_layer for use in callbacks
            self._processing_output_name = output_name
            self._processing_input_layer = input_layer
            
            # Get canvas CRS for crop geometry (polygon was drawn on canvas)
            canvas_crs = None
            if self.iface and self.iface.mapCanvas():
                project = QgsProject.instance()
                canvas_crs = project.crs() if project else None
            
            # Create worker thread
            self.worker = InferenceWorker(client, input_layer, crop_geometry, crop_geometry_crs=canvas_crs)
            self.worker.progress.connect(self._on_progress)
            self.worker.finished.connect(self._on_finished)
            self.worker.error.connect(self._on_error)
            
            self.worker.start()
            
        except Exception as e:
            # Clean up any partially created worker
            if hasattr(self, 'worker') and self.worker:
                try:
                    self.worker.progress.disconnect()
                    self.worker.finished.disconnect()
                    self.worker.error.disconnect()
                except:
                    pass
                self.worker = None
            
            self._is_processing = False
            
            # Re-enable buttons
            ok_button = self.dlg.button_box.button(QDialogButtonBox.Ok)
            cancel_button = self.dlg.button_box.button(QDialogButtonBox.Cancel)
            if ok_button:
                ok_button.setEnabled(True)
            if cancel_button:
                cancel_button.setEnabled(True)
            
            # Reconnect processing signal
            try:
                self.dlg.processing_started.disconnect()
            except:
                pass
            self.dlg.processing_started.connect(self._start_processing)
            
            self.iface.messageBar().pushMessage(
                "Error",
                f"Unexpected error occurred: {str(e)[:200]}",
                level=2,
                duration=10
            )
    
    def _on_progress(self, value, message):
        """Update progress bar."""
        self.dlg.progressBar.setValue(value)
        self.dlg.statusLabel.setText(message)
    
    def _on_finished(self, shapefile_path):
        """Handle successful inference completion."""
        try:
            output_name = self._processing_output_name
            input_layer = self._processing_input_layer
            
            # Create vector layer from shapefile
            vector_layer = self._create_vector_layer(shapefile_path, output_name, input_layer.crs())
            
            # Add layer to QGIS
            QgsProject.instance().addMapLayer(vector_layer)
            
            # Zoom to layer
            self.iface.mapCanvas().setExtent(vector_layer.extent())
            self.iface.mapCanvas().refresh()
            
            # Count features
            feature_count = vector_layer.featureCount()
            
            self.iface.messageBar().pushMessage(
                "Success", 
                f"Created layer '{output_name}' with {feature_count} building(s) detected.",
                level=0,
                duration=5
            )
            
            self.dlg.progressBar.setVisible(False)
            self.dlg.statusLabel.setText("Complete!")
            
            # Re-enable buttons
            ok_button = self.dlg.button_box.button(QDialogButtonBox.Ok)
            cancel_button = self.dlg.button_box.button(QDialogButtonBox.Cancel)
            if ok_button:
                ok_button.setEnabled(True)
            if cancel_button:
                cancel_button.setEnabled(True)
            
            # Reset processing flag
            self._is_processing = False
            
            # Close dialog after a short delay (use done() to avoid triggering accept() signal)
            QTimer.singleShot(1000, lambda: self.dlg.done(QDialog.Accepted))
            
        except Exception as e:
            # Ensure cleanup happens
            if hasattr(self, 'worker') and self.worker:
                try:
                    self.worker.progress.disconnect()
                    self.worker.finished.disconnect()
                    self.worker.error.disconnect()
                except:
                    pass
            self._is_processing = False
            self._on_error(str(e))
    
    def _on_error(self, error_message):
        """Handle inference errors."""
        try:
            # Clean up worker thread
            if hasattr(self, 'worker') and self.worker:
                try:
                    # Disconnect all signals first
                    self.worker.progress.disconnect()
                    self.worker.finished.disconnect()
                    self.worker.error.disconnect()
                except:
                    pass
                
                # Wait for thread to finish (with timeout)
                if self.worker.isRunning():
                    self.worker.wait(1000)  # Wait up to 1 second
                    if self.worker.isRunning():
                        # Force terminate if still running
                        self.worker.terminate()
                        self.worker.wait(500)
                
                # Clean up reference
                self.worker = None
            
            # Reset processing flag
            self._is_processing = False
            
            # Reset UI
            self.dlg.progressBar.setVisible(False)
            self.dlg.statusLabel.setText(f"Error: {error_message[:100]}")
            
            # Show error message
            self.iface.messageBar().pushMessage(
                "Error",
                f"Inference failed: {error_message[:200]}",
                level=2,  # Critical
                duration=10
            )
            
            # Re-enable buttons
            ok_button = self.dlg.button_box.button(QDialogButtonBox.Ok)
            cancel_button = self.dlg.button_box.button(QDialogButtonBox.Cancel)
            if ok_button:
                ok_button.setEnabled(True)
            if cancel_button:
                cancel_button.setEnabled(True)
            
            # Reconnect processing signal to allow retry
            try:
                self.dlg.processing_started.disconnect()
            except:
                pass
            self.dlg.processing_started.connect(self._start_processing)
            
        except Exception as e:
            # Last resort cleanup
            self._is_processing = False
            # Sanitize URLs from error messages
            error_msg = str(e)
            import re
            url_pattern = r'https?://[^\s<>"{}|\\^`\[\]]+'
            sanitized_error = re.sub(url_pattern, '[REDACTED]', error_msg)
            QgsMessageLog.logMessage(
                f"Error in error handler: {sanitized_error}",
                "VEC Plugin",
                Qgis.Critical
            )
    
    def _create_vector_layer(self, shapefile_path, layer_name, crs):
        """
        Create a QGIS vector layer from shapefile path.
        
        :param shapefile_path: Path to shapefile (.shp file)
        :type shapefile_path: str
        :param layer_name: Name for the new layer
        :type layer_name: str
        :param crs: Coordinate reference system
        :type crs: QgsCoordinateReferenceSystem
        :returns: QGIS vector layer
        :rtype: QgsVectorLayer
        """
        # Create vector layer directly from shapefile
        vector_layer = QgsVectorLayer(
            shapefile_path,
            layer_name,
            "ogr"
        )
        
        if not vector_layer.isValid():
            raise Exception(f"Failed to create vector layer from shapefile: {shapefile_path}")
        
        # Set CRS
        vector_layer.setCrs(crs)
        
        return vector_layer
