# -*- coding: utf-8 -*-
import os
import csv
from qgis.PyQt import QtWidgets, QtGui, QtCore
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsProject,
    QgsPointXY,
)

try:
    import pandas as pd  # optional for Excel support
    PANDAS_AVAILABLE = True
except Exception:
    PANDAS_AVAILABLE = False

class LogConsole(QtWidgets.QDockWidget):
    SETTINGS_GEOM = 'utm_converter_dock/geometry'
    SETTINGS_VISIBLE = 'utm_converter_dock/visible'

    def __init__(self, parent=None):
        super().__init__('UTM Converter Console', parent)
        self.setObjectName('UTMConverterConsoleDock')
        self.setAllowedAreas(QtCore.Qt.AllDockWidgetAreas)
        widget = QtWidgets.QWidget()
        self.setWidget(widget)
        v = QtWidgets.QVBoxLayout(widget)
        # toolbar with controls
        toolbar = QtWidgets.QHBoxLayout()
        self.btn_clear = QtWidgets.QPushButton('Clear Logs')
        self.btn_close = QtWidgets.QPushButton('Hide Console')
        toolbar.addWidget(self.btn_clear)
        toolbar.addStretch(1)
        toolbar.addWidget(self.btn_close)
        v.addLayout(toolbar)
        # log view (rich text)
        self.log_view = QtWidgets.QTextEdit()
        self.log_view.setReadOnly(True)
        self.log_view.setAcceptRichText(True)
        v.addWidget(self.log_view)
        # autoscroll toggle
        h2 = QtWidgets.QHBoxLayout()
        self.autoscroll_cb = QtWidgets.QCheckBox('Auto-scroll')
        self.autoscroll_cb.setChecked(True)
        h2.addWidget(self.autoscroll_cb)
        h2.addStretch(1)
        v.addLayout(h2)

        self.btn_clear.clicked.connect(self.clear)
        self.btn_close.clicked.connect(self.hide)

    def log(self, level, message):
        # level: 'INFO', 'WARN', 'ERROR'
        color = {'INFO':'#0a8f08', 'WARN':'#b58c00', 'ERROR':'#b00000'}.get(level, '#000000')
        html = f'<div><span style="color:{color};font-weight:bold">[{level}]</span> {QtCore.QCoreApplication.translate("UTM","")}{message}</div>'
        self.log_view.append(html)
        if self.autoscroll_cb.isChecked():
            self.log_view.verticalScrollBar().setValue(self.log_view.verticalScrollBar().maximum())

    def clear(self):
        self.log_view.clear()

    def save_settings(self):
        settings = QtCore.QSettings()
        settings.setValue(self.SETTINGS_GEOM, self.saveGeometry())
        settings.setValue(self.SETTINGS_VISIBLE, self.isVisible())

    def restore_settings(self):
        settings = QtCore.QSettings()
        geom = settings.value(self.SETTINGS_GEOM)
        if isinstance(geom, (bytes, bytearray)):
            try:
                self.restoreGeometry(geom)
            except Exception:
                pass
        visible = settings.value(self.SETTINGS_VISIBLE)
        if visible in (True, 'true', 'True', '1'):
            self.show()
        else:
            self.hide()

class UTMConverterDialog(QtWidgets.QDialog):
    def __init__(self, parent=None, console=None):
        super().__init__(parent)
        self.setWindowTitle('UTM Converter — Dock version')
        self.resize(700, 420)
        self.console = console
        layout = QtWidgets.QVBoxLayout(self)

        tabs = QtWidgets.QTabWidget()
        layout.addWidget(tabs)

        # Single tab
        single = QtWidgets.QWidget(); tabs.addTab(single, 'Single')
        f = QtWidgets.QFormLayout(single)
        self.lat_edit = QtWidgets.QLineEdit(); self.lat_edit.setPlaceholderText('e.g., 9.12345')
        self.lon_edit = QtWidgets.QLineEdit(); self.lon_edit.setPlaceholderText('e.g., 7.54321')
        f.addRow('Latitude (deg):', self.lat_edit)
        f.addRow('Longitude (deg):', self.lon_edit)
        btn_h = QtWidgets.QHBoxLayout()
        self.btn_to_utm = QtWidgets.QPushButton('Convert → UTM')
        self.btn_to_ll = QtWidgets.QPushButton('UTM → Lat/Lon')
        btn_h.addWidget(self.btn_to_utm); btn_h.addWidget(self.btn_to_ll)
        f.addRow(btn_h)
        self.result_label = QtWidgets.QLabel('Result:')
        f.addRow(self.result_label)

        # Reverse inputs for UTM->LL
        self.easting_edit = QtWidgets.QLineEdit(); self.northing_edit = QtWidgets.QLineEdit(); self.zone_edit = QtWidgets.QLineEdit()
        f.addRow('Easting (m):', self.easting_edit)
        f.addRow('Northing (m):', self.northing_edit)
        f.addRow('Zone (e.g., 32N):', self.zone_edit)

        # Batch tab
        batch = QtWidgets.QWidget(); tabs.addTab(batch, 'Batch')
        bv = QtWidgets.QVBoxLayout(batch)
        self.batch_hint = QtWidgets.QLabel("Batch: select CSV/.xlsx with 'lat' & 'lon' or 'latitude' & 'longitude' columns.")
        self.batch_hint.setWordWrap(True)
        bv.addWidget(self.batch_hint)
        bh = QtWidgets.QHBoxLayout()
        self.btn_select_file = QtWidgets.QPushButton('Select File')
        self.lbl_file = QtWidgets.QLabel('No file selected')
        bh.addWidget(self.btn_select_file); bh.addWidget(self.lbl_file)
        bv.addLayout(bh)
        self.btn_run_batch = QtWidgets.QPushButton('Run Batch Conversion')
        bv.addWidget(self.btn_run_batch)
        self.batch_out = QtWidgets.QPlainTextEdit(); self.batch_out.setReadOnly(True)
        bv.addWidget(self.batch_out)

        # Bottom controls
        bottom = QtWidgets.QHBoxLayout()
        self.btn_toggle_console = QtWidgets.QPushButton('Toggle Console')
        bottom.addWidget(self.btn_toggle_console)
        bottom.addStretch(1)
        layout.addLayout(bottom)

        # Connections
        self.btn_to_utm.clicked.connect(self.convert_to_utm)
        self.btn_to_ll.clicked.connect(self.convert_to_latlon)
        self.btn_select_file.clicked.connect(self.select_file)
        self.btn_run_batch.clicked.connect(self.run_batch)
        self.btn_toggle_console.clicked.connect(self.toggle_console)

        self.selected_file = None

    def log(self, level, msg):
        if self.console:
            self.console.log(level, msg)
        else:
            # fallback: show in batch_out
            prefix = f'[{level}] '
            self.batch_out.appendPlainText(prefix + msg)

    def validate_latlon(self, lat_s, lon_s):
        try:
            lat = float(lat_s); lon = float(lon_s)
        except Exception:
            raise ValueError('lat/lon must be numeric')
        if not (-90 <= lat <= 90 and -180 <= lon <= 180):
            raise ValueError('lat/lon out of range')
        return lat, lon

    def lon_to_zone(self, lon):
        return max(1, min(60, int((lon + 180) / 6) + 1))

    def build_crs(self, zone, northern=True):
        epsg = 32600 + int(zone) if northern else 32700 + int(zone)
        return QgsCoordinateReferenceSystem(epsg)

    def convert_to_utm(self):
        try:
            lat, lon = self.validate_latlon(self.lat_edit.text().strip(), self.lon_edit.text().strip())
        except Exception as e:
            QtWidgets.QMessageBox.warning(self, 'Input error', str(e))
            self.log('ERROR', f'Single conversion input error: {e}')
            return
        zone = self.lon_to_zone(lon)
        northern = lat >= 0
        try:
            src = QgsCoordinateReferenceSystem(4326)
            tgt = self.build_crs(zone, northern)
            xform = QgsCoordinateTransform(src, tgt, QgsProject.instance())
            pt = xform.transform(QgsPointXY(lon, lat))
            zone_str = f"{zone}{'N' if northern else 'S'}"
            self.result_label.setText(f'Easting={pt.x():.3f}, Northing={pt.y():.3f}, Zone={zone_str}')
            self.log('INFO', f'Converted lat={lat}, lon={lon} -> easting={pt.x():.3f}, northing={pt.y():.3f}, zone={zone_str}')
        except Exception as e:
            QtWidgets.QMessageBox.critical(self, 'Transform error', str(e))
            self.log('ERROR', f'Transform failed: {e}')

    def convert_to_latlon(self):
        try:
            easting = float(self.easting_edit.text().strip())
            northing = float(self.northing_edit.text().strip())
        except Exception as e:
            QtWidgets.QMessageBox.warning(self, 'Input error', 'Easting/Northing must be numeric')
            self.log('ERROR', f'UTM->LL input error: {e}')
            return
        zone_text = self.zone_edit.text().strip()
        if not zone_text:
            QtWidgets.QMessageBox.warning(self, 'Input error', 'Zone required (e.g., 32N)')
            self.log('ERROR', 'UTM->LL zone missing')
            return
        zone_digits = ''.join(ch for ch in zone_text if ch.isdigit())
        hemi = ''.join(ch for ch in zone_text if ch.isalpha()).upper()
        if not zone_digits:
            QtWidgets.QMessageBox.warning(self, 'Input error', 'Zone number not recognized')
            self.log('ERROR', 'Zone parse error')
            return
        zone = int(zone_digits); northern = 'S' not in hemi
        try:
            src = self.build_crs(zone, northern)
            tgt = QgsCoordinateReferenceSystem(4326)
            xform = QgsCoordinateTransform(src, tgt, QgsProject.instance())
            ll = xform.transform(QgsPointXY(easting, northing))
            self.result_label.setText(f'Lat={ll.y():.8f}, Lon={ll.x():.8f}')
            self.log('INFO', f'Converted easting={easting}, northing={northing}, zone={zone}{"N" if northern else "S"} -> lat={ll.y():.8f}, lon={ll.x():.8f}')
        except Exception as e:
            QtWidgets.QMessageBox.critical(self, 'Transform error', str(e))
            self.log('ERROR', f'UTM->LL transform failed: {e}')

    def select_file(self):
        path, _ = QtWidgets.QFileDialog.getOpenFileName(self, 'Open CSV/XLSX', '', 'Data files (*.csv *.xlsx *.xls);;CSV (*.csv);;Excel (*.xlsx *.xls)')
        if path:
            self.selected_file = path
            self.lbl_file.setText(os.path.basename(path))
            self.log('INFO', f'Selected file: {path}')

    def run_batch(self):
        if not self.selected_file:
            QtWidgets.QMessageBox.warning(self, 'No file', 'Please select a file first')
            return
        path = self.selected_file
        self.batch_out.clear()
        self.log('INFO', f'Starting batch conversion for {path}')
        ext = os.path.splitext(path)[1].lower()
        rows = []
        headers = []
        try:
            if ext == '.csv':
                with open(path, newline='', encoding='utf-8') as f:
                    r = csv.DictReader(f)
                    headers = r.fieldnames or []
                    for i,row in enumerate(r, start=1):
                        rows.append((i, row))
            elif ext in ('.xlsx', '.xls'):
                if not PANDAS_AVAILABLE:
                    QtWidgets.QMessageBox.information(self, 'Excel support missing', 'Install pandas and openpyxl to use Excel batch mode, or save as CSV.')
                    self.log('WARN', 'Excel batch requested but pandas not available')
                    return
                df = pd.read_excel(path)
                headers = list(df.columns)
                for i, rec in enumerate(df.to_dict(orient='records'), start=1):
                    rows.append((i, rec))
            else:
                QtWidgets.QMessageBox.warning(self, 'Unsupported', f'Unsupported file type: {ext}')
                self.log('ERROR', f'Unsupported extension {ext}')
                return
        except Exception as e:
            QtWidgets.QMessageBox.critical(self, 'Read error', str(e))
            self.log('ERROR', f'Failed to read file: {e}')
            return

        # find lat/lon columns
        lower = [h.lower() for h in headers]
        lat_key = None; lon_key = None
        for cand in ('lat','latitude'):
            if cand in lower:
                lat_key = headers[lower.index(cand)]; break
        for cand in ('lon','longitude','lng','long'):
            if cand in lower:
                lon_key = headers[lower.index(cand)]; break
        if not lat_key or not lon_key:
            QtWidgets.QMessageBox.warning(self, 'Columns not found', "Could not find 'lat'/'lon' columns.")
            self.log('ERROR', "lat/lon columns not found in batch file")
            return

        out_rows = []
        for idx, row in rows:
            try:
                lat = float(row.get(lat_key) if isinstance(row, dict) else row[lat_key])
                lon = float(row.get(lon_key) if isinstance(row, dict) else row[lon_key])
            except Exception as e:
                msg = f'Row {idx}: invalid lat/lon: {e}'
                self.batch_out.appendPlainText(msg); self.log('WARN', msg)
                out_rows.append({**(row if isinstance(row, dict) else row), 'utm_easting':'', 'utm_northing':'', 'utm_zone':'', 'error':'invalid lat/lon'})
                continue
            try:
                zone = self.lon_to_zone(lon)
                northern = lat >= 0
                src = QgsCoordinateReferenceSystem(4326)
                tgt = self.build_crs(zone, northern)
                xform = QgsCoordinateTransform(src, tgt, QgsProject.instance())
                pt = xform.transform(QgsPointXY(lon, lat))
                zone_str = f"{zone}{'N' if northern else 'S'}"
                msg = f'Row {idx}: converted -> Easting={pt.x():.3f}, Northing={pt.y():.3f}, Zone={zone_str}'
                self.batch_out.appendPlainText(msg); self.log('INFO', msg)
                r = dict(row) if isinstance(row, dict) else dict(row)
                r.update({'utm_easting':f'{pt.x():.3f}', 'utm_northing':f'{pt.y():.3f}', 'utm_zone':zone_str, 'error':''})
                out_rows.append(r)
            except Exception as e:
                msg = f'Row {idx}: transform error: {e}'
                self.batch_out.appendPlainText(msg); self.log('ERROR', msg)
                r = dict(row) if isinstance(row, dict) else dict(row)
                r.update({'utm_easting':'', 'utm_northing':'', 'utm_zone':'', 'error':str(e)})
                out_rows.append(r)

        # write output next to input
        out_path = os.path.splitext(path)[0] + '_utm.csv'
        try:
            with open(out_path, 'w', newline='', encoding='utf-8') as f:
                writer = csv.DictWriter(f, fieldnames=list(out_rows[0].keys()))
                writer.writeheader()
                for r in out_rows:
                    writer.writerow(r)
            self.log('INFO', f'Batch finished. Output: {out_path}')
            QtWidgets.QMessageBox.information(self, 'Done', f'Batch conversion complete. Output: {out_path}')
        except Exception as e:
            self.log('ERROR', f'Failed to write output: {e}')
            QtWidgets.QMessageBox.critical(self, 'Write error', str(e))

    def toggle_console(self):
        parent = self.parent()
        if not parent:
            return
        dock = parent.findChild(QtWidgets.QDockWidget, 'UTMConverterConsoleDock')
        if dock:
            dock.setVisible(not dock.isVisible())

class UTMConverterPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.iface_main = iface.mainWindow()
        self.dock = None
        self.dialog = None
        self.action = None

    def initGui(self):
        icon_path = os.path.join(os.path.dirname(__file__), 'icon.svg')
        icon = QtGui.QIcon(icon_path) if os.path.exists(icon_path) else QtGui.QIcon()
        self.action = QtWidgets.QAction(icon, 'UTM Converter (dock)', self.iface_main)
        self.action.triggered.connect(self.run)
        self.iface.addPluginToMenu('&UTM Converter', self.action)
        self.iface.addToolBarIcon(self.action)

    def unload(self):
        try:
            if self.action:
                self.iface.removePluginMenu('&UTM Converter', self.action)
                self.iface.removeToolBarIcon(self.action)
            if self.dock:
                self.iface_main.removeDockWidget(self.dock)
        except Exception:
            pass

    def run(self):
        # create dock if not exists
        if not self.dock:
            self.dock = LogConsole(self.iface_main)
            self.iface_main.addDockWidget(QtCore.Qt.RightDockWidgetArea, self.dock)
            self.dock.restore_settings()
        # create dialog and pass dock for logging
        if not self.dialog:
            self.dialog = UTMConverterDialog(parent=self.iface_main, console=self.dock)
        self.dialog.show()
        self.dialog.raise_()
        self.dialog.activateWindow()

    def unload(self):
        # duplicate unload for safety
        try:
            if self.action:
                self.iface.removePluginMenu('&UTM Converter', self.action)
                self.iface.removeToolBarIcon(self.action)
            if self.dock:
                # save settings
                self.dock.save_settings()
                self.iface_main.removeDockWidget(self.dock)
        except Exception:
            pass
