# -*- coding: utf-8 -*-
"""
/***************************************************************************
 UsgsStreamMapper
                                 A QGIS plugin
 Map upstream/downstream tributaries and basins using USGS NWIS station ids
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-04-21
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Austin Raney
        email                : aaraney@crimson.ua.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.core import (
    Qgis,
    QgsProject,
    QgsVectorLayer,
    )
from qgis.PyQt.QtCore import (
    QSettings,
    QTranslator,
    QCoreApplication,
    )
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .usgs_stream_mapper_dialog import UsgsStreamMapperDialog
import os.path
# Import for layer coloring
from PyQt5.QtGui import QColor


class UsgsStreamMapper:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'UsgsStreamMapper_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Usgs Stream Mapper')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('UsgsStreamMapper', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/usgs_stream_mapper/resources/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Usgs Stream Mapper'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Usgs Stream Mapper'),
                action)
            self.iface.removeToolBarIcon(action)
    
    def get_navigation_type(self):
        """
        Return the current selected navigation type. Return lowercase verion
        with spaces replaced with _.
        """
        navigation_type = self.dlg.combo_box_navigation_type.currentText()
        return navigation_type.lower().replace(' ', '_')


    def get_usgs_site_number(self):
        """Retrieve usgs site number from QLineEdit"""
        text = self.dlg.input_usgs_site_number.text()
        if len(text) == 8:
            return text
        else:
            raise ValueError('Not a valid USGS station number')

    @staticmethod
    def base_station_url(station_id):
        """Return USGS stations prepended with NLDI service end point"""
        return f'https://labs.waterdata.usgs.gov/api/nldi/linked-data/nwissite/USGS-{station_id}'

    @staticmethod
    def navigate(base_station_url, navigation_type):
        """Map a navigation type to a given base url

        navigation_type:
            str: basin, downstream_diversions, downstream_main,
            upstream_main, upstream_tributaries
        """

        suffix_map = {
            'basin': 'basin',
            'downstream_diversions': 'navigate/DD',
            'downstream_main': 'navigate/DM',
            'upstream_main': 'navigate/UM',
            'upstream_tributaries': 'navigate/UT',
        }

        try:
            suffix = suffix_map[navigation_type]
            return f"{base_station_url.rstrip('/')}/{suffix}"

        except KeyError:
            raise KeyError(
                'Acceptable keys are: basin, downstream_diversions, '
                + 'downstream_main, upstream_main, upstream_tributaries.'
                )

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = UsgsStreamMapperDialog()

        # bring widget to focus if already on screen
        if self.dlg.isVisible():
            self.dlg.raise_()
            return

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Retrieve usgs station number and navigation type
            site_number = self.get_usgs_site_number()
            navigation_type = self.get_navigation_type()

            # Get layer nldi url
            base_station_url = UsgsStreamMapper.base_station_url(site_number)
            layer_url = UsgsStreamMapper.navigate(base_station_url, navigation_type)

            # Load layer into QGIS
            usgs_layer = QgsVectorLayer(layer_url, f'{site_number}: {navigation_type}')

            if not usgs_layer.isValid():
                error_subject = "Error"
                error_message = (f"Failed to load layer data for USGS NWIS site {site_number!r}. "
                                 f"Verify {site_number!r} is a valid USGS NWIS site and you "
                                 "have an internet connection.")
                self.iface.messageBar().pushMessage(error_subject, error_message, level=Qgis.Critical)
                return

            # Add layer styling
            if navigation_type == 'basin':
                usgs_layer.renderer().symbol().setColor(QColor('green'))
                usgs_layer.renderer().symbol().setOpacity(0.5)
            else:
                usgs_layer.renderer().symbol().setColor(QColor('blue'))
                usgs_layer.renderer().symbol().setWidth(0.5)

            # Add layer to map
            QgsProject.instance().addMapLayer(usgs_layer)
