# -*- coding: utf-8 -*-
"""
/***************************************************************************
 UldkGugik
                                 A QGIS plugin
 Wtyczka pozwala na pobieranie geometrii granic działek katastralnych, obrębów, gmin, powiatów i województw.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-05-31
        git sha              : $Format:%H$
        copyright            : (C) 2019 by EnviroSolutions Sp. z o.o.
        email                : office@envirosolutions.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import (QSettings, QTranslator, qVersion, 
                              QCoreApplication, QVariant, Qt)
from qgis.PyQt.QtGui import QIcon, QPixmap, QKeySequence
from qgis.PyQt.QtWidgets import QAction, QToolBar, QShortcut, QWidget, QLabel, QDialog
from qgis.gui import QgsMessageBar, QgsMapToolEmitPoint, QgsDockWidget
from qgis.core import (Qgis, QgsVectorLayer, QgsGeometry, QgsFeature, QgsProject, QgsField,
                       QgsCoordinateReferenceSystem, QgsPoint, QgsCoordinateTransform,
                       QgsSettings, QgsNetworkAccessManager)
from qgis.PyQt.QtCore import QUrl, QEventLoop
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkReply

# Kompatybilność Qt5/Qt6 - w Qt5 metoda to exec_(), w Qt6 to exec()
if not hasattr(QEventLoop, 'exec'):
    QEventLoop.exec = QEventLoop.exec_

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .uldk_gugik_dialog import UldkGugikDialog
from .uldk_gugik_dialog_parcel import UldkGugikDialogParcel
from .uldk import RegionFetch
from .qgis_feed import QgisFeedDialog
import os.path
from . import utils, uldk_api, uldk_xy, uldk_parcel
from .constants import (
    DEFAULT_SRID, ENV_MENU_NAME, SWAP_XY_SRIDS, TOOLTIP_FROM_MAP,
    OBJECT_TYPES, ULDK_MIN_LINE_LEN,
)
from .utils import VersionUtils

from . import PLUGIN_VERSION as plugin_version
from . import PLUGIN_NAME as plugin_name
from .utils import MessageUtils


class UldkGugik:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.settings = QgsSettings()
        self.iface = iface

        if Qgis.QGIS_VERSION_INT >= 31000:
            from .qgis_feed import QgisFeed

            #qgis feed
            self.selected_industry = self.settings.value("selected_industry", None)
            show_dialog = self.settings.value("showDialog", True, type=bool)
            
            if self.selected_industry is None and show_dialog:
                self.showBranchSelectionDialog()
        
            select_indust_session = self.settings.value('selected_industry')
            
            self.feed = QgisFeed(selected_industry=select_indust_session, plugin_name=plugin_name)
            self.feed.initFeed()

        #DialogOnTop

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'UldkGugik_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(f"&{ENV_MENU_NAME}")

        #toolbar
        toolbar_name = ENV_MENU_NAME
        self.toolbar = self.iface.mainWindow().findChild(QToolBar, toolbar_name)
        if not self.toolbar:
            self.toolbar = self.iface.addToolBar(toolbar_name)
            self.toolbar.setObjectName(toolbar_name)

        self.shortcut = None
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        self.canvas = self.iface.mapCanvas()
        
        # out click tool will emit a QgsPoint on every click
        self.click_tool = QgsMapToolEmitPoint(self.canvas)
        self.click_tool.canvasClicked.connect(self.canvasClicked)

        self.dlg = UldkGugikDialog()
        self.region_name = None
        self.project = QgsProject.instance()
        

    def showBranchSelectionDialog(self):
        """Wyświetla dialog wyboru branży dla QGIS Feed."""
        dialog = QgisFeedDialog(self.iface.mainWindow())
        dialog.exec()
        self.selected_industry = self.settings.value("selected_industry", None)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('UldkGugik', message)

    def addAction(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            #self.iface.addToolBarIcon(action)
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.tr(f"&{ENV_MENU_NAME}"),
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""


        icon_path = os.path.join(self.plugin_dir, 'images', 'uldk.svg')
        self.addAction(
            icon_path,
            text=self.tr(u'Usługa Lokalizacji Działek Katastralnych (ULDK)'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True
        #self.dock = UldkGugikDialog()
        #self.iface.addDockWidget(Qt.RigthDockWidgetArea, self.dock)


        # Inicjacja grafik
        self.dlg.img_tab2.setPixmap(QPixmap(os.path.join(self.plugin_dir, 'images', 'coords.png')))

        # rozmiar okna
        self.dlg.setFixedSize(self.dlg.size())

        # informacje o wersji
        self.dlg.setWindowTitle('%s %s' % (plugin_name, plugin_version))
        self.dlg.lbl_pluginVersion.setText('%s %s' % (plugin_name, plugin_version))

        self.dlg.btn_download_tab1.clicked.connect(self.btnDownloadTab1Clicked)
        self.dlg.btn_download_tab2.clicked.connect(self.btnDownloadTab2Clicked)
        self.dlg.btn_download_tab3.clicked.connect(self.btnDownloadTab3Clicked)
        self.dlg.btn_search_tab3.clicked.connect(self.btnSearchTab3Clicked)
        
        self.dlg.btn_frommap.clicked.connect(self.btnFrommapClicked)
        self.dlg.btn_frommap.setToolTip(TOOLTIP_FROM_MAP)


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(ENV_MENU_NAME),
                action)
            # self.iface.removeToolBarIcon(action)
            self.toolbar.removeAction(action)

    def run(self):
        """Otwarcie okna wtyczki"""
        try:
            srid = self.project.crs().postgisSrid()
        except IndexError:
            MessageUtils.pushLogWarning(
                "Projekt QGIS nie posiada zdefiniowanego układu współrzędnych."
                "W celu dalszej pracy zdefiniuj układ współrzędnych dla projektu"
            )
            return 1
        if self.first_start == True:
            self.first_start = False
            # skrot klawiszowy
            self.shortcut = QShortcut(self.iface.mainWindow())
            # Qt6 compatibility: use "Alt+F" string format instead of Qt.ALT + Qt.Key_F
            self.shortcut.setKey(QKeySequence("Alt+F"))
            if VersionUtils.isQt6():
                self.shortcut.setContext(Qt.ShortcutContext.ApplicationShortcut)
            else:
                self.shortcut.setContext(Qt.ApplicationShortcut)
            self.shortcut.activated.connect(self.shortcutActivated)
        
        try:
            url = QUrl('https://uldk.gugik.gov.pl')
            request = QNetworkRequest(url)
            if hasattr(QNetworkRequest, 'KnownHeaders'):
                ua_header = QNetworkRequest.KnownHeaders.UserAgentHeader
            else:
                ua_header = QNetworkRequest.UserAgentHeader
            request.setHeader(ua_header, f"QGIS-Plugin-{plugin_name}")

            manager = QgsNetworkAccessManager.instance()
            reply = manager.get(request)

            loop = QEventLoop()
            reply.finished.connect(loop.quit)
            loop.exec()

            error_val = reply.error()

            if hasattr(QNetworkReply, 'NetworkError'):
                no_err = QNetworkReply.NetworkError.NoError  # Qt6
                is_no_error = (error_val == no_err)
            else:
                no_err = QNetworkReply.NoError  # Qt5
                is_no_error = (int(error_val) == int(no_err))

            if is_no_error:
                if hasattr(QNetworkRequest, 'Attribute'):
                    status_code = reply.attribute(QNetworkRequest.Attribute.HttpStatusCodeAttribute)  # Qt6
                else:
                    status_code = reply.attribute(QNetworkRequest.HttpStatusCodeAttribute)  # Qt5

                if status_code and int(status_code) == 200:
                    self.setupDialog()
                    # self.disable_button_download()
                else:
                    msg = "Serwer ULDK nie odpowiada. Spróbuj ponownie później"
                    MessageUtils.pushWarning(self.iface, msg)
                    MessageUtils.pushLogWarning(msg)
            else:
                MessageUtils.pushLogWarning(
                    "Ostrzeżenie: Brak połączenia z internetem"
                )

            reply.deleteLater()

        except Exception as e:
            MessageUtils.pushLogWarning(
                "Ostrzeżenie: Brak połączenia z internetem"
            )

        self.dlg.projectionWidget.setCrs(
            QgsCoordinateReferenceSystem.fromEpsgId(int(srid))
        )


    def setupDialog(self):
        if self.dlg.RegionFetch is None:
            try:
                self.dlg.RegionFetch = RegionFetch(teryt='')
                self.dlg.fillVoivodeships()
            except Exception as e:
                MessageUtils.pushLogWarning(
                    "Ostrzeżenie: "
                    f"Nie udało się zainicjować listy województw: {e}"
                )
        self.dlg.show()

    def btnDownloadTab1Clicked(self):
        """kliknięcie klawisza pobierania po numerze TERYT w oknie wtyczki"""

        teryt = self.dlg.edit_id.text().strip()

        if not teryt:
            MessageUtils.pushLogWarning(
                "Błąd formularza: Musisz wpisać identyfikator"
            )
        elif utils.isInternetConnected():
            try:
                self.performRequestTeryt(teryt=teryt)
            except:
                msg = (
                    "Nie udało się pobrać obiektu: API nie zwróciło "
                    "odpowiedzi dla żądanego zapytania"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)


        else:
            msg = "Nie udało się pobrać obiektu: Brak połączenia z internetem"
            MessageUtils.pushLogCritical(msg)
            MessageUtils.pushCritical(self.iface, msg)


    def btnDownloadTab2Clicked(self):
        """kliknięcie klawisza pobierania według X i Y wpisanych w oknie wtyczki"""
        srid = self.dlg.projectionWidget.crs().authid().split(":")[1]
        self.downloadByXY(srid, type="form", zoomToFeature=False)

    def btnSearchTab3Clicked(self):
        arkusze_numery = set()
        arkusze_numery_posortowane = set()
        if str(self.dlg.obrcomboBox.currentText().strip()):
            obj_region = str(self.dlg.obrcomboBox.currentText().strip())
            object_type = self.checkedFeatureType()
        else:
            obj_region = str(self.dlg.gmicomboBox.currentText().strip())
            object_type = self.checkedFeatureType()

        if object_type != 1:
            return
        self.dlg.parcel_lineedit.setReadOnly(True)
        current_idx = self.dlg.gmicomboBox.currentIndex()
        teryt = self.dlg.gmicomboBox.itemData(current_idx)
        obj_parcel = self.dlg.parcel_lineedit.text().strip() # nr działki

        if not obj_region:
            msg = "Błąd formularza: Musisz wpisać obręb"
            MessageUtils.pushLogWarning(msg)
            MessageUtils.pushWarning(self.iface, msg)

        if not obj_parcel:
            msg = "Błąd formularza: Musisz wpisać numer działki"
            MessageUtils.pushLogWarning(msg)
            MessageUtils.pushWarning(self.iface, msg)

        elif utils.isInternetConnected():
            self.dlg.arkcomboBox.clear()

            obr_idx = self.dlg.obrcomboBox.currentIndex()
            teryt = self.dlg.obrcomboBox.itemData(obr_idx)

            result_obreb = uldk_parcel.getRegionById(id=teryt)
            result_obreb = list(result_obreb)

            #sprawdzanie obrebow po usunieciu niepotrzebnych numerow
            for obreb in result_obreb:
                if len(obreb) < ULDK_MIN_LINE_LEN:
                    result_obreb.remove(obreb)
                else:
                    pass

            #sprawdzanie czy obręb leży w tym województwie, w którym powinien
            for obreb in result_obreb:
                if obreb.split("|")[-1].rstrip() == self.dlg.wojcomboBox.currentText():
                    pass
                else:
                    result_obreb.remove(obreb)

            #sprawdzanie czy obręb leży w tym powiecie, w którym powinien
            for obreb in result_obreb:
                if obreb.split("|")[-2] == self.dlg.powcomboBox.currentText():
                    pass
                else:
                    result_obreb.remove(obreb)

            #sprawdzanie czy obręb leży w tej gminie, w której powinien
            for obreb in result_obreb:
                if obreb.split("|")[-3] == self.dlg.gmicomboBox.currentText():
                    pass
                else:
                    result_obreb.remove(obreb)

            try:
                self.region_name = result_obreb[0].split("|")[0]
                name = self.region_name + '.' + obj_parcel

                result = uldk_parcel.getParcelById(name)
                result = list(result)

                for result_item in result:
                    if result_item.find("-1 brak wyników") >= 1 or result_item.find("usługa nie zwróciła odpowiedzi") >= 1 or result_item.find("błędny format odpowiedzi XML, usługa zwróciła odpowiedź") >= 1 or result_item.find("XML") >= 1 or result_item.find("błędny format") >= 1:
                        response = False
                    else:
                        response = True

                if response == True:
                    for i in range(len(result)):
                        if len(result[i]) < ULDK_MIN_LINE_LEN:
                            pass
                        elif result[i].find(";") > -1:
                            if result[i].split(";")[1].split("|")[1].split(".")[-2].find("AR") > -1:
                                arkusze_numery.add(result[i].split(";")[1].split("|")[1].split(".")[-2].strip())
                            else:
                                pass
                        else:
                            if result[i].split("|")[1].split(".")[-2].find("AR") > -1:
                                arkusze_numery.add(result[i].split("|")[1].split(".")[-2].strip())
                            else:
                                pass

                    arkusze_numery = list(arkusze_numery)
                    if len(arkusze_numery) >= 1:
                        for arkusz in arkusze_numery:
                            arkusze_numery_posortowane.add(arkusz.split("AR_")[-1])
                        for sort_ark in sorted(arkusze_numery_posortowane,key=int):
                            self.dlg.arkcomboBox.addItem(f"AR_{sort_ark}")
                    else:
                        pass
                    self.successDownload(arkusze_numery)
                else:
                    msg = "Nie zwrócono żadnej działki dla podanych parametrów"
                    MessageUtils.pushLogWarning(msg)
                    MessageUtils.pushWarning(self.iface, msg)
            except IndexError:
                msg = "Nie zwrócono żadnej działki dla podanych parametrów"
                MessageUtils.pushLogWarning(msg)
                MessageUtils.pushWarning(self.iface, msg)
        self.dlg.parcel_lineedit.setReadOnly(False)

    def successDownload(self, arkusze_numery):
        if len(arkusze_numery) >= 1:
            MessageUtils.pushLogInfo(
                "Znaleziono działkę/i dla podanych parametrów, "
                "wybierz numer arkusza."
            )
        else:
            MessageUtils.pushLogInfo(
                "Znaleziono działkę dla podanych parametrów. "
                "Kliknij przycisk Pobierz."
            )
        self.dlg.btn_download_tab3.setEnabled(True)

    def btnDownloadTab3Clicked(self):
        if self.region_name:
            obj_region = self.region_name
        else:
            obj_region = str(self.dlg.gmicomboBox.currentText().strip())

        object_type = self.checkedFeatureType()
        if object_type == 1:
            current_idx = self.dlg.gmicomboBox.currentIndex()
            teryt = self.dlg.gmicomboBox.itemData(current_idx)
            obj_parcel = self.dlg.parcel_lineedit.text().strip() # nr działki
            if not obj_region:
                msg = "Błąd formularza: Musisz wpisać obręb"
                MessageUtils.pushLogWarning(msg)
                MessageUtils.pushWarning(self.iface, msg)

            if not obj_parcel:
                MessageUtils.pushLogWarning(self.iface,
                    "Błąd formularza: Musisz wpisać numer działki"
                )

            elif utils.isInternetConnected():
                self.performRequestParcel(
                    region=obj_region, 
                    parcel=obj_parcel, 
                    teryt=None, 
                    zoomToFeature=True
                )

            else:
                msg = "Nie udało się pobrać obiektu: Brak połączenia z internetem"
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)

            return
        elif object_type == 2:  # obręb
            obr_idx = self.dlg.obrcomboBox.currentIndex()
            teryt = self.dlg.obrcomboBox.itemData(obr_idx)

            resp = uldk_api.getRegionById(teryt, object_type=2, obreb=True)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu:"
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]

        elif object_type == 3:  # gmina
            current_idx = self.dlg.gmicomboBox.currentIndex()
            teryt = self.dlg.gmicomboBox.itemData(current_idx)
            resp = uldk_api.getCommuneById(teryt, object_type=3)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = res[2]
            county = res[3]
            voivodeship = res[4]

        elif object_type == 4:
            current_idx = self.dlg.powcomboBox.currentIndex()
            teryt = self.dlg.powcomboBox.itemData(current_idx)
            resp = uldk_api.getCountyById(teryt, object_type=4)

            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = res[2]
            voivodeship = res[3]

        elif object_type == 5: # województwo
            current_idx = self.dlg.wojcomboBox.currentIndex()
            teryt = self.dlg.wojcomboBox.itemData(current_idx)
            resp = uldk_api.getVoivodeshipById(teryt, object_type=5)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = None
            voivodeship = res[2]

        self.addResultsToLayer(
            object_type=object_type,
            wkt=wkt,
            teryt=teryt,
            parcel=parcel,
            region=region,
            commune=commune,
            county=county,
            voivodeship=voivodeship,
            zoomToFeature=True,
        )

        label = OBJECT_TYPES[object_type]["success_label"]
        success_message = f"Pobrano {label}" % teryt if "%s" in label else f"Pobrano {label}"

        MessageUtils.pushLogSuccess(success_message)
        MessageUtils.pushSuccess(self.iface,
            success_message
        )

    def downloadByXY(self, srid, type, zoomToFeature=False):
        """pobranie według X i Y i SRID"""

        objX = self.dlg.doubleSpinBoxX.text().strip()
        objY = self.dlg.doubleSpinBoxY.text().strip()

        if type == "form" and srid in SWAP_XY_SRIDS:
            objX = self.dlg.doubleSpinBoxY.text().strip()
            objY = self.dlg.doubleSpinBoxX.text().strip()

        if not objX:
            MessageUtils.pushLogWarning(
                "Błąd formularza: Musisz wpisać współrzędną X"
            )

        if not objY:
            MessageUtils.pushLogWarning(
                "Błąd formularza: musisz wpisać współrzędną Y"
            )

        elif utils.isInternetConnected():
            try:
                self.performRequestXY(x=objX, y=objY, srid=srid, zoomToFeature=zoomToFeature)
                # self.dlg.hide()  # jeżeli wtyczka ma zostawiać włączone okno, zamiast hide wpisz show
            except Exception as e:
                MessageUtils.pushLogWarning(f"Błąd: {e}")
                MessageUtils.pushLogWarning(
                    "Nie udało się pobrać obiektu: "
                    "API nie zwróciło obiektu dla wybranego zapytania"
                )


        else:
            msg = (
                "Nie udało się pobrać obiektu:"
                "Brak połączenia z internetem"
            )
            MessageUtils.pushLogCritical(msg)
            MessageUtils.pushCritical(self.iface, msg)

    def shortcutActivated(self):
        """zdarzenie aktywowania klawisza skrótu wskazania działki na mapie"""
        self.canvas.setMapTool(self.click_tool)

    def btnFrommapClicked(self):
        """zdarzenie wciśnięcia w oknie wtyczki klawisza wskazania działki na mapie"""
        self.canvas.setMapTool(self.click_tool)
        self.dlg.hide()

    def canvasClicked(self, point):
        """kliknięcie na mapie"""
        self.canvas.unsetMapTool(self.click_tool)
        self.dlg.doubleSpinBoxX.setValue(point.x())
        self.dlg.doubleSpinBoxY.setValue(point.y())
        srid = self.project.crs().postgisSrid()
        self.downloadByXY(srid, type="click", zoomToFeature=False)

    def performRequestParcel(self, region, parcel, teryt, zoomToFeature=True):
        object_type = self.checkedFeatureType()
        try:
            if self.dlg.arkcomboBox.currentText() != '':
                name = region + '.' + self.dlg.arkcomboBox.currentText() + '.' + parcel
            else:
                name = region + '.' + parcel
            result = uldk_parcel.getParcelByIdSearch(name, object_type=1)

            if result is None:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {name}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = result.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {name}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = res[2]
            region = res[3]
            commune = res[4]
            county = res[5]
            voivodeship = res[6]

            nazwa = OBJECT_TYPES[object_type]["layer_name"]
            layers = self.project.mapLayersByName(nazwa)

            if layers:
                # jezeli istnieje to dodaj obiekt do warstwy
                layer = layers[0]
            else:
                # jezeli nie istnieje to stworz warstwe
                epsg = f"Polygon?crs=EPSG:{DEFAULT_SRID}"
                layer = QgsVectorLayer(epsg, nazwa, "memory")
                self.project.addMapLayer(layer)
            
            provider = layer.dataProvider()
            layer.updateExtents()

            if not layers:
                voivField = QgsField('województwo', QVariant.String, len=30)
                provider.addAttributes([voivField])

                conField = QgsField('powiat', QVariant.String, len=30)
                provider.addAttributes([conField])

                comField = QgsField('gmina', QVariant.String, len=30)
                provider.addAttributes([comField])

                regField = QgsField('obręb', QVariant.String, len=30)
                provider.addAttributes([regField])

                parField = QgsField('numer', QVariant.String, len=30)
                provider.addAttributes([parField])
                
                idField = QgsField('teryt', QVariant.String, len=40)
                provider.addAttributes([idField])

                layer.updateFields()

            feat = QgsFeature(provider.fields())
            feat.setGeometry(QgsGeometry().fromWkt(wkt))

            fields_mapping = {
                'teryt': teryt,
                'numer': parcel,
                'obręb': region,
                'gmina': commune,
                'powiat': county,
            }

            feat.setAttribute('województwo', voivodeship)
            for field_name, attr in fields_mapping.items():
                if field_name in feat.fields().names():
                    feat.setAttribute(field_name, attr or None)

            provider.addFeature(feat)
            msg = f"Pobrano działkę dla obiektu: {name}"
            MessageUtils.pushLogSuccess(msg)
            MessageUtils.pushSuccess(self.iface, msg)

            if zoomToFeature:
                project_crs = self.project.crs().postgisSrid()
                if project_crs != DEFAULT_SRID:
                    source_crs = QgsCoordinateReferenceSystem.fromEpsgId(DEFAULT_SRID)
                    dest_crs = QgsCoordinateReferenceSystem.fromEpsgId(project_crs)
                    tr = QgsCoordinateTransform(source_crs, dest_crs, self.project)
                    box = tr.transform(feat.geometry().boundingBox())
                else:
                    box = feat.geometry().boundingBox()

                self.canvas.setExtent(box)
                self.canvas.refresh()
            else:
                layer.triggerRepaint()

        except IndexError:
            msg = "Nie pobrano żadnej działki dla podanych parametrów"
            MessageUtils.pushLogWarning(msg)
            MessageUtils.pushWarning(self.iface, msg)

    def performRequestTeryt(self, teryt, zoomToFeature=True):
        """wykonanie zapytania pobierającego obiekt na podstawie kodu TERYT"""
        object_type = self.checkedFeatureType()
        if object_type == 1:
            resp = uldk_api.getParcelById(teryt, object_type=1)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            if teryt != res[1] and "AR" in res[1]:
                MessageUtils.pushLogInfo(self.iface,
                    "W wybranym obrębie znaleziono więcej działek o "
                    f"identyfikatorze TERYT: {teryt}. Dodaj numer "
                    "arkusza w celu odnalezienia właściwej działki"
                )
            wkt = res[0]
            teryt = res[1]
            parcel = res[2]
            region = res[3]
            commune = res[4]
            county = res[5]
            voivodeship = res[6]


        elif object_type == 2:
            resp = uldk_api.getRegionById(teryt, object_type=2)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]


        elif object_type == 3:
            resp = uldk_api.getCommuneById(teryt, object_type=3)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = res[2]
            county = res[3]
            voivodeship = res[4]


        elif object_type == 4:
            resp = uldk_api.getCountyById(teryt, object_type=4)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = res[2]
            voivodeship = res[3]

        elif object_type == 5:
            resp = uldk_api.getVoivodeshipById(teryt, object_type=5)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = None
            voivodeship = res[2]

        elif object_type == 6:
            resp = uldk_api.getBuildingById(teryt, object_type=6)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            if res[0] == '':
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło geometrii dla id {teryt}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]

        self.addResultsToLayer(
            object_type=object_type,
            wkt=wkt,
            teryt=teryt,
            parcel=parcel,
            region=region,
            commune=commune,
            county=county,
            voivodeship=voivodeship,
            zoomToFeature=zoomToFeature)

        label = OBJECT_TYPES[object_type]["success_label"]
        success_message = f"Pobrano {label}" % teryt if "%s" in label else f"Pobrano {label}"
        MessageUtils.pushLogSuccess(success_message)
        MessageUtils.pushSuccess(self.iface, success_message)

    def performRequestXY(self, x, y, srid, zoomToFeature):
        """wykonanie zapytania pobierającego obiekt na podstawie współrzędnych"""

        object_type = self.checkedFeatureType()

        #Sprawdzenie, czy współrzędne są z przecinkiem czy kropką
        if "," in x or "," in y:
            x = float(x.replace(",", "."))
            y = float(y.replace(",", "."))
        else:
            x = float(x)
            y = float(y)

        requestPoint = QgsPoint(x, y)

        if str(srid) != DEFAULT_SRID:
            sourceCrs = QgsCoordinateReferenceSystem.fromEpsgId(int(srid))
            destCrs = QgsCoordinateReferenceSystem.fromEpsgId(DEFAULT_SRID)
            tr = QgsCoordinateTransform(sourceCrs, destCrs, self.project)
            requestPoint.transform(tr)
        pid = f"{str(requestPoint.x())},{str(requestPoint.y())}"

        if object_type == 1:# działka
            resp = uldk_xy.getParcelByXY(xy=pid, object_type=1)

            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla współrzędnych {pid}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = res[2]
            region = res[3]
            commune = res[4]
            county = res[5]
            voivodeship = res[6]

        elif object_type == 2:
            resp = uldk_xy.getRegionByXY(xy=pid, object_type=2)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla współrzędnych {pid}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]


        elif object_type == 3:
            resp = uldk_xy.getCommuneByXY(xy=pid, object_type=3)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla współrzędnych {pid}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = res[2]
            county = res[3]
            voivodeship = res[4]


        elif object_type == 4:
            resp = uldk_xy.getCountyByXY(xy=pid, object_type=4)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla współrzędnych {pid}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = res[2]
            voivodeship = res[3]


        elif object_type == 5:
            resp = uldk_xy.getVoivodeshipByXY(xy=pid, object_type=5)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla współrzędnych {pid}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return

            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = None
            commune = None
            county = None
            voivodeship = res[2]

        elif object_type == 6:
            resp = uldk_xy.getBuildingByXY(xy=pid, object_type=6)
            if not resp:
                msg = (
                    "Nie udało się pobrać obiektu: "
                    f"API nie zwróciło obiektu dla współrzędnych {pid}"
                )
                MessageUtils.pushLogCritical(msg)
                MessageUtils.pushCritical(self.iface, msg)
                return
            res = resp.split("|")
            wkt = res[0]
            teryt = res[1]
            parcel = None
            region = res[2]
            commune = res[3]
            county = res[4]
            voivodeship = res[5]

        self.addResultsToLayer(
            object_type=object_type,
            wkt=wkt,
            teryt=teryt,
            parcel=parcel,
            region=region,
            commune=commune,
            county=county,
            voivodeship=voivodeship,
            zoomToFeature=zoomToFeature)

        label = OBJECT_TYPES[object_type]["success_label"]
        success_message = f"Pobrano {label}" % teryt if "%s" in label else f"Pobrano {label}"
        MessageUtils.pushLogSuccess(success_message)
        MessageUtils.pushSuccess(self.iface, success_message)

    def addResultsToLayer(self, object_type, wkt, teryt, parcel, region, commune, county, voivodeship, zoomToFeature):
        """dodaje wyniki (odpowiedź z serwera) do mapy jako warstwę z atrybutami i geometrią"""

        # layer
        nazwa = OBJECT_TYPES[object_type]["layer_name"]
        layers = self.project.mapLayersByName(nazwa)

        # usuwanie pustych warstw z projektu
        for layer in layers:
            if layer.featureCount() == 0:
                self.project.removeMapLayer(layer)
                layers.remove(layer)


        if layers:
            # jezeli istnieje to dodaj obiekt do warstwy
            layer = layers[0]
            provider = layer.dataProvider()

        else:
            # jezeli nie istnieje to stworz warstwe
            layer = QgsVectorLayer(f"Polygon?crs=EPSG:{DEFAULT_SRID}", nazwa, "memory")
            self.project.addMapLayer(layer)

            provider = layer.dataProvider()

            voivField = QgsField('województwo', QVariant.String, len=30)
            provider.addAttributes([voivField])

            if object_type==6 or object_type == 4 or object_type == 3 or object_type == 2 or object_type == 1:
                conField = QgsField('powiat', QVariant.String, len=30)
                provider.addAttributes([conField])

            if object_type==6 or object_type == 3 or object_type == 2 or object_type == 1:
                comField = QgsField('gmina', QVariant.String, len=30)
                provider.addAttributes([comField])

            if object_type==6 or object_type == 2 or object_type == 1:
                regField = QgsField('obręb', QVariant.String, len=30)
                provider.addAttributes([regField])

            if object_type == 1:
                parField = QgsField('numer', QVariant.String, len=30)
                provider.addAttributes([parField])

            idField = QgsField('teryt', QVariant.String, len=40)
            provider.addAttributes([idField])

            layer.updateFields()

        feat = QgsFeature(provider.fields())
        feat.setGeometry(QgsGeometry().fromWkt(wkt))

        fields_mapping = {
            'teryt': teryt,
            'numer': parcel,
            'obręb': region,
            'gmina': commune,
            'powiat': county,
        }

        if voivodeship != None:
            feat.setAttribute('województwo', voivodeship)

        for field_name, attr in fields_mapping.items():
            if field_name in feat.fields().names():
                feat.setAttribute(field_name, attr or None)
        provider.addFeature(feat)

        if zoomToFeature:
            project_crs = self.project.crs().postgisSrid()

            if project_crs != DEFAULT_SRID:
                source_crs = QgsCoordinateReferenceSystem.fromEpsgId(DEFAULT_SRID)
                dest_crs = QgsCoordinateReferenceSystem.fromEpsgId(project_crs)
                tr = QgsCoordinateTransform(source_crs, dest_crs, self.project)
                box = tr.transform(feat.geometry().boundingBox())
            else:
                box = feat.geometry().boundingBox()

            self.canvas.setExtent(box)
            self.canvas.refresh()
        else:
            layer.triggerRepaint()

        # Funkcja odświeża wszystkie elementy jakie są w warstwie
        self.iface.mapCanvas().refreshAllLayers()


    def checkedFeatureType(self):
        """
        Funkcja pomocnicza sprawdzająca jaki typ obiektu jest zaznaczony w oknie wtyczki
        @returns:
        1 - działka ewidencyjna
        2 - obręb ewidencyjny
        3 - gmina
        4 - powiat
        5 - województwo
        6 - budynek
        0 - niezdefiniowany
        """
        dlg = self.dlg
        if dlg.rdb_dz.isChecked():
            return 1
        elif dlg.rdb_ob.isChecked():
            return 2
        elif dlg.rdb_gm.isChecked():
            return 3
        elif dlg.rdb_pw.isChecked():
            return 4
        elif dlg.rdb_wo.isChecked():
            return 5
        elif dlg.rdb_bu.isChecked():
            return 6
        else:
            return 0
